/**
 * 
 */
package plugins.tprovoost.Microscopy.microscopelive3d;

import icy.gui.component.NumberTextField;
import icy.gui.component.NumberTextField.ValueChangeListener;
import icy.gui.component.button.IcyButton;
import icy.gui.dialog.MessageDialog;
import icy.gui.frame.IcyFrame;
import icy.gui.frame.IcyFrameAdapter;
import icy.gui.frame.IcyFrameEvent;
import icy.gui.util.GuiUtil;
import icy.gui.viewer.Viewer;
import icy.image.IcyBufferedImage;
import icy.main.Icy;
import icy.preferences.PluginsPreferences;
import icy.preferences.XMLPreferences;
import icy.sequence.Sequence;
import icy.sequence.SequenceEvent;
import icy.sequence.SequenceListener;
import icy.system.IcyExceptionHandler;
import icy.type.collection.array.ArrayUtil;
import icy.util.StringUtil;

import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.List;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingConstants;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import plugins.kernel.canvas.VtkCanvasPlugin;
import plugins.tprovoost.Microscopy.MicroManager.MicroManager;
import plugins.tprovoost.Microscopy.MicroManager.event.LiveListener;
import plugins.tprovoost.Microscopy.MicroManager.tools.FrameUtils;
import plugins.tprovoost.Microscopy.MicroManager.tools.StageMover;
import plugins.tprovoost.Microscopy.MicroManager.tools.StageMover.StageListener;

public class MicroscopeLive3D implements LiveListener, StageListener, SequenceListener
{
    // -----------
    // PREFERENCES
    // -----------
    private static final String PREF_SLICES = "slices";
    private static final String PREF_INTERVAL = "interval";
    private static final String PREF_REFRESH = "refresh";

    MicroscopeLive3DPlugin plugin;
    IcyFrame frame;
    final Sequence liveSequence;
    final XMLPreferences _prefs;

    int _slices;
    double initialPos;
    int sliceIdx;
    double _interval;
    boolean refreshPerStack;
    boolean startedHere;
    boolean modifyingExposure;

    // ---
    // GUI
    // ---
    NumberTextField _tf_interval, _tfExposure;
    SpinnerNumberModel _model_slices;
    IcyButton btn_start, btn_stop;
    SequenceListener listener;
    JLabel currentStagePosition, stackPerSecond;

    MicroscopeLive3D(MicroscopeLive3DPlugin plugin)
    {
        super();

        this.plugin = plugin;
        _prefs = PluginsPreferences.root(plugin);

        // default parameters
        _slices = 5;
        sliceIdx = 0;
        _interval = 10;
        refreshPerStack = true;
        startedHere = false;
        modifyingExposure = false;

        try
        {
            initialPos = StageMover.getZ();
        }
        catch (Exception e)
        {
            initialPos = 0;
        }

        initializeGui();

        // create live sequence
        final List<Sequence> sequences = Icy.getMainInterface().getSequences("Live Mode 3D");
        if (sequences.size() > 0)
            liveSequence = sequences.get(0);
        else
            liveSequence = new Sequence("Live Mode 3D");

        liveSequence.addListener(this);

        StageMover.addListener(MicroscopeLive3D.this);
        MicroManager.addLiveListener(MicroscopeLive3D.this);

        updateState();
    }

    public void initializeGui()
    {
        double exp;

        try
        {
            // retrieve current exposure
            exp = MicroManager.getCore().getExposure();
        }
        catch (Exception e)
        {
            exp = 10;
        }

        frame = new IcyFrame("Live 3D Options", true, true, false, true);
        // -------------------
        // OBSERVATION OPTIONS
        // -------------------
        final JPanel panel_observation = GuiUtil.generatePanel("Observation");
        panel_observation.setLayout(new GridLayout(2, 2));

        _tfExposure = new NumberTextField();
        _tfExposure.setNumericValue(exp);
        _tfExposure.setHorizontalAlignment(SwingConstants.RIGHT);
        _tfExposure.setToolTipText("Time of exposure. Zero means continuous acquisition.");
        _tfExposure.addValueListener(new ValueChangeListener()
        {
            @Override
            public void valueChanged(double newValue, boolean validate)
            {
                if (modifyingExposure)
                {
                    _prefs.putDouble(PREF_REFRESH, newValue);
                    return;
                }

                if (validate)
                {
                    try
                    {
                        MicroManager.setExposure(newValue);
                    }
                    catch (Exception e)
                    {
                        IcyExceptionHandler.showErrorMessage(e, true);
                    }

                    _prefs.putDouble(PREF_REFRESH, newValue);

                }
            }
        });

        stackPerSecond = new JLabel();
        stackPerSecond.setText(StringUtil.toString(1000d / (exp * _slices), 3));
        stackPerSecond.setHorizontalAlignment(SwingConstants.RIGHT);
        stackPerSecond.setToolTipText("Stack per second calculated by 1000/ (exposure * number of slice).");

        panel_observation.add(new JLabel("Exposure (ms):"));
        panel_observation.add(_tfExposure);
        panel_observation.add(new JLabel("Stack Per Second :"));
        panel_observation.add(stackPerSecond);

        // ---------------
        // POSITION OPTIONS
        // ---------------
        final JPanel panel_position = GuiUtil.generatePanel("Position");
        panel_position.setLayout(new GridLayout(2, 2));
        currentStagePosition = new JLabel(StringUtil.toString(initialPos, 3) + " m");

        final NumberTextField positionText = new NumberTextField();
        positionText.setNumericValue(initialPos);
        positionText.addValueListener(new ValueChangeListener()
        {
            @Override
            public void valueChanged(double newValue, boolean validate)
            {
                if (validate)
                {
                    initialPos = newValue;

                    try
                    {
                        StageMover.moveZAbsolute(newValue);
                    }
                    catch (Exception e)
                    {
                        IcyExceptionHandler.showErrorMessage(e, true);
                    }
                }
            }
        });

        panel_position.add(new JLabel("Start Z position (m) :"));
        panel_position.add(positionText);
        panel_position.add(new JLabel("Current Z position (m) : "));
        panel_position.add(currentStagePosition);

        // ---------------
        // SLICES OPTIONS
        // ---------------
        JPanel panel_slice = GuiUtil.generatePanel("Slices");
        panel_slice.setLayout(new GridLayout(2, 2));

        _model_slices = new SpinnerNumberModel(10, 1, 1000, 1);
        _model_slices.addChangeListener(new ChangeListener()
        {
            @Override
            public void stateChanged(ChangeEvent changeevent)
            {
                _slices = _model_slices.getNumber().intValue();
                stackPerSecond.setText(StringUtil.toString(1000d / (_tfExposure.getNumericValue() * _slices), 3));
                _prefs.putInt(PREF_SLICES, _slices);
                // reset slice index
                sliceIdx = 0;
            }
        });

        _tf_interval = new NumberTextField();
        _tf_interval.setNumericValue(_interval);
        _tf_interval.setHorizontalAlignment(SwingConstants.RIGHT);
        _tf_interval.addValueListener(new ValueChangeListener()
        {
            @Override
            public void valueChanged(double newValue, boolean validate)
            {
                if (validate)
                {
                    if (newValue < 1)
                    {
                        // force value to 1
                        _tf_interval.setNumericValue(1);
                        return;
                    }

                    _interval = newValue;
                    _prefs.putDouble(PREF_INTERVAL, _interval);
                }
            }
        });

        panel_slice.add(new JLabel("Slices Count: "));
        panel_slice.add(new JSpinner(_model_slices));
        panel_slice.add(new JLabel("Interval (m): "));
        panel_slice.add(_tf_interval);

        // ------------
        // REFRESH OPTIONS
        // ------------

        JPanel panel_refresh = GuiUtil.generatePanel("Refresh option");
        panel_refresh.setLayout(new GridLayout(1, 2));
        final JCheckBox byStack = new JCheckBox("Refresh each stack"), byFrame = new JCheckBox("Refresh each frame");
        byFrame.setSelected(false);
        byFrame.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent arg0)
            {
                refreshPerStack = !byFrame.isSelected();
                byStack.setSelected(!byFrame.isSelected());
            }
        });

        byStack.setSelected(true);
        byStack.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent e)
            {
                refreshPerStack = byStack.isSelected();
                byFrame.setSelected(!byStack.isSelected());
            }
        });

        panel_refresh.add(byStack);
        panel_refresh.add(byFrame);

        // ------------
        // RUN OPTIONS
        // ------------
        btn_start = FrameUtils.createUIButton("", "playback_play.png", new Runnable()
        {
            @Override
            public void run()
            {
                try
                {
                    startedHere = MicroManager.startLiveMode();
                    updateState();
                }
                catch (Exception e)
                {
                    MessageDialog.showDialog("Unable to start live, please check if an acquisition is running",
                            MessageDialog.ERROR_MESSAGE);
                }
            }
        });
        btn_stop = FrameUtils.createUIButton("", "playback_stop.png", new Runnable()
        {
            @Override
            public void run()
            {
                stop();
            }
        });
        JPanel panel_buttons = GuiUtil.generatePanel("Run");
        panel_buttons.setLayout(new BoxLayout(panel_buttons, BoxLayout.X_AXIS));
        panel_buttons.add(Box.createHorizontalGlue());
        panel_buttons.add(btn_start);
        panel_buttons.add(Box.createHorizontalGlue());
        panel_buttons.add(btn_stop);
        panel_buttons.add(Box.createHorizontalGlue());

        // ----------------
        // DISPLAY
        // ----------------
        JPanel mainPanel = new JPanel();
        mainPanel.setLayout(new GridLayout(5, 1));
        mainPanel.add(panel_slice);
        mainPanel.add(panel_observation);
        mainPanel.add(panel_position);
        mainPanel.add(panel_refresh);
        mainPanel.add(panel_buttons);

        loadPreferences();

        frame.setLayout(new GridLayout(1, 1));
        frame.add(mainPanel);
        frame.pack();
        frame.setVisible(true);
        frame.addToDesktopPane();
        frame.requestFocus();

        frame.addFrameListener(new IcyFrameAdapter()
        {
            @Override
            public void icyFrameClosing(IcyFrameEvent e)
            {
                super.icyFrameClosing(e);

                plugin.shutdown();
            }
        });
    }

    public void toFront()
    {
        frame.toFront();
    }

    void stop()
    {
        try
        {
            MicroManager.stopLiveMode();
        }
        catch (Exception ex)
        {
        }
    }

    public void shutdown()
    {
        // stop live if we started it
        if (startedHere)
            stop();

        // remove listener
        MicroManager.removeLiveListener(this);
        StageMover.removeListener(this);
        liveSequence.removeListener(this);

        savePreferences();

        frame.dispose();
    }

    void savePreferences()
    {
        _prefs.putDouble(PREF_INTERVAL, _interval);
        _prefs.getInt(PREF_SLICES, _slices);
    }

    void loadPreferences()
    {
        _interval = _prefs.getDouble(PREF_INTERVAL, _interval);
        _slices = _prefs.getInt(PREF_SLICES, _slices);

        _model_slices.setValue(Integer.valueOf(_slices));
        _tf_interval.setText("" + _interval);
    }

    @Override
    public void sequenceChanged(SequenceEvent sequenceEvent)
    {

    }

    @Override
    public void sequenceClosed(Sequence sequence)
    {
        // stop live
        if (startedHere)
            stop();
    }

    public void onExposureChanged(double newExposure)
    {
        modifyingExposure = true;
        try
        {
            stackPerSecond.setText(StringUtil.toString(1000 / (newExposure * _slices), 3));
            _tfExposure.setText("" + newExposure);
        }
        finally
        {
            modifyingExposure = false;
        }
    }

    @Override
    public void onStagePositionChanged(String zStage, double z)
    {
        currentStagePosition.setText(StringUtil.toString(z, 3) + " m");
    }

    @Override
    public void onStagePositionChangedRelative(String zStage, double z)
    {
    }

    @Override
    public void onXYStagePositionChanged(String XYStage, double x, double y)
    {
    }

    @Override
    public void onXYStagePositionChangedRelative(String XYStage, double x, double y)
    {
    }

    @Override
    public void liveStarted()
    {
        updateState();
    }

    @Override
    public void liveStopped()
    {
        updateState();

        // be sure to notify data changed even if we stop in middle of stack
        if (refreshPerStack)
            liveSequence.dataChanged();

        startedHere = false;
    }

    void updateState()
    {
        if (MicroManager.isLiveRunning())
        {
            btn_start.setEnabled(false);
            btn_stop.setEnabled(true);

            // make sequence visible
            if (!Icy.getMainInterface().isOpened(liveSequence))
            {
                final Viewer viewer = new Viewer(liveSequence);
                // use VTK canvas
                viewer.setCanvas(VtkCanvasPlugin.class.getName());
            }
        }
        else
        {
            btn_start.setEnabled(true);
            btn_stop.setEnabled(false);
        }
    }

    @Override
    public void liveImgReceived(IcyBufferedImage image)
    {
        // need to reload sequence ?
        if (liveSequence.isEmpty() || !liveSequence.isCompatible(image) || (liveSequence.getSizeZ() > _slices))
        {
            liveSequence.beginUpdate();
            try
            {
                liveSequence.removeAllImages();
                liveSequence.setImage(0, sliceIdx, image);
            }
            finally
            {
                liveSequence.endUpdate();
            }
        }
        else
        {
            final IcyBufferedImage seqImage = liveSequence.getImage(0, sliceIdx);

            if (seqImage == null)
                liveSequence.setImage(0, sliceIdx, image);
            else
            {
                // change data
                for (int c = 0; c < seqImage.getSizeC(); c++)
                    ArrayUtil.arrayToArray(image.getDataXY(c), seqImage.getDataXY(c), seqImage.isSignedDataType());

                // notify data changed
                if ((!refreshPerStack) || (sliceIdx == (_slices - 1)))
                    seqImage.dataChanged();
            }
        }

        // pass to next slice
        sliceIdx = (sliceIdx + 1) % _slices;

        try
        {
            // reset position
            if (sliceIdx == 0)
                StageMover.moveZAbsolute(initialPos);
            else
                StageMover.moveZRelative(_interval);
        }
        catch (Exception e)
        {
            IcyExceptionHandler.showErrorMessage(e, true);
        }
    }

}