/*
 * Decompiled with CFR 0.152.
 */
package loci.formats.tiff;

import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.DataOutputStream;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.TreeSet;
import loci.common.ByteArrayHandle;
import loci.common.RandomAccessInputStream;
import loci.common.RandomAccessOutputStream;
import loci.formats.FormatException;
import loci.formats.FormatTools;
import loci.formats.codec.CodecOptions;
import loci.formats.tiff.IFD;
import loci.formats.tiff.IFDList;
import loci.formats.tiff.IFDType;
import loci.formats.tiff.PhotoInterp;
import loci.formats.tiff.TiffCompression;
import loci.formats.tiff.TiffIFDEntry;
import loci.formats.tiff.TiffParser;
import loci.formats.tiff.TiffRational;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TiffSaver
implements Closeable {
    private static final Logger LOGGER = LoggerFactory.getLogger(TiffSaver.class);
    protected RandomAccessOutputStream out;
    protected String filename;
    protected ByteArrayHandle bytes;
    private boolean bigTiff = false;
    private boolean sequentialWrite = false;
    private long[] sequentialTileOffsets;
    private long[] sequentialTileByteCounts;
    private Long sequentialTileFilePointer;
    private CodecOptions options;
    private ByteArrayHandle ifdBuffer = new ByteArrayHandle();

    public TiffSaver(String filename) throws IOException {
        this(new RandomAccessOutputStream(filename), filename);
    }

    public TiffSaver(RandomAccessOutputStream out, String filename) {
        if (out == null) {
            throw new IllegalArgumentException("Output stream expected to be not-null");
        }
        if (filename == null) {
            throw new IllegalArgumentException("Filename expected to be not null");
        }
        this.out = out;
        this.filename = filename;
    }

    public TiffSaver(RandomAccessOutputStream out, ByteArrayHandle bytes) {
        if (out == null) {
            throw new IllegalArgumentException("Output stream expected to be not-null");
        }
        if (bytes == null) {
            throw new IllegalArgumentException("Bytes expected to be not null");
        }
        this.out = out;
        this.bytes = bytes;
    }

    @Override
    public void close() throws IOException {
        if (this.out != null) {
            this.out.close();
        }
    }

    public void setWritingSequentially(boolean sequential) {
        this.sequentialWrite = sequential;
    }

    public RandomAccessOutputStream getStream() {
        return this.out;
    }

    public void setLittleEndian(boolean littleEndian) {
        this.out.order(littleEndian);
    }

    public void setBigTiff(boolean bigTiff) {
        this.bigTiff = bigTiff;
    }

    public boolean isLittleEndian() {
        return this.out.isLittleEndian();
    }

    public boolean isBigTiff() {
        return this.bigTiff;
    }

    public void setCodecOptions(CodecOptions options) {
        this.options = options;
    }

    public void writeHeader() throws IOException {
        this.out.seek(0L);
        if (this.isLittleEndian()) {
            this.out.writeByte(73);
            this.out.writeByte(73);
        } else {
            this.out.writeByte(77);
            this.out.writeByte(77);
        }
        if (this.bigTiff) {
            this.out.writeShort(43);
        } else {
            this.out.writeShort(42);
        }
        if (this.bigTiff) {
            this.out.writeShort(8);
            this.out.writeShort(0);
            this.out.writeLong(16L);
        } else {
            this.out.writeInt(8);
        }
    }

    public void writeImage(byte[][] buf, IFDList ifds, int pixelType) throws FormatException, IOException {
        if (ifds == null) {
            throw new FormatException("IFD cannot be null");
        }
        if (buf == null) {
            throw new FormatException("Image data cannot be null");
        }
        for (int i = 0; i < ifds.size(); ++i) {
            if (i >= buf.length) continue;
            this.writeImage(buf[i], (IFD)ifds.get(i), i, pixelType, i == ifds.size() - 1);
        }
    }

    public void writeImage(byte[] buf, IFD ifd, int no, int pixelType, boolean last) throws FormatException, IOException {
        if (ifd == null) {
            throw new FormatException("IFD cannot be null");
        }
        int w = (int)ifd.getImageWidth();
        int h2 = (int)ifd.getImageLength();
        this.writeImage(buf, ifd, no, pixelType, 0, 0, w, h2, last);
    }

    public void writeImage(byte[] buf, IFD ifd, int no, int pixelType, int x, int y, int w, int h2, boolean last) throws FormatException, IOException {
        this.writeImage(buf, ifd, no, pixelType, x, y, w, h2, last, null, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void writeImage(byte[] buf, IFD ifd, int no, int pixelType, int x, int y, int w, int h2, boolean last, Integer nChannels, boolean copyDirectly) throws FormatException, IOException {
        ByteArrayOutputStream[] stripBuf;
        int nStrips;
        int tileHeight;
        int tileWidth;
        TiffCompression compression;
        boolean interleaved;
        LOGGER.debug("Attempting to write image.");
        if (buf == null) {
            throw new FormatException("Image data cannot be null");
        }
        if (ifd == null) {
            throw new FormatException("IFD cannot be null");
        }
        TiffSaver tiffSaver = this;
        synchronized (tiffSaver) {
            int effectiveStrips;
            int bytesPerPixel = FormatTools.getBytesPerPixel(pixelType);
            int blockSize = w * h2 * bytesPerPixel;
            if (nChannels == null) {
                nChannels = buf.length / (w * h2 * bytesPerPixel);
            }
            interleaved = ifd.getPlanarConfiguration() == 1;
            this.makeValidIFD(ifd, pixelType, nChannels);
            compression = ifd.getCompression();
            tileWidth = (int)ifd.getTileWidth();
            tileHeight = (int)ifd.getTileLength();
            int tilesPerRow = (int)ifd.getTilesPerRow();
            int rowsPerStrip = (int)ifd.getRowsPerStrip()[0];
            int stripSize = rowsPerStrip * tileWidth * bytesPerPixel;
            nStrips = (w + tileWidth - 1) / tileWidth * ((h2 + tileHeight - 1) / tileHeight);
            if (interleaved) {
                stripSize *= nChannels.intValue();
            } else {
                nStrips *= nChannels.intValue();
            }
            stripBuf = new ByteArrayOutputStream[nStrips];
            DataOutputStream[] stripOut = new DataOutputStream[nStrips];
            for (int strip = 0; strip < nStrips; ++strip) {
                stripBuf[strip] = new ByteArrayOutputStream(stripSize);
                stripOut[strip] = new DataOutputStream(stripBuf[strip]);
            }
            int[] bps = ifd.getBitsPerSample();
            boolean channelsAllSameSize = true;
            for (int c = 0; c < nChannels; ++c) {
                if (bps[c] == bytesPerPixel * 8) continue;
                channelsAllSameSize = false;
            }
            if (channelsAllSameSize && ifd.getImageWidth() == (long)w && ifd.getTileWidth() == (long)w || tileHeight * tileWidth * nChannels * bytesPerPixel == buf.length) {
                if (buf.length % stripSize == 0) {
                    for (int strip = 0; strip < nStrips; ++strip) {
                        stripOut[strip].write(buf, strip * stripSize, stripSize);
                    }
                } else {
                    effectiveStrips = !interleaved ? nStrips / nChannels : nStrips;
                    int planarChannels = !interleaved ? nChannels : 1;
                    int totalBytesPerChannel = buf.length / planarChannels;
                    for (int p = 0; p < planarChannels; ++p) {
                        for (int strip = 0; strip < effectiveStrips - 1; ++strip) {
                            stripOut[p * effectiveStrips + strip].write(buf, strip * stripSize, stripSize);
                        }
                        int pos = p * totalBytesPerChannel + (effectiveStrips - 1) * stripSize;
                        int len = (p + 1) * totalBytesPerChannel - pos;
                        int lastStripIndex = p * effectiveStrips + (effectiveStrips - 1);
                        stripOut[lastStripIndex].write(buf, pos, len);
                        byte[] extra = new byte[stripSize - len];
                        Arrays.fill(extra, (byte)0);
                        stripOut[lastStripIndex].write(extra);
                    }
                }
            } else {
                int n = effectiveStrips = !interleaved ? nStrips / nChannels : nStrips;
                if (effectiveStrips == 1 && copyDirectly) {
                    stripOut[0].write(buf);
                } else {
                    for (int strip = 0; strip < effectiveStrips; ++strip) {
                        int xOffset = strip % tilesPerRow * tileWidth;
                        int yOffset = strip / tilesPerRow * tileHeight;
                        for (int row = 0; row < tileHeight; ++row) {
                            for (int col = 0; col < tileWidth; ++col) {
                                int ndx = ((row + yOffset) * w + col + xOffset) * bytesPerPixel;
                                for (int c = 0; c < nChannels; ++c) {
                                    for (int n2 = 0; n2 < bps[c] / 8; ++n2) {
                                        int off;
                                        if (interleaved) {
                                            off = ndx * nChannels + c * bytesPerPixel + n2;
                                            if (row >= h2 || col >= w) {
                                                stripOut[strip].writeByte(0);
                                                continue;
                                            }
                                            if (off < buf.length) {
                                                stripOut[strip].writeByte(buf[off]);
                                                continue;
                                            }
                                            stripOut[strip].writeByte(0);
                                            continue;
                                        }
                                        off = c * blockSize + ndx + n2;
                                        int realStrip = c * (nStrips / nChannels) + strip;
                                        if (row >= h2 || col >= w) {
                                            stripOut[realStrip].writeByte(0);
                                            continue;
                                        }
                                        if (off < buf.length) {
                                            stripOut[realStrip].writeByte(buf[off]);
                                            continue;
                                        }
                                        stripOut[realStrip].writeByte(0);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        byte[][] strips = new byte[nStrips][];
        for (int strip = 0; strip < nStrips; ++strip) {
            strips[strip] = stripBuf[strip].toByteArray();
            TiffCompression.difference(strips[strip], ifd);
            CodecOptions codecOptions = compression.getCompressionCodecOptions(ifd, this.options);
            codecOptions.height = tileHeight;
            codecOptions.width = tileWidth;
            codecOptions.channels = interleaved ? nChannels : 1;
            strips[strip] = compression.compress(strips[strip], codecOptions);
            if (!LOGGER.isDebugEnabled()) continue;
            LOGGER.debug(String.format("Compressed strip %d/%d length %d", strip + 1, nStrips, strips[strip].length));
        }
        TiffSaver tiffSaver2 = this;
        synchronized (tiffSaver2) {
            this.writeImageIFD(ifd, no, strips, nChannels, last, x, y);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void writeImageIFD(IFD ifd, int no, byte[][] strips, int nChannels, boolean last, int x, int y) throws FormatException, IOException {
        LOGGER.debug("Attempting to write image IFD.");
        boolean isTiled = ifd.isTiled();
        long defaultByteCount = 0L;
        try (RandomAccessInputStream in = null;){
            if (!this.sequentialWrite) {
                if (this.filename != null) {
                    in = new RandomAccessInputStream(this.filename);
                } else if (this.bytes != null) {
                    in = new RandomAccessInputStream(this.bytes);
                } else {
                    throw new IllegalArgumentException("Filename and bytes are null, cannot create new input stream!");
                }
                TiffParser parser = new TiffParser(in);
                long[] ifdOffsets = parser.getIFDOffsets();
                LOGGER.debug("IFD offsets: {}", (Object)Arrays.toString(ifdOffsets));
                if (no < ifdOffsets.length) {
                    this.out.seek(ifdOffsets[no]);
                    LOGGER.debug("Reading IFD from {} in non-sequential write.", (Object)ifdOffsets[no]);
                    ifd = parser.getIFD(ifdOffsets[no]);
                } else if (no > 0 && no - 1 < ifdOffsets.length) {
                    IFD copy = parser.getIFD(ifdOffsets[no - 1]);
                    for (Integer tag : copy.keySet()) {
                        if (ifd.containsKey(tag)) continue;
                        ifd.put(tag, copy.get(tag));
                    }
                    long next = parser.getNextOffset(ifdOffsets[no - 1]);
                    this.out.seek(next);
                }
            } else if (isTiled) {
                defaultByteCount = strips[0].length;
            }
            this.writeIFDStrips(ifd, no, strips, nChannels, last, x, y, defaultByteCount);
        }
    }

    public void writeIFD(IFD ifd, long nextOffset) throws FormatException, IOException {
        this.writeIFD(ifd, nextOffset, false);
    }

    public void writeIFD(IFD ifd, long nextOffset, boolean calculateOffset) throws FormatException, IOException {
        TreeSet keys2 = new TreeSet(ifd.keySet());
        keys2.remove(0);
        keys2.remove(1);
        keys2.remove(3);
        int keyCount = keys2.size();
        long fp = this.out.getFilePointer();
        int bytesPerEntry = this.bigTiff ? 20 : 12;
        int ifdBytes = (this.bigTiff ? 16 : 6) + bytesPerEntry * keyCount;
        if (this.bigTiff) {
            this.out.writeLong(keyCount);
        } else {
            this.out.writeShort(keyCount);
        }
        ByteArrayHandle extra = this.ifdBuffer;
        extra.setLength(0L);
        RandomAccessOutputStream extraStream = new RandomAccessOutputStream(extra);
        for (Integer key : keys2) {
            Object value = ifd.get(key);
            this.writeIFDValue(extraStream, (long)ifdBytes + fp, key, value);
        }
        if (this.bigTiff) {
            this.out.seek(this.out.getFilePointer());
        }
        if (calculateOffset) {
            nextOffset = fp + (long)ifdBytes + extra.length();
        }
        this.writeIntValue(this.out, nextOffset);
        this.out.write(extra.getBytes(), 0, (int)extra.length());
        extraStream.close();
    }

    public void writeIFDValue(RandomAccessOutputStream extraOut, long offset, int tag, Object value) throws FormatException, IOException {
        extraOut.order(this.isLittleEndian());
        if (value instanceof Short) {
            value = new short[]{(Short)value};
        } else if (value instanceof Integer) {
            value = new int[]{(Integer)value};
        } else if (value instanceof Long) {
            value = new long[]{(Long)value};
        } else if (value instanceof TiffRational) {
            value = new TiffRational[]{(TiffRational)value};
        } else if (value instanceof Float) {
            value = new float[]{((Float)value).floatValue()};
        } else if (value instanceof Double) {
            value = new double[]{(Double)value};
        }
        int dataLength = this.bigTiff ? 8 : 4;
        this.out.writeShort(tag);
        if (value instanceof short[]) {
            short[] q = (short[])value;
            this.out.writeShort(IFDType.BYTE.getCode());
            this.writeIntValue(this.out, q.length);
            if (q.length <= dataLength) {
                int i;
                for (i = 0; i < q.length; ++i) {
                    this.out.writeByte(q[i]);
                }
                for (i = q.length; i < dataLength; ++i) {
                    this.out.writeByte(0);
                }
            } else {
                this.writeIntValue(this.out, offset + extraOut.length());
                for (int i = 0; i < q.length; ++i) {
                    extraOut.writeByte(q[i]);
                }
            }
        } else if (value instanceof String) {
            byte[] q = ((String)value).getBytes(Charset.forName("UTF-8"));
            this.out.writeShort(IFDType.ASCII.getCode());
            this.writeIntValue(this.out, q.length + 1);
            if (q.length < dataLength) {
                int i;
                for (i = 0; i < q.length; ++i) {
                    this.out.writeByte(q[i]);
                }
                for (i = q.length; i < dataLength; ++i) {
                    this.out.writeByte(0);
                }
            } else {
                this.writeIntValue(this.out, offset + extraOut.length());
                for (int i = 0; i < q.length; ++i) {
                    extraOut.writeByte(q[i]);
                }
                extraOut.writeByte(0);
            }
        } else if (value instanceof int[]) {
            int[] q = (int[])value;
            this.out.writeShort(IFDType.SHORT.getCode());
            this.writeIntValue(this.out, q.length);
            if (q.length <= dataLength / 2) {
                int i;
                for (i = 0; i < q.length; ++i) {
                    this.out.writeShort(q[i]);
                }
                for (i = q.length; i < dataLength / 2; ++i) {
                    this.out.writeShort(0);
                }
            } else {
                this.writeIntValue(this.out, offset + extraOut.length());
                for (int i = 0; i < q.length; ++i) {
                    extraOut.writeShort(q[i]);
                }
            }
        } else if (value instanceof long[]) {
            int div;
            long[] q = (long[])value;
            int type = this.bigTiff ? IFDType.LONG8.getCode() : IFDType.LONG.getCode();
            this.out.writeShort(type);
            this.writeIntValue(this.out, q.length);
            int n = div = this.bigTiff ? 8 : 4;
            if (q.length <= dataLength / div) {
                int i;
                for (i = 0; i < q.length; ++i) {
                    this.writeIntValue(this.out, q[0]);
                }
                for (i = q.length; i < dataLength / div; ++i) {
                    this.writeIntValue(this.out, 0L);
                }
            } else {
                this.writeIntValue(this.out, offset + extraOut.length());
                for (int i = 0; i < q.length; ++i) {
                    this.writeIntValue(extraOut, q[i]);
                }
            }
        } else if (value instanceof TiffRational[]) {
            Object[] q = value;
            this.out.writeShort(IFDType.RATIONAL.getCode());
            this.writeIntValue(this.out, q.length);
            if (this.bigTiff && q.length == 1) {
                this.out.writeInt((int)((TiffRational)q[0]).getNumerator());
                this.out.writeInt((int)((TiffRational)q[0]).getDenominator());
            } else {
                this.writeIntValue(this.out, offset + extraOut.length());
                for (int i = 0; i < q.length; ++i) {
                    extraOut.writeInt((int)((TiffRational)q[i]).getNumerator());
                    extraOut.writeInt((int)((TiffRational)q[i]).getDenominator());
                }
            }
        } else if (value instanceof float[]) {
            float[] q = (float[])value;
            this.out.writeShort(IFDType.FLOAT.getCode());
            this.writeIntValue(this.out, q.length);
            if (q.length <= dataLength / 4) {
                int i;
                for (i = 0; i < q.length; ++i) {
                    this.out.writeFloat(q[0]);
                }
                for (i = q.length; i < dataLength / 4; ++i) {
                    this.out.writeInt(0);
                }
            } else {
                this.writeIntValue(this.out, offset + extraOut.length());
                for (int i = 0; i < q.length; ++i) {
                    extraOut.writeFloat(q[i]);
                }
            }
        } else if (value instanceof double[]) {
            double[] q = (double[])value;
            this.out.writeShort(IFDType.DOUBLE.getCode());
            this.writeIntValue(this.out, q.length);
            this.writeIntValue(this.out, offset + extraOut.length());
            for (int i = 0; i < q.length; ++i) {
                extraOut.writeDouble(q[i]);
            }
        } else {
            throw new FormatException("Unknown IFD value type (" + value.getClass().getName() + "): " + value);
        }
    }

    public void overwriteLastIFDOffset(RandomAccessInputStream raf) throws FormatException, IOException {
        if (raf == null) {
            throw new FormatException("Output cannot be null");
        }
        TiffParser parser = new TiffParser(raf);
        long[] offsets = parser.getIFDOffsets();
        this.out.seek(raf.getFilePointer() - (long)(this.bigTiff ? 8 : 4));
        this.writeIntValue(this.out, 0L);
    }

    public void overwriteIFDOffset(RandomAccessInputStream raf, long offset, long nextPointer) throws FormatException, IOException {
        if (raf == null) {
            throw new FormatException("Input stream cannot be null");
        }
        int bytesPerEntry = this.bigTiff ? 20 : 12;
        raf.seek(offset);
        long entries = this.bigTiff ? raf.readLong() : (long)raf.readUnsignedShort();
        long overwriteOffset = offset + (long)bytesPerEntry * entries + (long)(this.bigTiff ? 8 : 2);
        this.out.seek(overwriteOffset);
        this.writeIntValue(this.out, nextPointer);
    }

    public void overwriteIFDValue(RandomAccessInputStream raf, int ifd, int tag, Object value) throws FormatException, IOException {
        if (raf == null) {
            throw new FormatException("Output cannot be null");
        }
        LOGGER.debug("overwriteIFDValue (ifd={}; tag={}; value={})", ifd, tag, value);
        raf.seek(0L);
        TiffParser parser = new TiffParser(raf);
        Boolean valid = parser.checkHeader();
        if (valid == null) {
            throw new FormatException("Invalid TIFF header");
        }
        boolean little = valid;
        boolean bigTiff = parser.isBigTiff();
        this.setLittleEndian(little);
        this.setBigTiff(bigTiff);
        long offset = bigTiff ? 8L : 4L;
        int bytesPerEntry = bigTiff ? 20 : 12;
        raf.seek(offset);
        long[] offsets = parser.getIFDOffsets();
        if (ifd >= offsets.length) {
            throw new FormatException("No such IFD (" + ifd + " of " + offsets.length + ")");
        }
        this.overwriteIFDValue(raf, offsets[ifd], tag, value, true);
    }

    public void overwriteIFDValue(RandomAccessInputStream raf, long ifdOffset, int tag, Object value) throws FormatException, IOException {
        this.overwriteIFDValue(raf, ifdOffset, tag, value, false);
    }

    public void overwriteIFDValue(RandomAccessInputStream raf, long ifdOffset, int tag, Object value, boolean skipHeaderCheck) throws FormatException, IOException {
        TiffParser parser;
        if (!skipHeaderCheck) {
            raf.seek(0L);
            parser = new TiffParser(raf);
            Boolean valid = parser.checkHeader();
            if (valid == null) {
                throw new FormatException("Invalid TIFF header");
            }
            boolean little = valid;
            boolean bigTiff = parser.isBigTiff();
            this.setLittleEndian(little);
            this.setBigTiff(bigTiff);
        }
        parser = new TiffParser(raf);
        boolean little = this.isLittleEndian();
        boolean bigTiff = this.isBigTiff();
        long offset = bigTiff ? 8L : 4L;
        int bytesPerEntry = bigTiff ? 20 : 12;
        raf.seek(ifdOffset);
        long num = bigTiff ? raf.readLong() : (long)raf.readUnsignedShort();
        int i = 0;
        while ((long)i < num) {
            raf.seek(ifdOffset + (long)(bigTiff ? 8 : 2) + (long)(bytesPerEntry * i));
            TiffIFDEntry entry = parser.readTiffIFDEntry();
            if (entry.getTag() == tag) {
                long newOffset;
                int newCount;
                ByteArrayHandle ifdBuf = new ByteArrayHandle(bytesPerEntry);
                RandomAccessOutputStream ifdOut = new RandomAccessOutputStream(ifdBuf);
                ByteArrayHandle extraBuf = new ByteArrayHandle();
                RandomAccessOutputStream extraOut = new RandomAccessOutputStream(extraBuf);
                ifdOut.order(little);
                extraOut.order(little);
                TiffSaver saver = new TiffSaver(ifdOut, ifdBuf);
                saver.setLittleEndian(little);
                saver.setBigTiff(bigTiff);
                saver.writeIFDValue(extraOut, entry.getValueOffset(), tag, value);
                ifdOut.close();
                saver.close();
                extraOut.close();
                ifdBuf.seek(0L);
                extraBuf.seek(0L);
                short newTag = ifdBuf.readShort();
                short newType = ifdBuf.readShort();
                if (bigTiff) {
                    newCount = (int)ifdBuf.readLong();
                    newOffset = ifdBuf.readLong();
                } else {
                    newCount = ifdBuf.readInt();
                    newOffset = ifdBuf.readInt();
                }
                LOGGER.debug("overwriteIFDValue:");
                LOGGER.debug("\told ({});", (Object)entry);
                LOGGER.debug("\tnew: (tag={}; type={}; count={}; offset={})", (int)newTag, (int)newType, newCount, newOffset);
                int bytesPerElement = entry.getType().getBytesPerElement();
                if ((long)entry.getValueCount() > offset / (long)bytesPerElement) {
                    this.out.seek(entry.getValueOffset());
                    for (int b = 0; b < entry.getValueCount() * bytesPerElement; ++b) {
                        this.out.writeByte(0);
                    }
                }
                if (extraBuf.length() == 0L) {
                    LOGGER.debug("overwriteIFDValue: new entry is inline");
                } else if (entry.getValueOffset() + (long)(entry.getValueCount() * bytesPerElement) == raf.length()) {
                    newOffset = entry.getValueOffset();
                    LOGGER.debug("overwriteIFDValue: old entry is at EOF");
                } else if (newCount <= entry.getValueCount()) {
                    newOffset = entry.getValueOffset();
                    LOGGER.debug("overwriteIFDValue: new entry is <= old entry");
                } else {
                    newOffset = raf.length();
                    LOGGER.debug("overwriteIFDValue: old entry will be orphaned");
                }
                this.out.seek(ifdOffset + (long)(bigTiff ? 8 : 2) + (long)(bytesPerEntry * i) + 2L);
                this.out.writeShort(newType);
                this.writeIntValue(this.out, newCount);
                this.writeIntValue(this.out, newOffset);
                if (extraBuf.length() > 0L) {
                    this.out.seek(newOffset);
                    this.out.write(extraBuf.getByteBuffer(), 0, (int)extraBuf.length());
                }
                return;
            }
            ++i;
        }
        throw new FormatException("Tag not found (" + IFD.getIFDTagName(tag) + ")");
    }

    public void overwriteComment(RandomAccessInputStream in, Object value) throws FormatException, IOException {
        this.overwriteIFDValue(in, 0, 270, value);
    }

    private void writeIntValue(RandomAccessOutputStream out, long offset) throws IOException {
        if (this.bigTiff) {
            out.writeLong(offset);
        } else {
            out.writeInt((int)offset);
        }
    }

    public void makeValidIFD(IFD ifd, int pixelType, int nChannels) {
        int bytesPerPixel = FormatTools.getBytesPerPixel(pixelType);
        int bps = 8 * bytesPerPixel;
        int[] bpsArray = new int[nChannels];
        Arrays.fill(bpsArray, bps);
        ifd.putIFDValue(258, bpsArray);
        if (FormatTools.isFloatingPoint(pixelType)) {
            ifd.putIFDValue(339, 3);
        }
        if (ifd.getIFDValue(259) == null) {
            ifd.putIFDValue(259, TiffCompression.UNCOMPRESSED.getCode());
        }
        PhotoInterp pi = PhotoInterp.BLACK_IS_ZERO;
        if (nChannels == 1 && ifd.getIFDValue(320) != null) {
            pi = PhotoInterp.RGB_PALETTE;
        } else if (nChannels == 3 || nChannels == 4) {
            pi = ifd.getIFDValue(259).equals(TiffCompression.JPEG.getCode()) ? PhotoInterp.Y_CB_CR : PhotoInterp.RGB;
            if (nChannels == 4) {
                int[] extraSamplesArray = new int[]{0};
                ifd.putIFDValue(338, extraSamplesArray);
            }
        }
        ifd.putIFDValue(262, pi.getCode());
        ifd.putIFDValue(277, nChannels);
        if (ifd.get(282) == null) {
            ifd.putIFDValue(282, new TiffRational(1L, 1L));
        }
        if (ifd.get(283) == null) {
            ifd.putIFDValue(283, new TiffRational(1L, 1L));
        }
        if (ifd.get(305) == null) {
            ifd.putIFDValue(305, FormatTools.CREATOR);
        }
        if (ifd.get(278) == null && ifd.get(322) == null && ifd.get(323) == null) {
            ifd.putIFDValue(278, new long[]{1L});
        }
        if (ifd.get(270) == null) {
            ifd.putIFDValue(270, "");
        }
    }

    private void writeIFDStrips(IFD ifd, int no, byte[][] strips, int nChannels, boolean last, int x, int y, long defaultByteCount) throws FormatException, IOException {
        long[] offsets;
        long[] byteCounts;
        int tilesPerRow = (int)ifd.getTilesPerRow();
        int tilesPerColumn = (int)ifd.getTilesPerColumn();
        boolean interleaved = ifd.getPlanarConfiguration() == 1;
        boolean isTiled = ifd.isTiled();
        long totalTiles = tilesPerRow * tilesPerColumn;
        if (!interleaved) {
            totalTiles *= (long)nChannels;
        }
        if (ifd.containsKey(279) || ifd.containsKey(325)) {
            long[] ifdByteCounts;
            byteCounts = ifdByteCounts = isTiled ? ifd.getIFDLongArray(325) : ifd.getStripByteCounts();
        } else {
            byteCounts = new long[(int)totalTiles];
            Arrays.fill(byteCounts, defaultByteCount);
        }
        int tileOrStripOffsetX = x / (int)ifd.getTileWidth();
        int tileOrStripOffsetY = y / (int)ifd.getTileLength();
        int firstOffset = tileOrStripOffsetY * tilesPerRow + tileOrStripOffsetX;
        if (ifd.containsKey(273) || ifd.containsKey(324)) {
            long[] ifdOffsets = isTiled ? ifd.getIFDLongArray(324) : ifd.getStripOffsets();
            offsets = (long[])ifdOffsets.clone();
        } else {
            offsets = new long[(int)totalTiles];
            if (isTiled && tileOrStripOffsetX == 0 && tileOrStripOffsetY == 0) {
                this.sequentialTileOffsets = offsets;
                this.sequentialTileByteCounts = byteCounts;
            } else if (isTiled) {
                offsets = this.sequentialTileOffsets;
                byteCounts = this.sequentialTileByteCounts;
            }
        }
        if (isTiled) {
            ifd.putIFDValue(325, byteCounts);
            ifd.putIFDValue(324, offsets);
        } else {
            ifd.putIFDValue(279, byteCounts);
            ifd.putIFDValue(273, offsets);
        }
        long fp = this.out.getFilePointer();
        if (isTiled && tileOrStripOffsetX == 0 && tileOrStripOffsetY == 0) {
            this.sequentialTileFilePointer = fp;
        } else if (isTiled) {
            fp = this.sequentialTileFilePointer;
        }
        if (fp == this.out.getFilePointer()) {
            this.writeIFD(ifd, 0L);
        }
        int tileCount = isTiled ? tilesPerRow * tilesPerColumn : 1;
        long stripStartPos = this.out.length();
        long totalStripSize = 0L;
        for (int i = 0; i < strips.length; ++i) {
            totalStripSize += (long)strips[i].length;
        }
        this.out.seek(stripStartPos + totalStripSize);
        this.out.seek(stripStartPos);
        ByteArrayOutputStream stripBuffer = new ByteArrayOutputStream();
        long stripOffset = 0L;
        for (int i = 0; i < strips.length; ++i) {
            int index = interleaved ? i : i / nChannels * nChannels;
            int c = interleaved ? 0 : i % nChannels;
            int thisOffset = firstOffset + index + c * tileCount;
            offsets[thisOffset] = stripStartPos + stripOffset;
            byteCounts[thisOffset] = strips[i].length;
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug(String.format("Writing tile/strip %d/%d size: %d offset: %d", thisOffset + 1, totalTiles, byteCounts[thisOffset], offsets[thisOffset]));
            }
            stripBuffer.write(strips[i]);
            stripOffset += (long)strips[i].length;
        }
        stripBuffer.writeTo(this.out);
        stripBuffer.close();
        stripBuffer = null;
        if (isTiled) {
            ifd.putIFDValue(325, byteCounts);
            ifd.putIFDValue(324, offsets);
        } else {
            ifd.putIFDValue(279, byteCounts);
            ifd.putIFDValue(273, offsets);
        }
        long endFP = this.out.getFilePointer();
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Offset before IFD write: {} Seeking to: {}", (Object)this.out.getFilePointer(), (Object)fp);
        }
        this.out.seek(fp);
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Writing tile/strip offsets: {}", (Object)Arrays.toString(offsets));
            LOGGER.debug("Writing tile/strip byte counts: {}", (Object)Arrays.toString(byteCounts));
        }
        this.writeIFD(ifd, last ? 0L : endFP);
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Offset after IFD write: {}", (Object)this.out.getFilePointer());
        }
    }
}

