package plugins.nchenouard.trackprocessorperformance;

import icy.canvas.IcyCanvas;
import icy.canvas.IcyCanvas2D;
import icy.gui.frame.IcyFrame;
import icy.gui.frame.progress.FailedAnnounceFrame;
import icy.gui.frame.progress.ProgressFrame;
import icy.main.Icy;
import icy.painter.Overlay;
import icy.sequence.Sequence;
import icy.sequence.SequenceEvent;
import icy.sequence.SequenceListener;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.FlowLayout;
import java.awt.Graphics2D;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.RenderingHints;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.geom.Line2D;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.text.NumberFormat;
import java.text.ParseException;
import java.util.ArrayList;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import javax.swing.border.TitledBorder;

import plugins.fab.trackmanager.PluginTrackManagerProcessor;
import plugins.fab.trackmanager.TrackGroup;
import plugins.fab.trackmanager.TrackSegment;
import plugins.nchenouard.spot.Detection;

/**
 * Main class for the track processor that computes tracking quality with respect to a reference set
 * of tracks.
 * 
 * @version February 3, 2012
 * @author Nicolas Chenouard
 */

public class TrackProcessorPerformance extends PluginTrackManagerProcessor implements SequenceListener
{

    TrackGroup referenceGroup = null;
    TrackGroup candidateGroup = null;
    JLabel refGroupsLabel = new JLabel("Reference group: -");
    JLabel candidateGroupsLabel = new JLabel("Candidate group: -");
    JFormattedTextField maxDistTF;
    NumberFormat maxDistFormat = NumberFormat.getNumberInstance();

    ArrayList<TrackPair> trackPairs = new ArrayList<TrackPair>();
    ArrayList<TrackSegment> recoveredTracks = new ArrayList<TrackSegment>();
    ArrayList<TrackSegment> correctTracks = new ArrayList<TrackSegment>();
    ArrayList<TrackSegment> missedTracks = new ArrayList<TrackSegment>();
    ArrayList<TrackSegment> spuriousTracks = new ArrayList<TrackSegment>();

    Color correctTracksColor = Color.GREEN;
    Color recoveredTracksColor = Color.CYAN;
    Color missedTracksColor = Color.MAGENTA;
    Color spuriousTracksColor = Color.RED;

    MeasuresPanel measuresPanel;

    JButton pairButton = new JButton("Pair tracks");

    ArrayList<Thread> runningThreads = new ArrayList<Thread>();

    Overlay overlay = new PerformanceOverlay("Tracking performance display");
    
    public TrackProcessorPerformance()
    {
            setName("Tracking Performance Computation");

            panel.setLayout(new GridBagLayout());
            GridBagConstraints c = new GridBagConstraints();
            c.fill = GridBagConstraints.HORIZONTAL;
            c.weightx = 1.0;
            c.weighty = 0;

            c.gridx = 0;
            c.gridy = 0;
            panel.add(buildTrackGroupSelectionPanel(), c);

            c.gridx = 0;
            c.gridy = 1;
            panel.add(buildPairingPanel(), c);

            c.gridx = 0;
            c.gridy = 2;
            panel.add(buildVisualizationPanel(), c);

            measuresPanel = buildMeasuresPanel();
            c.gridx = 0;
            c.gridy = 3;
            panel.add(measuresPanel, c);
    }

    private JPanel buildPairingPanel()
    {
        JPanel pairingPanel = new JPanel();
        pairingPanel.setLayout(new GridBagLayout());
        GridBagConstraints c = new GridBagConstraints();
        c.fill = GridBagConstraints.HORIZONTAL;
        c.weightx = 1.0;
        c.weighty = 0;

        c.gridx = 0;
        c.gridy = 0;
        pairingPanel.add(pairButton, c);
        pairButton.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent arg0)
            {
                if (referenceGroup != null && candidateGroup != null)
                {
                    Thread pairThread = new Thread()
                    {
                        public void run()
                        {
                            ProgressFrame frame = new ProgressFrame("Pairing tracks");
                            SwingUtilities.invokeLater(new Runnable()
                            {
                                @Override
                                public void run()
                                {
                                    pairButton.setEnabled(false);
                                }
                            });
                            pairTracks(referenceGroup.getTrackSegmentList(), candidateGroup.getTrackSegmentList());
                            frame.close();
                            SwingUtilities.invokeLater(new Runnable()
                            {
                                @Override
                                public void run()
                                {
                                    pairButton.setEnabled(true);
                                }
                            });
                            runningThreads.remove(this);
                        }
                    };
                    runningThreads.add(pairThread);
                    pairThread.start();
                    // try {
                    // pairThread.join();
                    // } catch (InterruptedException e) {
                    // e.printStackTrace();
                    // }
                }
            }
        });

        JLabel maxDistLabel = new JLabel("Maximum distance between detections");
        c.gridx = 0;
        c.gridy = 1;
        pairingPanel.add(maxDistLabel, c);

        maxDistTF = new JFormattedTextField(maxDistFormat);
        maxDistTF.setText("5");
        c.gridx = 0;
        c.gridy = 2;
        pairingPanel.add(maxDistTF, c);
        pairingPanel.setBorder(new TitledBorder("Track pairing"));

        return pairingPanel;
    }

    private JPanel buildTrackGroupSelectionPanel()
    {
        JPanel groupSelectionPanel = new JPanel();
        groupSelectionPanel.setLayout(new GridBagLayout());
        GridBagConstraints c = new GridBagConstraints();
        c.fill = GridBagConstraints.HORIZONTAL;
        c.weightx = 1.0;
        c.weighty = 0;

        c.gridx = 0;
        c.gridy = 0;
        JButton selectGroupsButton = new JButton("Select track groups");
        groupSelectionPanel.add(selectGroupsButton, c);
        selectGroupsButton.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent arg0)
            {
                selectGroups();
            }

        });

        c.gridx = 0;
        c.gridy = 1;
        JPanel refGroupsLabelPanel = new JPanel();
        refGroupsLabelPanel.setLayout(new FlowLayout(FlowLayout.LEADING));
        refGroupsLabelPanel.add(refGroupsLabel);
        groupSelectionPanel.add(refGroupsLabelPanel, c);

        c.gridx = 0;
        c.gridy = 2;
        JPanel candidateGroupsLabelPanel = new JPanel();
        candidateGroupsLabelPanel.setLayout(new FlowLayout(FlowLayout.LEADING));
        candidateGroupsLabelPanel.add(candidateGroupsLabel);
        groupSelectionPanel.add(candidateGroupsLabelPanel, c);
        groupSelectionPanel.setBorder(new TitledBorder("Track groups selection"));
        return groupSelectionPanel;
    }

    final JCheckBox colorTracksBox = new JCheckBox();
    final JCheckBox displayPairsBox = new JCheckBox();
    final JCheckBox displaySpuriousBox = new JCheckBox();
    final JCheckBox displayMissingBox = new JCheckBox();

    private JPanel buildVisualizationPanel()
    {
        JPanel visualizationPanel = new JPanel();
        visualizationPanel.setLayout(new GridBagLayout());
        GridBagConstraints c = new GridBagConstraints();
        c.fill = GridBagConstraints.HORIZONTAL;
        c.weightx = 1.0;
        c.weighty = 0;

        c.gridx = 0;
        c.gridy = 0;
        colorTracksBox.setText("Color tracks");
        colorTracksBox.setSelected(true);
        visualizationPanel.add(colorTracksBox, c);

        c.gridx = 1;
        c.gridy = 0;
        displayPairsBox.setText("Display pairs");
        displayPairsBox.setSelected(true);
        visualizationPanel.add(displayPairsBox, c);

        c.gridx = 0;
        c.gridy = 1;
        displaySpuriousBox.setText("Display spurious tracks");
        displaySpuriousBox.setSelected(true);
        visualizationPanel.add(displaySpuriousBox, c);

        c.gridx = 1;
        c.gridy = 1;
        displayMissingBox.setText("Display missing tracks");
        displayMissingBox.setSelected(true);
        visualizationPanel.add(displayMissingBox, c);

        ItemListener listener = new ItemListener()
        {

            @Override
            public void itemStateChanged(ItemEvent arg0)
            {
                refreshSequences();
            }
        };
        colorTracksBox.addItemListener(listener);
        displayPairsBox.addItemListener(listener);
        displaySpuriousBox.addItemListener(listener);
        displayMissingBox.addItemListener(listener);

        visualizationPanel.setBorder(new TitledBorder("Visualization options"));
        return visualizationPanel;
    }

    class MeasuresPanel extends JPanel
    {
        /**
		 * 
		 */
        private static final long serialVersionUID = -8802797303612586042L;

        JLabel pairsDistanceLabel = new JLabel();
        JLabel pairsNormalizedDistanceLabel = new JLabel();
        JLabel pairsFullDistanceLabel = new JLabel();

        JLabel spuriousTracksLabel = new JLabel();
        JLabel missedTracksLabel = new JLabel();
        JLabel correctTracksLabel = new JLabel();
        JLabel tracksSimilarityLabel = new JLabel();

        JLabel rmseLabel = new JLabel();
        JLabel minDistanceLabel = new JLabel();
        JLabel maxDistanceLabel = new JLabel();
        JLabel stdDistanceLabel = new JLabel();
        
        double pairsDistance;
        double pairsNormalizedDistance;
        double pairsFullDistance;

        int numSpuriousTracks;
        int numMissedTracks;
        int numCorrectTracks;
        int numRefTracks;
        int numCandidateTracks;
        double tracksSimilarity;

        JLabel numRecoveredDetectionsLabel = new JLabel();
        JLabel numMissedDetectionsLabel = new JLabel();
        JLabel numWrongDetectionsLabel = new JLabel();
        JLabel detectionsSimilarityLabel = new JLabel();

        int numRecoveredDetections;
        int numMissedDetections;
        int numWrongDetections;
        int numRefDetections;
        int numCandidateDetections;
        double detectionsSimilarity;

        double rmseDetection;
		double minDistanceDetection;
		double maxDistanceDetection;
		double stdDistanceDetection;
		
        JButton saveResultsButton;

        public MeasuresPanel()
        {
            this.setLayout(new GridBagLayout());
            GridBagConstraints c = new GridBagConstraints();
            c.fill = GridBagConstraints.HORIZONTAL;
            c.weightx = 1.0;
            c.weighty = 0;

            c.gridx = 0;
            c.gridy = 0;
            c.weightx = 1;
            c.gridwidth = 2;
            JPanel globalPanel = new JPanel();
            globalPanel.add(new JLabel("Global measures"));
            this.add(globalPanel, c);
            c.weightx = 0.25;
            c.gridwidth = 1;

            c.gridx = 0;
            c.gridy++;
            c.weightx = 0.25;
            this.add(new JLabel("Pairing distance"), c);

            c.gridx = 1;
            c.weightx = 0.75;
            this.add(pairsDistanceLabel, c);

            c.gridx = 0;
            c.gridy++;
            c.weightx = 0.25;
            this.add(new JLabel("Normalized pairing score (alpha)"), c);

            c.gridx = 1;
            c.weightx = 0.75;
            this.add(pairsNormalizedDistanceLabel, c);

            c.gridx = 0;
            c.gridy++;
            c.weightx = 0.25;
            this.add(new JLabel("Full normalized score (beta)"), c);

            c.gridx = 1;
            c.weightx = 0.75;
            this.add(pairsFullDistanceLabel, c);

            c.gridx = 0;
            c.gridy++;
            c.weightx = 1;
            c.gridwidth = 2;
            JPanel tracksPanel = new JPanel();
            tracksPanel.add(new JLabel("Tracks"));
            this.add(tracksPanel, c);
            c.weightx = 0.25;
            c.gridwidth = 1;

            c.gridx = 0;
            c.gridy++;
            c.weightx = 0.25;
            this.add(new JLabel("Similarity between tracks (Jaccard)"), c);

            c.gridx = 1;
            c.weightx = 0.75;
            this.add(tracksSimilarityLabel, c);

            c.gridx = 0;
            c.gridy++;
            c.weightx = 0.25;
            this.add(new JLabel("Number of paired tracks"), c);

            c.gridx = 1;
            c.weightx = 0.75;
            this.add(correctTracksLabel, c);

            c.gridx = 0;
            c.gridy++;
            c.weightx = 0.25;
            this.add(new JLabel("Number of missed tracks"), c);

            c.gridx = 1;
            c.weightx = 0.75;
            this.add(missedTracksLabel, c);

            c.gridx = 0;
            c.gridy++;
            c.weightx = 0.25;
            this.add(new JLabel("Number of spurious tracks"), c);

            c.gridx = 1;
            c.weightx = 0.75;
            this.add(spuriousTracksLabel, c);

            c.gridx = 0;
            c.gridy++;
            c.weightx = 1;
            c.gridwidth = 2;
            JPanel detectionsPanel = new JPanel();
            detectionsPanel.add(new JLabel("Detections"));
            this.add(detectionsPanel, c);
            c.weightx = 0.25;
            c.gridwidth = 1;

            c.gridx = 0;
            c.gridy++;
            c.weightx = 0.25;
            this.add(new JLabel("Similarity between detections (Jaccard)"), c);

            c.gridx = 1;
            c.weightx = 0.75;
            this.add(detectionsSimilarityLabel, c);

            c.gridx = 0;
            c.gridy++;
            c.weightx = 0.25;
            this.add(new JLabel("Number of paired detections"), c);

            c.gridx = 1;
            c.weightx = 0.75;
            this.add(numRecoveredDetectionsLabel, c);

            c.gridx = 0;
            c.gridy++;
            c.weightx = 0.25;
            this.add(new JLabel("Number of missed detections"), c);

            c.gridx = 1;
            c.weightx = 0.75;
            this.add(numMissedDetectionsLabel, c);

            c.gridx = 0;
            c.gridy++;
            c.weightx = 0.25;
            this.add(new JLabel("Number of spurious detections"), c);

            c.gridx = 1;
            c.weightx = 0.75;
            this.add(numWrongDetectionsLabel, c);

            c.gridx = 0;
            c.gridy++;
            c.weightx = 0.25;
            c.gridwidth = 2;
            
            c.gridx = 0;
            c.gridy++;
            c.weightx = 1;
            c.gridwidth = 2;
            JPanel detectionAccuracyPanel = new JPanel();
            detectionAccuracyPanel.add(new JLabel("Detection accuracy"));
            this.add(detectionAccuracyPanel, c);
            c.weightx = 0.25;
            c.gridwidth = 1;

            c.gridx = 0;
            c.gridy++;
            c.weightx = 0.25;
            this.add(new JLabel("Root mean-square error"), c);

            c.gridx = 1;
            c.weightx = 0.75;
            this.add(rmseLabel, c);
            
            c.gridx = 0;
            c.gridy++;
            c.weightx = 0.25;
            this.add(new JLabel("Minimum distance"), c);

            c.gridx = 1;
            c.weightx = 0.75;
            this.add(minDistanceLabel, c);

            c.gridx = 0;
            c.gridy++;
            c.weightx = 0.25;
            this.add(new JLabel("Maximum distance"), c);

            c.gridx = 1;
            c.weightx = 0.75;
            this.add(maxDistanceLabel, c);
            
            c.gridx = 0;
            c.gridy++;
            c.weightx = 0.25;
            this.add(new JLabel("Distance standard deviation"), c);

            c.gridx = 1;
            c.weightx = 0.75;
            this.add(stdDistanceLabel, c);
            
            c.gridy++;
            c.gridx = 0;
            c.weightx = 1;
            c.gridwidth = 2;
            saveResultsButton = new JButton("Save results");
            saveResultsButton.addActionListener(new ActionListener()
            {
                @Override
                public void actionPerformed(ActionEvent arg0)
                {
                    saveResults();
                }
            });
            JPanel savePanel = new JPanel();
            savePanel.add(saveResultsButton);
            this.add(savePanel, c);
            c.gridwidth = 1;

            resetScores();
        }

        /**
         * update the GUI with the new values for the tracking criteria
         */
        public void setScores(int numRefTracks, int numCandidateTracks, int numRefDetections,
                int numCandidateDetections, double pairsDistance, double pairsNormalizedDistance,
                double pairsFullDistance, int numSpurious, int numMissed, int numCorrect,
                int numRecoveredDetections, int numMissedDetections, int numWrongDetections,
                double rmseDetection, double minDistanceDetection, double maxDistanceDetection, double stdDistanceDetection
        		)
        {
            this.numCandidateDetections = numCandidateDetections;
            this.numCandidateTracks = numCandidateTracks;
            this.numRefDetections = numRefDetections;
            this.numRefTracks = numRefTracks;
            this.pairsDistance = pairsDistance;
            this.pairsNormalizedDistance = pairsNormalizedDistance;
            this.pairsFullDistance = pairsFullDistance;
            this.numSpuriousTracks = numSpurious;
            this.numMissedTracks = numMissed;
            this.numCorrectTracks = numCorrect;
            this.numRecoveredDetections = numRecoveredDetections;
            this.numMissedDetections = numMissedDetections;
            this.numWrongDetections = numWrongDetections;
            this.detectionsSimilarity = (double) numRecoveredDetections
                    / ((double) numRecoveredDetections + (double) numMissedDetections + (double) numWrongDetections);
            this.tracksSimilarity = (double) numCorrectTracks
                    / ((double) numCorrectTracks + (double) numMissedTracks + (double) numSpuriousTracks);
           
            this.rmseDetection= rmseDetection;
    		this.minDistanceDetection = minDistanceDetection;
    		this.maxDistanceDetection = maxDistanceDetection;
    		this.stdDistanceDetection = stdDistanceDetection;
            SwingUtilities.invokeLater(new Runnable()
            {

                @Override
                public void run()
                {
                    refreshLabels();
                }
            });
        }

        public void resetScores()
        {
            this.pairsDistance = 0;
            this.pairsNormalizedDistance = 0;
            this.pairsFullDistance = 0;
            this.numSpuriousTracks = 0;
            this.numMissedTracks = 0;
            this.numCorrectTracks = 0;
            this.tracksSimilarity = 0;
            this.numCandidateDetections = 0;
            this.numCandidateTracks = 0;
            this.numRefTracks = 0;
            this.numRefDetections = 0;
            pairsDistanceLabel.setText(": -");
            pairsNormalizedDistanceLabel.setText(": -");
            pairsFullDistanceLabel.setText(": -");
            correctTracksLabel.setText(": -");
            missedTracksLabel.setText(": -");
            spuriousTracksLabel.setText(": -");
            tracksSimilarityLabel.setText(": -");

            this.numRecoveredDetections = 0;
            this.numMissedDetections = 0;
            this.numWrongDetections = 0;
            this.detectionsSimilarity = 0;
            numRecoveredDetectionsLabel.setText(": -");
            numMissedDetectionsLabel.setText(": -");
            numWrongDetectionsLabel.setText(": -");
            detectionsSimilarityLabel.setText(": -");
            
            this.rmseDetection = 0d;
            this.minDistanceDetection = 0d;
            this.maxDistanceDetection = 0d;
            this.stdDistanceDetection = 0d;
            rmseLabel.setText(": -");
            minDistanceLabel.setText(": -");
            maxDistanceLabel.setText(": -");
            stdDistanceLabel.setText(": -");
        }

        public void refreshLabels()
        {
            pairsDistanceLabel.setText(": " + pairsDistance);
            pairsNormalizedDistanceLabel.setText(": " + pairsNormalizedDistance);
            pairsFullDistanceLabel.setText(": " + pairsFullDistance);
            tracksSimilarityLabel.setText(": " + tracksSimilarity);
            correctTracksLabel.setText(": " + numCorrectTracks + " (out of " + numRefTracks + ")");
            missedTracksLabel.setText(": " + numMissedTracks + " (out of " + numRefTracks + ")");
            spuriousTracksLabel.setText(": " + numSpuriousTracks);
            // spuriousTracksLabel.setText(": "+numSpuriousTracks+" (out of "+numCandidateTracks+")");
            numRecoveredDetectionsLabel.setText(": " + numRecoveredDetections + " (out of " + numRefDetections + ")");
            numMissedDetectionsLabel.setText(": " + numMissedDetections + " (out of " + numRefDetections + ")");
            numWrongDetectionsLabel.setText(": " + numWrongDetections);
            // numWrongDetectionsLabel.setText(": "+numWrongDetections+" (out of "+numCandidateDetections+")");
            detectionsSimilarityLabel.setText(": " + detectionsSimilarity);
            rmseLabel.setText(": " + rmseDetection);
            minDistanceLabel.setText(": " + minDistanceDetection);
            maxDistanceLabel.setText(": " + maxDistanceDetection);
            stdDistanceLabel.setText(": " + stdDistanceDetection);
        }

        private void saveResults()
        {
            File file = getValidSaveFile();
            if (file == null)
                return;
            try
            {
                FileWriter outFile = new FileWriter(file);
                PrintWriter out = new PrintWriter(outFile);

                out.println(pairsDistance + "\t : pairing distance");
                out.println(pairsNormalizedDistance + "\t : normalized pairing score (alpha)");
                out.println(pairsFullDistance + "\t : full normalized score (beta)");
                out.println(numRefTracks + "\t : number of reference tracks");
                out.println(numCandidateTracks + "\t : number of candidate tracks");
                out.println(tracksSimilarity + "\t : Similarity between tracks (Jaccard)");
                out.println(numCorrectTracks + "\t : number of paired tracks (out of " + numRefTracks + ")");
                out.println(numMissedTracks + "\t : number of missed tracks (out of " + numRefTracks + ")");
                out.println(numSpuriousTracks + "\t : number of spurious tracks)");
                // out.println(numSpuriousTracks+"\t : number of spurious tracks (out of "+numCandidateTracks+")");
                out.println(numRefDetections + "\t : number of reference detections");
                out.println(numCandidateDetections + "\t : number of candidate detections");
                out.println(detectionsSimilarity + "\t : Similarity between detections (Jaccard)");
                out.println(numRecoveredDetections + "\t : number of paired detections (out of " + numRefDetections
                        + ")");
                out.println(numMissedDetections + "\t : number of missed detections (out of " + numRefDetections + ")");
                out.println(numWrongDetections + "\t : number of spurious detections");
                // out.println(numWrongDetections+"\t : number of spurious detections (out of "+numCandidateDetections+")");
                out.close();
            }
            catch (IOException e)
            {
                e.printStackTrace();
                new FailedAnnounceFrame("Writing the save file has failed.");
            }
        }
    }

    private MeasuresPanel buildMeasuresPanel()
    {
        MeasuresPanel measuresPanel = new MeasuresPanel();
        measuresPanel.setBorder(new TitledBorder("Tracking performance"));
        return measuresPanel;
    }

    /**
     * ask the user to choose a file in which to save results
     */
    private File getValidSaveFile()
    {
        boolean hasValidSaveFile = false;
        File file = null;
        while (!hasValidSaveFile)
        {
            JFileChooser fileChooser = new JFileChooser();
            fileChooser.setMultiSelectionEnabled(false);
            fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
            fileChooser.setName("Text file for saving results");
            int returnVal = fileChooser.showDialog(panel, "Set as save file");
            if (returnVal == JFileChooser.APPROVE_OPTION)
            {
                file = fileChooser.getSelectedFile();
                if (file.exists())
                {
                    int n = JOptionPane.showConfirmDialog(Icy.getMainInterface().getMainFrame(),
                            "This file already exists. Do you want to overwrite it?",
                            "Save tracking performance results", JOptionPane.YES_NO_CANCEL_OPTION);
                    switch (n)
                    {
                        case JOptionPane.YES_OPTION:
                            hasValidSaveFile = true;
                            break;
                        case JOptionPane.CANCEL_OPTION:
                            hasValidSaveFile = true;
                            file = null;
                            break;
                        case JOptionPane.NO_OPTION:
                            hasValidSaveFile = false;
                    }
                }
                else
                    hasValidSaveFile = true;
            }
            else
                return null;
        }
        return file;
    }

    /**
     * ask the user to select the groups of tracks to compare
     */
    public void selectGroups()
    {
        GroupSelectionFrame frame = new GroupSelectionFrame();
        addIcyFrame(frame);
        frame.setVisible(true);
    }

    public void refreshSequences()
    {
        for (Sequence sequence : Icy.getMainInterface().getSequencesContaining(overlay))
        {
            sequence.overlayChanged(overlay);
        }
    }

    public void setTrackGroups(TrackGroup refTG, TrackGroup candidateTG)
    {
        trackPairs.clear();
        recoveredTracks.clear();
        missedTracks.clear();
        spuriousTracks.clear();
        correctTracks.clear();
        measuresPanel.resetScores();

        referenceGroup = refTG;
        candidateGroup = candidateTG;
        if (referenceGroup == null)
            refGroupsLabel.setText("Reference group: -");
        else
            refGroupsLabel.setText("Reference group: " + referenceGroup.getDescription());
        if (candidateGroup == null)
            candidateGroupsLabel.setText("Candidate group: -");
        else
            candidateGroupsLabel.setText("Candidate group: " + candidateGroup.getDescription());
    }

    public PerformanceAnalyzer pairTracks(ArrayList<TrackSegment> trackSegmentList1,
            ArrayList<TrackSegment> trackSegmentList2, boolean newMethod)
    {  	
    	OneToOneMatcher matcher = new OneToOneMatcher(trackSegmentList1, trackSegmentList2);
        DistanceTypes distType = DistanceTypes.DISTANCE_EUCLIDIAN;
        double maxDist;
        try
        {
            maxDist = maxDistFormat.parse(maxDistTF.getText()).doubleValue();
        }
        catch (ParseException e)
        {
            e.printStackTrace();
            return null;
        }
        if (maxDist < 0)
            return null;
        ArrayList<TrackPair> pairs = new ArrayList<TrackPair>();
        try
        {
            pairs.addAll(matcher.pairTracks(maxDist, distType, newMethod));
        }
        catch (Exception e)
        {
            new FailedAnnounceFrame("Track pairing failed. See log message.");
            e.printStackTrace();
            pairs.clear();
        }

        //debug1.displayMs();

        // remove spurious candidate tracks
        recoveredTracks.clear();
        correctTracks.clear();
        missedTracks.clear();
        spuriousTracks.clear();

        //Chronometer debug2 = new Chronometer("Second score loop (pairing)");

        for (TrackPair tp : pairs)
        {
            if (tp.candidateTrack.getDetectionList().isEmpty())
            {
                tp.candidateTrack = null;
                missedTracks.add(tp.referenceTrack);
            }
            else
            {
                recoveredTracks.add(tp.referenceTrack);
                correctTracks.add(tp.candidateTrack);
            }
        }
        for (TrackSegment ts : trackSegmentList2)
        {
            if (!correctTracks.contains(ts))
                spuriousTracks.add(ts);
        }

        //debug2.displayMs();

        trackPairs.clear();
        trackPairs.addAll(pairs);

        //Chronometer debug3 = new Chronometer("Perf Analyzer");
        PerformanceAnalyzer analyzer = new PerformanceAnalyzer(trackSegmentList1, trackSegmentList2, trackPairs);
        //debug3.displayMs();
        
        double[] detectionDistance = analyzer.getDistanceDetectionData(maxDist);
       
        measuresPanel.setScores(analyzer.getNumRefTracks(), analyzer.getNumCandidateTracks(),
                analyzer.getNumRefDetections(), analyzer.getNumCandidateDetections(),
                analyzer.getPairedTracksDistance(distType, maxDist),
                analyzer.getPairedTracksNormalizedDistance(distType, maxDist),
                analyzer.getFullTrackingScore(distType, maxDist), analyzer.getNumSpuriousTracks(),
                analyzer.getNumMissedTracks(), analyzer.getNumPairedTracks(), analyzer.getNumPairedDetections(maxDist),
                analyzer.getNumMissedDetections(maxDist), analyzer.getNumWrongDetections(maxDist),
                detectionDistance[0], detectionDistance[1], detectionDistance[2], detectionDistance[3]
        		);        
        return analyzer;
    }

    public PerformanceAnalyzer pairTracks(ArrayList<TrackSegment> trackSegmentList1,
            ArrayList<TrackSegment> trackSegmentList2)
    {
        return pairTracks(trackSegmentList1, trackSegmentList2, true);
    }

    public static ArrayList<TrackPair> pairTracks(ArrayList<TrackSegment> trackSegmentList1,
            ArrayList<TrackSegment> trackSegmentList2, DistanceTypes distType, double maxDist) throws Exception
    {
        if (maxDist < 0)
            throw new IllegalArgumentException("Maximum distance needs to be a positive value");
        OneToOneMatcher matcher = new OneToOneMatcher(trackSegmentList1, trackSegmentList2);
        ArrayList<TrackPair> pairs = new ArrayList<TrackPair>();
        pairs.addAll(matcher.pairTracks(maxDist, distType));
        return pairs;
    }

    @Override
    public void Close()
    {
        try
        {
            for (Sequence sequence : Icy.getMainInterface().getSequencesContaining(overlay))
            {
                sequence.removeOverlay(overlay);
            }
        }
        catch (NullPointerException e)
        {
            // no sequence.
        }
    }

    @Override
    public void Compute()
    {
        if (!isEnabled())
            return;
        try
        {
            trackPool.getDisplaySequence().addOverlay(overlay);
        }
        catch (NullPointerException e)
        {
        }
    }

    @Override
    public void displaySequenceChanged()
    {
        for (Sequence sequence : Icy.getMainInterface().getSequencesContaining(overlay))
        {
            sequence.removeOverlay(overlay);
        }

        Sequence displaySequence = trackPool.getDisplaySequence();
        if (displaySequence != null)
        {
            trackPool.getDisplaySequence().addOverlay(overlay);
        }
    }

    @Override
    public void sequenceChanged(SequenceEvent sequenceEvent)
    {
    }

    @Override
    public void sequenceClosed(Sequence sequence)
    {
        sequence.removeOverlay(overlay);
    }

    /**
     * Frame that asks the user to select two track groups in the list of existing TrackGroup
     * objects
     */
    class GroupSelectionFrame extends IcyFrame
    {
        final JComboBox<String> refBox;
        final JComboBox<String> candidateBox;
        final ArrayList<TrackGroup> tgList;

        public GroupSelectionFrame()
        {
            super("Group selection");
            tgList = new ArrayList<TrackGroup>();
            for (TrackGroup tg : trackPool.getTrackGroupList())
                if (!tg.getTrackSegmentList().isEmpty() && tg.getDescription() != null)
                    tgList.add(tg);
            String[] groupList = new String[tgList.size()];
            for (int i = 0; i < tgList.size(); i++)
                groupList[i] = tgList.get(i).getDescription();
            refBox = new JComboBox<String>(groupList);
            candidateBox = new JComboBox<String>(groupList);
            if (groupList.length > 0)
                refBox.setSelectedIndex(0);
            if (groupList.length > 1)
                candidateBox.setSelectedIndex(1);
            else if (groupList.length > 0)
                candidateBox.setSelectedIndex(0);
            JPanel mainPane = new JPanel();
            mainPane.setLayout(new GridBagLayout());
            GridBagConstraints c = new GridBagConstraints();
            c.fill = GridBagConstraints.HORIZONTAL;
            c.weightx = 1.0;
            c.weighty = 0;

            c.gridx = 0;
            c.gridy = 0;
            mainPane.add(new JLabel("Reference group"), c);

            c.gridx = 1;
            c.gridy = 0;
            mainPane.add(new JLabel("Candidate group"), c);

            c.gridx = 0;
            c.gridy = 1;
            mainPane.add(refBox, c);

            c.gridx = 1;
            c.gridy = 1;
            mainPane.add(candidateBox, c);

            JButton selectButton = new JButton("Select");
            selectButton.addActionListener(new ActionListener()
            {
                @Override
                public void actionPerformed(ActionEvent e)
                {
                    int refIdx = refBox.getSelectedIndex();
                    int candIdx = candidateBox.getSelectedIndex();
                    if (refIdx >= 0 && candIdx >= 0)
                    {
                        setTrackGroups(tgList.get(refIdx), tgList.get(candIdx));
                    }
                    close();
                }
            });
            c.gridx = 0;
            c.gridy = 2;
            c.gridwidth = 2;
            mainPane.add(selectButton, c);

            JButton cancelButton = new JButton("Cancel");
            cancelButton.addActionListener(new ActionListener()
            {
                @Override
                public void actionPerformed(ActionEvent arg0)
                {
                    close();
                }
            });
            c.gridx = 0;
            c.gridy = 3;
            c.gridwidth = 2;
            mainPane.add(cancelButton, c);

            this.setContentPane(mainPane);
            this.pack();
        }
    }
    
    private class PerformanceOverlay extends Overlay
    {

		public PerformanceOverlay(String name) {
			super(name);
		}
    	
		 @Override
		    public void paint(Graphics2D g, Sequence sequence, IcyCanvas canvas)
		    {
		    	if ( ! ( canvas instanceof IcyCanvas2D  ) )return;
		        boolean colorTracks = colorTracksBox.isSelected();
		        boolean displayPairs = displayPairsBox.isSelected();
		        boolean displayMissing = displayMissingBox.isSelected();
		        boolean displaySpurious = displaySpuriousBox.isSelected();

		        if (colorTracks)
		        {
		            for (TrackSegment ts : recoveredTracks)
		                for (Detection d : ts.getDetectionList())
		                {
		                    d.setColor(recoveredTracksColor);
		                    // d.setEnabled(d.isEnabled() && displayPairs);
		                    d.setEnabled(displayPairs);
		                }
		            for (TrackSegment ts : correctTracks)
		                for (Detection d : ts.getDetectionList())
		                {
		                    d.setColor(correctTracksColor);
		                    // d.setEnabled(d.isEnabled() && displayPairs);
		                    d.setEnabled(displayPairs);
		                }
		            for (TrackSegment ts : missedTracks)
		                for (Detection d : ts.getDetectionList())
		                {
		                    d.setColor(missedTracksColor);
		                    // d.setEnabled(d.isEnabled() && displayMissing);
		                    d.setEnabled(displayMissing);
		                }
		            for (TrackSegment ts : spuriousTracks)
		                for (Detection d : ts.getDetectionList())
		                {
		                    d.setColor(spuriousTracksColor);
		                    // d.setEnabled(d.isEnabled() && displaySpurious);
		                    d.setEnabled(displaySpurious);
		                }
		        }
		        else
		        {
		            for (TrackSegment ts : recoveredTracks)
		                ts.setAllDetectionEnabled(displayPairs);
		            // for (Detection d:ts.getDetectionList())
		            // d.setSelected(d.isSelected() && displayPairs);
		            for (TrackSegment ts : correctTracks)
		                ts.setAllDetectionEnabled(displayPairs);
		            // for (Detection d:ts.getDetectionList())
		            // d.setEnabled(d.isEnabled() && displayPairs);
		            for (TrackSegment ts : missedTracks)
		                ts.setAllDetectionEnabled(displayMissing);
		            // for (Detection d:ts.getDetectionList())
		            // d.setEnabled(d.isEnabled() && displayMissing);
		            for (TrackSegment ts : spuriousTracks)
		                ts.setAllDetectionEnabled(displaySpurious);
		            // for (Detection d:ts.getDetectionList())
		            // d.setEnabled(d.isEnabled() && displaySpurious);
		        }
		        if (displayPairs)
		        {
		            float linkWidth = (float)  canvas.canvasToImageLogDeltaX(1);
		            g.setStroke(new BasicStroke(linkWidth));
		            // emphasis all detections.
		            g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
		            g.setColor(Color.YELLOW);
		            ArrayList<TrackPair> tpList = new ArrayList<TrackPair>();
		            tpList.addAll(trackPairs);
		            for (TrackPair tp : tpList)
		            {
		                if (tp.candidateTrack != null)
		                {
		                    int firstT;
		                    int lastT;
		                    if (tp.firstMatchingTime >= 0)
		                        firstT = tp.firstMatchingTime;
		                    else
		                        firstT = Math.max(tp.referenceTrack.getFirstDetection().getT(), tp.candidateTrack
		                                .getFirstDetection().getT());
		                    if (tp.lastMatchingTime >= 0)
		                        lastT = tp.lastMatchingTime;
		                    else
		                        lastT = Math.min(tp.referenceTrack.getLastDetection().getT(), tp.candidateTrack
		                                .getLastDetection().getT());
		                    for (int t = firstT; t <= lastT; t++)
		                    {
		                        Detection d1 = tp.referenceTrack.getDetectionAtTime(t);
		                        Detection d2 = tp.candidateTrack.getDetectionAtTime(t);
		                        if (d1.isEnabled() && d2.isEnabled())
		                        {
		                            Line2D line = new Line2D.Double(d1.getX(), d1.getY(), d2.getX(), d2.getY());
		                            g.draw(line);
		                        }
		                    }
		                }
		            }
		        }
		    }
    }
}
