package plugins.vannary.morphomaths;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;

import icy.gui.dialog.MessageDialog;
import icy.gui.frame.IcyFrame;
import icy.gui.util.GuiUtil;
import icy.sequence.Sequence;
import icy.sequence.SequenceUtil;
import icy.type.DataType;
import plugins.adufour.ezplug.EzButton;
import plugins.adufour.ezplug.EzGroup;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzVar;
import plugins.adufour.ezplug.EzVarBoolean;
import plugins.adufour.ezplug.EzVarDouble;
import plugins.adufour.ezplug.EzVarEnum;
import plugins.adufour.ezplug.EzVarInteger;
import plugins.adufour.ezplug.EzVarListener;
import plugins.adufour.ezplug.EzVarSequence;
import plugins.vannary.morphomaths.MorphOp.UnhandledImageTypeException;

/**
 * 
 * @author Paul Compere
 * 
 *         Plugin containing a GUI to use the different morphology methods
 *         proposed by the class MorphOp.
 * 
 */
public class MorphoMaths extends EzPlug {

	// ----------------------------------------------------
	// EzPlug variables declaration
	// ----------------------------------------------------

	// Input sequence
	EzVarSequence in;
	// Method used
	EzVarEnum<MethodType> method;
	// Group for parameters of the skeleton method

	EzVarEnum<SkelType> skel;
	EzVarDouble threshSkel;
	EzVarBoolean aboveThreshSkel;
	EzVarInteger smoothOrder;
	EzGroup groupSkel;
	// Group for parameters of the water shed method
	EzVarEnum<WshedDisplay> wshedDisplay;
	EzGroup groupWshed;
	// Group for distance map
	EzGroup groupDMap;
	EzVarDouble thresh;
	EzVarBoolean aboveThresh;
	// Group for the general parameters

	EzVarBoolean inplace;
	EzVarBoolean option3D;
	EzVarInteger zValue;
	EzButton changeElts;
	EzGroup groupGenSettings;
	// Output sequence
	Sequence out;

	// Indicates if there is a result to show
	boolean showResult = true;

	// --------------------------------------------------
	// Graphic components declaration and initialization
	// --------------------------------------------------

	final static private int SCROLLPANE_MAX_SIZE = 310;
	/* Structuring elements informations */
	int xElt = 3, yElt = 3, zElt = 3;
	int centerX = 1, centerY = 1, centerZ = 1;
	// Default structuring elements
	double[][][] eltS3D = new double[][][] { { { 1.0, 1.0, 1.0 }, { 1.0, 1.0, 1.0 }, { 1.0, 1.0, 1.0 } },
			{ { 1.0, 1.0, 1.0 }, { 1.0, 1.0, 1.0 }, { 1.0, 1.0, 1.0 } },
			{ { 1.0, 1.0, 1.0 }, { 1.0, 1.0, 1.0 }, { 1.0, 1.0, 1.0 } } };
	double[][] eltS2D = new double[][] { { 1.0, 1.0, 1.0 }, { 1.0, 1.0, 1.0 }, { 1.0, 1.0, 1.0 } };

	/* Gui for setting a custom structuring element */
	ArrayList<JTextField> listeFieldsEltS = new ArrayList<JTextField>();
	IcyFrame frameEltS = new IcyFrame("Structuring element", false, true, false, true);
	// Panel describing the structuring elt
	JPanel panEltDesc = GuiUtil.generatePanel("Structuring element description");
	JPanel panValiderElt = new JPanel();
	// Panel for the input of the structuring elt
	JPanel panEltInput = GuiUtil.generatePanel("Structuring element input");
	JScrollPane scrollPane = new JScrollPane(panEltInput);
	JPanel mainPan = new JPanel();
	JTextField fieldMatrixX = new JTextField("3", 3), fieldMatrixY = new JTextField("3", 3),
			fieldMatrixZ = new JTextField("3", 3), fieldCx = new JTextField("1", 3), fieldCy = new JTextField("1", 3),
			fieldCz = new JTextField("1", 3);

	JButton validerTaille = new JButton("Validate caracteristics");
	JButton validerElt = new JButton("Validate Structuring element");

	@Override
	public void initialize() {

		// EzSchtroumpf schtroumpf = new EzSchtroumpf();

		// Init variables
		in = new EzVarSequence("Input:");
		addEzComponent(in);

		method = new EzVarEnum<MethodType>("Method:", MethodType.values());
		addEzComponent(method);

		wshedDisplay = new EzVarEnum<WshedDisplay>("Display type :", WshedDisplay.values());
		groupWshed = new EzGroup("Watershed parameters", wshedDisplay);
		addEzComponent(groupWshed);

		/*
		 * init the value of the zValue, threshold, and 3D boolean params according to
		 * the sequence selected at the opening of the plug in
		 */
		String zValueName = new String("Z stack to transform: ");
		String threshName = new String("Threshold :");
		boolean firstvalue3D = false;

		// modif by Alex: removed pointer implementation from EzPlug (for now)
		// if (!in.isPointerAssigned()){
		if (true) {
			if (in.getValue(false) != null) {
				if (in.getValue().getSizeZ() > 1)
					firstvalue3D = true;

				zValue = new EzVarInteger(zValueName, 0, 0, in.getValue().getSizeZ() - 1, 1);

				DataType type = in.getValue().getFirstImage().getDataType_();
				boolean signed = in.getValue().getColorModel().isSignedDataType();
				switch (type) {
				case BYTE:
				case UBYTE:
					thresh = new EzVarDouble(threshName, 127.0, 0.0, 255.0, 1.0);
					threshSkel = new EzVarDouble(threshName, 127.0, 0.0, 255.0, 1.0);
					break;
				case SHORT:
				case USHORT:
					thresh = new EzVarDouble(threshName, signed ? 0.0 : 32767.0, signed ? -32768.0 : 0.0,
							signed ? 32767.0 : 65535.0, 1.0);
					threshSkel = new EzVarDouble(threshName, signed ? 0.0 : 32767.0, signed ? -32768.0 : 0.0,
							signed ? 32767.0 : 65535.0, 1.0);
					break;

				case INT:
				case UINT:
					thresh = new EzVarDouble(threshName, signed ? 0.0 : 2147483647.0, signed ? -2147483648.0 : 0.0,
							signed ? 2147483647.0 : 4294967295.0, 1.0);
					threshSkel = new EzVarDouble(threshName, signed ? 0.0 : 2147483647.0, signed ? -2147483648.0 : 0.0,
							signed ? 2147483647.0 : 4294967295.0, 1.0);
					break;

				case FLOAT:
					thresh = new EzVarDouble(threshName, 0.5, 0.0, (double) Float.MAX_VALUE, 0.1);
					threshSkel = new EzVarDouble(threshName, 0.5, 0.0, (double) Float.MAX_VALUE, 0.1);
					break;
				case DOUBLE:
					thresh = new EzVarDouble(threshName, 0.5, 0.0, 100000, 0.1);
					threshSkel = new EzVarDouble(threshName, 0.5, 0.0, 100000, 0.1);
					break;
				default:
					throw new RuntimeException("Unsupported data type: " + type);
				}
			} else {
				// no sequence is opened, default values
				zValue = new EzVarInteger(zValueName, 0, 0, 0, 1);
				thresh = new EzVarDouble(threshName, 0.0, 0.0, 0.0, 0.1);
				threshSkel = new EzVarDouble(threshName, 0.0, 0.0, 0.0, 0.1);

			}
		}

		aboveThresh = new EzVarBoolean("Object above threshold", true);
		aboveThreshSkel = new EzVarBoolean("Object above threshold", true);
		inplace = new EzVarBoolean("Inplace image", false);

		option3D = new EzVarBoolean("3D Process", firstvalue3D);
		changeElts = new EzButton("Set custom structuring element", new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				initEltFrame();
			}
		});

		smoothOrder = new EzVarInteger("Smoothing order :", 3, 1, 10, 1);
		skel = new EzVarEnum<SkelType>("Skeleton type :", SkelType.values());
		groupSkel = new EzGroup("Skeleton method parameters", skel, smoothOrder, threshSkel, aboveThreshSkel);
		addEzComponent(groupSkel);

		groupDMap = new EzGroup("Distance map parameters", thresh, aboveThresh);
		addEzComponent(groupDMap);

		groupGenSettings = new EzGroup("General settings", inplace, option3D, zValue, changeElts);
		addEzComponent(groupGenSettings);

		in.addVarChangeListener(new EzVarListener<Sequence>() {
			@Override
			public void variableChanged(EzVar<Sequence> source, Sequence newValue) {
				updateDefaultParams();
			}

		});

		// Triggers
		method.addVisibilityTriggerTo(groupSkel, MethodType.SKEL);
		skel.addVisibilityTriggerTo(smoothOrder, SkelType.SKEL2);
		method.addVisibilityTriggerTo(groupWshed, MethodType.WSHED);
		method.addVisibilityTriggerTo(groupDMap, MethodType.DISTANCE_MAP);
		option3D.addVisibilityTriggerTo(zValue, false);

		super.setTimeDisplay(true);
	}

	// Update the default parameters when the input sequence changes
	private void updateDefaultParams() {

		// modif by Alex: removed pointer implementation from EzPlug (for now)
		// if (!in.isPointerAssigned()){
		if (true) {
			if (in.getValue(false) != null) {
				// Z value to transform changes according to the Z size of the image
				// modif by Alex: removed pointer implementation from EzPlug (for now)
				// if (!zValue.isPointerAssigned()){
				if (true) {
					int newZMax = in.getValue().getSizeZ() - 1;
					if (zValue.getValue() > newZMax)
						zValue.setValue(0);
					zValue.setMaxValue(newZMax);
				}
				// Set 3D boolean true when it is a 3D sequence
				if (in.getValue().getSizeZ() > 1) {
					option3D.setValue(true);
				} else {
					option3D.setValue(false);
				}
				// Max value of threshold changes according to the image type
				// modif by Alex: removed pointer implementation from EzPlug (for now)
				// if (!thresh.isPointerAssigned()){
				if (true) {
					DataType type = in.getValue().getFirstImage().getDataType_();
					// Prevent exception due to maxValue < minValue || maxValue < Value
					// Modified after a bug report on May 3rd 2012
					if (type.isFloat()) {
						thresh.setMaxValue(type.getMaxValue());
						thresh.setMinValue(0.0);
						thresh.setValue(0.5);
						thresh.setStep(0.1);
					} else {
						thresh.setMaxValue(type.getMaxValue());
						thresh.setMinValue(type.getMinValue());
						thresh.setValue((type.getMaxValue() + type.getMinValue()) / 2);
						thresh.setStep(1.0);

					}

					// modif by Alex: removed pointer implementation from EzPlug (for now)
					// if (!threshSkel.isPointerAssigned()){
					if (true) {
						threshSkel.setMaxValue(thresh.getMaxValue());
						threshSkel.setMinValue(thresh.getMinValue());
						threshSkel.setStep(thresh.getStep());
						threshSkel.setValue(thresh.getValue());
					}
				}
			}
			// No sequence
			else {
				// Reset Z value
				// modif by Alex: removed pointer implementation from EzPlug (for now)
				// if (!zValue.isPointerAssigned()){
				if (true) {
					zValue.setValue(0);
					zValue.setMaxValue(0);
				}
				// modif by Alex: removed pointer implementation from EzPlug (for now)
				// if (!thresh.isPointerAssigned()){
				if (true) {
					// Reset threshold value
					thresh.setValue(0.0);
					thresh.setMinValue(0.0);
					thresh.setMaxValue(0.0);
					threshSkel.setValue(0.0);
					threshSkel.setMinValue(0.0);
					threshSkel.setMaxValue(0.0);
				}
			}
		}
		// Pointer assigned
		// else {
		// thresh.setMaxValue(100000.0);
		// threshSkel.setMaxValue(100000.0);
		// zValue.setMaxValue(999);
		// }
	}

	@Override
	public void execute() {
		// Control that there is an input sequence
		if (in.getValue() == null) {
			showResult = false;
			return;
		}

		// Parameters from GUI
		boolean _inplace = inplace.getValue();
		Sequence seqToTransform;
		if (_inplace)
			seqToTransform = in.getValue();
		else
			seqToTransform = SequenceUtil.getCopy(in.getValue());
		boolean proc3D = option3D.getValue();
		int z = zValue.getValue();

		if (!proc3D && z >= seqToTransform.getSizeZ()) {
			System.err.println("Z is too high");
			showResult = false;
			return;
		}
		MethodType _method = method.getValue();
		double _thresh = thresh.getValue();
		boolean _aboveThresh = aboveThresh.getValue();


		try {
			switch (_method) {
			case ERODE_GREY:
				if (proc3D)
					MorphOp.erodeGreyScale3D(seqToTransform, eltS3D, centerX, centerY, centerZ);
				else
					MorphOp.erodeGreyScale(seqToTransform, z, eltS2D, centerX, centerY);
				seqToTransform.setName(in.getValue().getName() + "_Erosion");
				break;
			case DILATE_GREY:
				if (proc3D)
					MorphOp.dilateGreyScale3D(seqToTransform, eltS3D, centerX, centerY, centerZ);
				else
					MorphOp.dilateGreyScale(seqToTransform, z, eltS2D, centerX, centerY);
				seqToTransform.setName(in.getValue().getName() + "_Dilation");
				break;
			case OPEN:
				if (proc3D)
					MorphOp.openGreyScale3D(seqToTransform, eltS3D, centerX, centerY, centerZ);
				else
					MorphOp.openGreyScale(seqToTransform, z, eltS2D, centerX, centerY);
				seqToTransform.setName(in.getValue().getName() + "_Opening");

				break;
			case CLOSE:
				if (proc3D)
					MorphOp.closeGreyScale3D(seqToTransform, eltS3D, centerX, centerY, centerZ);
				else
					MorphOp.closeGreyScale(seqToTransform, z, eltS2D, centerX, centerY);
				seqToTransform.setName(in.getValue().getName() + "_Closing");

				break;
			case WTH:
				if (proc3D)
					MorphOp.whiteTopHat3D(seqToTransform, eltS3D, centerX, centerY, centerZ);
				else
					MorphOp.whiteTopHat(seqToTransform, z, eltS2D, centerX, centerY);
				seqToTransform.setName(in.getValue().getName() + "_WtopHat");
				break;
			case BTH:
				if (proc3D)
					MorphOp.blackTopHat3D(seqToTransform, eltS3D, centerX, centerY, centerZ);
				else
					MorphOp.blackTopHat(seqToTransform, z, eltS2D, centerX, centerY);
				seqToTransform.setName(in.getValue().getName() + "_BtopHat");
				break;
			case DISTANCE_MAP:
				if (proc3D)
					MorphOp.distanceMap3D(seqToTransform, _thresh, _aboveThresh);
				else
					MorphOp.distanceMap2D(seqToTransform, z, _thresh, _aboveThresh);
				seqToTransform.setName(in.getValue().getName() + "_Distance");
				seqToTransform.dataChanged();
				break;
			case DISTANCE_8:
				if (proc3D)
					MorphOp.distanceMap3D(seqToTransform, _thresh, _aboveThresh);
				else
					seqToTransform = MorphOp.getDistance(seqToTransform, 8);
				seqToTransform.setName(in.getValue().getName() + "_Distance_8");
				seqToTransform.dataChanged();
				break;
			case DISTANCE_:
				if (proc3D)
					MorphOp.distanceMap3D(seqToTransform, _thresh, _aboveThresh);
				else
					seqToTransform = MorphOp.getDistance(seqToTransform, 4);
				seqToTransform.setName(in.getValue().getName() + "_Distance_4");
				seqToTransform.dataChanged();
				break;
			case SKEL:
				SkelType _skel = skel.getValue();
				if (proc3D) {
					MorphOp.skeleton3D(seqToTransform, _thresh, _aboveThresh);
					// EzMessage.message("Method not implemented in 3D yet", MessageType.ERROR,
					// OutputType.DIALOG);
					// showResult = false;
					// out = null;
				} else {
					switch (_skel) {
					case SKEL1:
						MorphOp.skeleton(seqToTransform, z, _thresh, _aboveThresh, MorphOp.TYPESKEL_MAX, 0);
						break;
					case SKEL2:
						MorphOp.skeleton(seqToTransform, z, _thresh, _aboveThresh, MorphOp.TYPESKEL_SMOOTH,
								smoothOrder.getValue());
						break;
					}

				}
				seqToTransform.dataChanged();
				break;
			case SKELIMAGEJ:
				ImageJskeletton imageJSkel = new ImageJskeletton();
				imageJSkel.skeleton(seqToTransform);
				break;
			case WSHED:
				WshedDisplay _wshedDisplay = wshedDisplay.getValue();
				if (proc3D) {
					switch (_wshedDisplay) {
					case SEP_VERSANTS:
						MorphOp.watershed3D(seqToTransform, MorphOp.TYPEWSHED_SEP_BASINS);
						break;
					case CARTE_BASSINS_VERSANTS:
						MorphOp.watershed3D(seqToTransform, MorphOp.TYPEWSHED_BASINS_MAP);
						break;
					case WSHED_ONLY:
						MorphOp.watershed3D(seqToTransform, MorphOp.TYPEWSHED_WSHEDONLY);
						break;
					}
				} else {
					switch (_wshedDisplay) {
					case SEP_VERSANTS:
						MorphOp.watershed2D(seqToTransform, z, MorphOp.TYPEWSHED_SEP_BASINS);
						break;
					case CARTE_BASSINS_VERSANTS:
						MorphOp.watershed2D(seqToTransform, z, MorphOp.TYPEWSHED_BASINS_MAP);
						break;
					case WSHED_ONLY:
						MorphOp.watershed2D(seqToTransform, z, MorphOp.TYPEWSHED_WSHEDONLY);
						break;
					}
				}
				seqToTransform.dataChanged();
				break;
			case GRADIENT:
				if (proc3D)

					MorphOp.gradient3D(seqToTransform, eltS3D, centerX, centerY, centerZ);
				else {

					MorphOp.gradient(seqToTransform, z, eltS2D, centerX, centerY);
				}
				break;
			default:
				break;
			}

			out = seqToTransform;
		} catch (UnhandledImageTypeException e) {
			String message = "Unsupported image type";
			MessageDialog.showDialog("Error", message, MessageDialog.ERROR_MESSAGE);
		}

		if (!showResult || out == null) {
			out = null;
			showResult = true;
			return;
		}

		boolean _inplace1 = inplace.getValue();
		
		if (_inplace1) {
			// in.getValue().setName(in.getValue().getName() + "[MORPH]");
			in.getValue().dataChanged();
		} else {
			// out.setName(in.getValue().getName() + "[MORPH]");
			addSequence(out);
		}
	}

	// Initialization of the frame used to define the structuring element
	private void initEltFrame() {
		// Do nothing if the frame is already shown
		if (frameEltS.isVisible()) {
			return;
		}
		// Remove panels to prevent bug if it's not the first opening of the frame
		frameEltS.remove(panEltDesc);
		frameEltS.remove(panValiderElt);
		frameEltS.remove(scrollPane);
		panEltDesc.removeAll();
		GridBagConstraints c = new GridBagConstraints();
		c.fill = GridBagConstraints.HORIZONTAL;
		c.gridwidth = GridBagConstraints.REMAINDER;

		mainPan.setLayout(new GridBagLayout());

		panEltDesc.setLayout(new GridBagLayout());

		JPanel panSize = new JPanel();
		c.insets = new Insets(0, 5, 0, 5);
		panSize.add(new JLabel("Size on axis :"), c);
		panSize.add(new JLabel("X : "), c);
		panSize.add(fieldMatrixX, c);
		panSize.add(new JLabel("Y : "), c);
		panSize.add(fieldMatrixY, c);
		panSize.add(new JLabel("Z : "), c);
		panSize.add(fieldMatrixZ, c);

		JPanel panCenter = new JPanel();
		panCenter.add(new JLabel("Center position :"), c);
		panCenter.add(new JLabel("X : "), c);
		panCenter.add(fieldCx);
		panCenter.add(new JLabel("Y : "), c);
		panCenter.add(fieldCy);
		panCenter.add(new JLabel("Z : "), c);
		panCenter.add(fieldCz);

		c.gridy = 0;
		panEltDesc.add(panSize, c);
		c.gridy = 1;
		panEltDesc.add(panCenter, c);
		c.gridy = 2;
		panEltDesc.add(validerTaille, c);
		c.gridy = GridBagConstraints.RELATIVE;
		// show the matrices corresponding to the structuring element
		if (validerTaille.getActionListeners().length < 1)
			validerTaille.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent ev) {
					xElt = Integer.parseInt(fieldMatrixX.getText());
					yElt = Integer.parseInt(fieldMatrixY.getText());
					zElt = Integer.parseInt(fieldMatrixZ.getText());
					centerX = Integer.parseInt(fieldCx.getText());
					centerY = Integer.parseInt(fieldCy.getText());
					centerZ = Integer.parseInt(fieldCz.getText());

					if (xElt <= 0 || yElt <= 0 || zElt <= 0) {
						System.out.println("Invalid sutrcturing element size");
						return;
					}
					if (centerX < 0 || centerY < 0 || centerZ < 0 || centerX >= xElt || centerY >= yElt
							|| (zElt > 1 && centerZ >= zElt)) {
						System.out.println("Invalid center position");
						return;
					}
					panEltInput.removeAll();

					while (!listeFieldsEltS.isEmpty()) {
						listeFieldsEltS.remove(0);
					}
					scrollPane.setVisible(true);

					GridBagConstraints c = new GridBagConstraints();

					for (int z = 0; z < zElt; z++) {
						JPanel panelZ = GuiUtil.generatePanel("Z = " + z);
						panEltInput.add(GuiUtil.besidesPanel(panelZ));
						panelZ.setLayout(new GridBagLayout());
						for (int y = 0; y < yElt; y++) {
							for (int x = 0; x < xElt; x++) {
								c.gridx = x;
								c.gridy = y;
								c.fill = GridBagConstraints.BOTH;
								JTextField field = new JTextField("1.0", 0);
								// add the fields to an arraylist to can get the values back
								listeFieldsEltS.add(field);
								panelZ.add(field, c);
							}
						}
					}
					// Scrollpane size <= SCROLLPANE_MAX_SIZE
					scrollPane.setPreferredSize(null);
					scrollPane.setMaximumSize(new Dimension(scrollPane.getSize().width, SCROLLPANE_MAX_SIZE));
					validerElt.setVisible(true);
					frameEltS.pack();
					scrollPane.setPreferredSize(new Dimension(scrollPane.getSize().width,
							Math.min(scrollPane.getSize().height, SCROLLPANE_MAX_SIZE)));
					frameEltS.pack();
				}
			});

		scrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
		scrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		scrollPane.setVisible(false);

		panValiderElt.setLayout(new GridBagLayout());
		panValiderElt.add(validerElt);

		if (validerElt.getActionListeners().length < 1)
			validerElt.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {

					int offset = 0;
					if (zElt > 1) {
						eltS3D = new double[xElt][yElt][zElt];
					} else {
						eltS2D = new double[xElt][yElt];
					}
					for (int z = 0; z < zElt; z++) {
						for (int y = 0; y < yElt; y++) {
							for (int x = 0; x < xElt; x++, offset++) {
								if (zElt > 1) {
									eltS3D[x][y][z] = Double.parseDouble(listeFieldsEltS.get(offset).getText());
								} else {
									eltS2D[x][y] = Double.parseDouble(listeFieldsEltS.get(offset).getText());
								}
							}
						}
					}
					if (zElt > 1) {
						System.out.println("3D element updated");
					} else {
						System.out.println("2D element updated");
					}

					scrollPane.setVisible(false);
					validerElt.setVisible(false);
					frameEltS.pack();
				}
			});
		validerElt.setVisible(false);

		c.gridwidth = GridBagConstraints.REMAINDER;
		mainPan.add(panEltDesc, c);
		c.gridwidth = GridBagConstraints.REMAINDER;
		mainPan.add(scrollPane, c);
		c.gridwidth = GridBagConstraints.REMAINDER;
		mainPan.add(panValiderElt, c);

		frameEltS.add(mainPan);
		frameEltS.pack();

		frameEltS.setVisible(true);
		addIcyFrame(frameEltS);

	}

	public void clean() {
	}

	// Enum containing the methods that can be called in this plug in
	private enum MethodType {
		ERODE_GREY("Erosion"), DILATE_GREY("Dilation"), OPEN("Opening"), CLOSE("Closing"), WTH("White top-hat"),
		BTH("Black top-hat"), GRADIENT("Gradient"), DISTANCE_MAP("Distance map"), DISTANCE_8("Discrete Distance 8"),
		DISTANCE_("Discrete Distance 4"), SKEL("Skeleton"), SKELIMAGEJ("3D Medial Axis (Skeleton)"), WSHED("Watershed");

		private final String name;

		private MethodType(String string) {
			this.name = string;
		}

		public String toString() {
			return this.name;
		}
	}

	// Enum containing the different type of skeletons that can be calculated
	private enum SkelType {
		SKEL1("Maximum"), SKEL2("Smooth");

		private final String name;

		private SkelType(String string) {
			this.name = string;
		}

		public String toString() {
			return this.name;
		}
	}

	// Enum containing the different type of skeletons that can be calculated
	private enum WshedDisplay {
		SEP_VERSANTS("Separate basins"), CARTE_BASSINS_VERSANTS("Basins map"), WSHED_ONLY("Watershed only");

		private final String name;

		private WshedDisplay(String string) {
			this.name = string;
		}

		public String toString() {
			return this.name;
		}
	}

}