package plugins.ylemontag.ssim;

import icy.image.IcyBufferedImage;
import icy.sequence.Sequence;
import icy.sequence.SequenceBuilder;
import icy.system.SystemUtil;
import icy.system.thread.Processor;
import icy.system.thread.ThreadUtil;
import icy.type.DataType;
import icy.type.collection.array.Array1DUtil;

import java.util.LinkedList;

import plugins.adufour.filtering.Convolution;
import plugins.adufour.filtering.Convolution1D;
import plugins.adufour.filtering.ConvolutionException;
import plugins.adufour.filtering.Kernels1D;
import plugins.adufour.filtering.Kernels2D;
import plugins.adufour.vars.lang.VarBoolean;

/**
 * 
 * @author Yoann Le Montagner
 * 
 * Object used to compute the SSIM between two sequences (store the parameters).
 * 
 * The class SSIMCalculator is thread-safe, meaning that several computation
 * can be run in parallel from a given SSIMCalculator object shared between
 * several threads.
 * 
 * Usage:
 * {
 *   @code
 *   
 *   // Create a SSIM calculator object and set up the SSIM parameters
 *   SSIMCalculator myCalculator = SSIMCalculator.create();
 *   myCalculator.setL(255);
 *   myCalculator.setK1(0.02);
 *   
 *   // Compute an error map and a mean SSIM between 2 sequences 
 *   SSIMCalculator.Result result = myCalculator.compute(seq1, seq2);
 *   Sequence ssimMap = result.map;
 *   Sequence ssimMap12 = myCalculator.getSSIMMap();
 *   System.out.println("The mean SSIM between seq1 and seq2 is: " + result.mean);
 *   
 *   // Compute an error map and set the result in a pre-allocated sequence
 *   myCalculator.compute(seq3, seq4, ssimMap34);
 * }
 * 
 * The SSIM is an index measuring the structural similarity between two images.
 * It is valued between -1 and 1. When two images are nearly identical, their
 * SSIM is close to 1.
 * 
 * SSIM reference:
 *   Z. Wang, A. C. Bovik, H. R. Sheikh, E. P. Simoncelli (2004),
 *   Image quality assessment: from error visibility to structural similarity,
 *   IEEE Transactions on Image Processing, 13(4), 600-612.
 *   
 * This implementation sticks as much as possible to the Matlab SSIM
 * implementation provided by the authors at:
 *   https://ece.uwaterloo.ca/~z70wang/research/ssim/
 * 
 * Formula computing the SSIM between two sequences in1 and in2 at a given
 * pixel or voxel P:
 * 
 *             2*mu1(P)*mu2(P) + C1         2*cov(P) + C2
 * SSIM(P) = ------------------------ x ----------------------
 *           mu1(P)^2 + mu2(P)^2 + C1   s1(P)^2 + s2(P)^2 + C2
 * 
 * With:
 *   - mu1(P) and mu2(P): mean value of in1 and in2 computed over a small XY
 *     window located around P.
 *   - s1(P) and s2(P): standard deviation of in1 and in2 computed over the
 *     same window.
 *   - cov(P): covariance between in1 and in2 computed over the same window.
 *   - C1 = (K1*L)^2 (constant)
 *   - C2 = (K2*L)^2 (constant)
 *   - K1, K2: regularization parameters (must be >0). Default values:
 *       - K1 = 0.01
 *       - K2 = 0.03
 *   - L: dynamic range of the pixel values (example: L=255 if the sequence is
 *     8 bit encoded).
 *   - default window: Gaussian XY window with standard deviation = 1.5
 */
public class SSIMCalculator implements Cloneable
{
	private double    _L;
	private double    _K1;
	private double    _K2;
	private boolean   _useGaussianKernel;
	private double    _sigmaX;
	private double    _sigmaY;
	private Kernels2D _window;
	private Object    _mutex;
	
	/**
	 * Default L value
	 */
	public final static double DEFAULT_L = 255;
	
	/**
	 * Default K1 value
	 */
	public final static double DEFAULT_K1 = 0.01;
	
	/**
	 * Default K2 value
	 */
	public final static double DEFAULT_K2 = 0.03;
	
	/**
	 * Default standard deviation of the Gaussian kernel 
	 */
	public final static double DEFAULT_GAUSSIAN_SIGMA = 1.5;
	
	@Override 
	public SSIMCalculator clone()
	{
		SSIMCalculator retVal = new SSIMCalculator();
		retVal.cloneParameters(this);
		return retVal;
	}
	
	/**
	 * Factory method (default parameters)
	 */
	public static SSIMCalculator create()
	{
		return new SSIMCalculator();
	}
	
	/**
	 * Factory method with custom dynamic range parameter (L)
	 * @throws IllegalSSIMParameterException if l<=0
	 */
	public static SSIMCalculator create(double l)
	{
		SSIMCalculator retVal = new SSIMCalculator();
		retVal.setL(l);
		return retVal;
	}
	
	/**
	 * Constructor
	 */
	public SSIMCalculator()
	{
		_mutex = new Object();
		setL (DEFAULT_L );
		setK1(DEFAULT_K1);
		setK2(DEFAULT_K2);
		setGaussianWindow(DEFAULT_GAUSSIAN_SIGMA);
	}
	
	/**
	 * Dynamic range of the pixel values
	 */
	public double getL()
	{
		return _L;
	}
	
	/**
	 * K1 regularization parameter
	 */
	public double getK1()
	{
		return _K1;
	}
	
	/**
	 * K2 regularization parameter
	 */
	public double getK2()
	{
		return _K2;
	}
	
	/**
	 * Check whether a Gaussian profile is used for the window
	 */
	public boolean getUseGaussianWindow()
	{
		return _useGaussianKernel;
	}
	
	/**
	 * Return the standard deviation in the X direction if Gaussian profile is
	 * selected; otherwise, return -1
	 */
	public double getSigmaX()
	{
		synchronized (_mutex) {
			return _useGaussianKernel ? _sigmaX : -1;
		}
	}
	
	/**
	 * Return the standard deviation in the X direction if Gaussian profile is
	 * selected; otherwise, return -1
	 */
	public double getSigmaY()
	{
		synchronized (_mutex) {
			return _useGaussianKernel ? _sigmaY : -1;
		}
	}
	
	/**
	 * 2D-kernel used as a window
	 */
	public Kernels2D getWindow()
	{
		return _window;
	}
	
	/**
	 * Set the dynamic range
	 * @throws IllegalSSIMParameterException if l<=0
	 */
	public void setL(double l)
	{
		if(l<=0) {
			throw new IllegalSSIMParameterException("L must be > 0.");
		}
		synchronized (_mutex) {
			_L = l;
		}
	}
	
	/**
	 * Set the K1 regularization parameter
	 * @throws IllegalSSIMParameterException if k1<=0
	 */
	public void setK1(double k1)
	{
		if(k1<=0) {
			throw new IllegalSSIMParameterException("K1 must be > 0.");
		}
		synchronized (_mutex) {
			_K1 = k1;
		}
	}
	
	/**
	 * Set the K2 regularization parameter
	 * @throws IllegalSSIMParameterException if k2<=0
	 */
	public void setK2(double k2)
	{
		if(k2<=0) {
			throw new IllegalSSIMParameterException("K2 must be > 0.");
		}
		synchronized (_mutex) {
			_K2 = k2;
		}
	}
	
	/**
	 * Select a Gaussian profile for the window
	 * @param sigma Standard deviation (same value for both X and Y)
	 * @throws IllegalSSIMParameterException if sigma<0
	 */
	public void setGaussianWindow(double sigma)
	{
		setGaussianWindow(sigma, sigma);
	}
	
	/**
	 * Select a Gaussian profile for the window
	 * @param sigmaX Standard deviation along the X axis
	 * @param sigmaY Standard deviation along the Y axis
	 * @throws IllegalSSIMParameterException if sigmaX<0 or sigmaY<0
	 */
	public void setGaussianWindow(double sigmaX, double sigmaY)
	{
		if(sigmaX<0 || sigmaY<0) {
			throw new IllegalSSIMParameterException(
				"The standard deviation of the Gaussian window must be >= 0."
			);
		}
		synchronized (_mutex) {
			_useGaussianKernel = true;
			_sigmaX = sigmaX;
			_sigmaY = sigmaY;
			double[] rawDataX = createGaussianKernel(sigmaX);
			double[] rawDataY = createGaussianKernel(sigmaY);
			double[] rawWindow = new double[rawDataX.length * rawDataY.length];
			int offset = 0;
			for(int y=0; y<rawDataY.length; ++y) {
				for(int x=0; x<rawDataX.length; ++x) {
					rawWindow[offset] = rawDataX[x]*rawDataY[y];
					++offset;
				}
			}
			_window = Kernels2D.CUSTOM.createCustomKernel2D(rawWindow,
				rawDataX.length, rawDataY.length, true);
		}
	}
	
	/**
	 * Set a custom kernel for the window
	 * @throws IllegalSSIMParameterException if one of the coefficients of the
	 *         window is < 0 or if their sum is != 1.0
	 */
	public void setWindow(Kernels2D window)
	{
		double[] rawData = window.getData();
		double accu = 0;
		for(double v : rawData) {
			if(v<0) {
				throw new IllegalSSIMParameterException("The window coefficents must be >= 0."); 
			}
			accu += v;
		}
		if(accu!=1.0) {
			throw new IllegalSSIMParameterException(
				"The window coefficents must be normalized to have a sum equal to 1."
			);
		}
		synchronized (_mutex) {
			_useGaussianKernel = false;
			_window = window; 
		}
	}
	
	/**
	 * Copy the parameters (L, K1, etc...) of 'model' to the current SSIMCalculator object
	 */
	public void cloneParameters(SSIMCalculator model)
	{
		if(this==model) {
			return;
		}
		double L;
		double K1;
		double K2;
		boolean useGaussianKernel;
		double sigmaX = 0.0;
		double sigmaY = 0.0;
		Kernels2D window = null;
		synchronized (model._mutex) {
			L  = model.getL ();
			K1 = model.getK1();
			K2 = model.getK2();
			if(model.getUseGaussianWindow()) {
				useGaussianKernel = true;
				sigmaX = model.getSigmaX();
				sigmaY = model.getSigmaY();
			}
			else {
				useGaussianKernel = false;
				window = model.getWindow();
			}
		}
		synchronized (_mutex) {
			setL (L );
			setK1(K1);
			setK2(K2);
			if(useGaussianKernel)
				setGaussianWindow(sigmaX, sigmaY);
			else
				setWindow(window);
		}
	}
	
	/**
	 * Pair SSIM map + mean SSIM returned by the compute functions
	 */
	public static class Result
	{
		double mean;
		Sequence map;
	}
	
	/**
	 * Pair SSIM map + mean SSIM returned by the compute functions operating on double arrays
	 */
	public static class RawResult
	{
		double mean;
		double[] map;
	}
	
	/**
	 * Check is both sequence have the same size
	 */
	public static boolean haveSameSize(Sequence in1, Sequence in2)
	{
		return
			in1.getSizeX()==in2.getSizeX() &&
			in1.getSizeY()==in2.getSizeY() &&
			in1.getSizeZ()==in2.getSizeZ() &&
			in1.getSizeT()==in2.getSizeT() &&
			in1.getSizeC()==in2.getSizeC();
	}
	
	/**
	 * Compute both the SSIM map and the mean SSIM between two sequences
	 * @throws IllegalArgumentException if both sequences do not have the same size
	 */
	public Result compute(Sequence in1, Sequence in2)
	{
		try {
			return compute(in1, in2, new Controller());
		}
		catch(Controller.CanceledByUser err) {
			throw new RuntimeException("Unreachable code point");
		}
	}
	
	/**
	 * Compute both the SSIM map and the mean SSIM between two 2D images (represented as double arrays)
	 * @throws IllegalArgumentException if one of the arrays has a length different from sizeX*sizeY
	 */
	public RawResult compute(int sizeX, int sizeY, double[] in1, double[] in2)
	{
		try {
			return compute(sizeX, sizeY, in1, in2, new Controller());
		}
		catch(Controller.CanceledByUser err) {
			throw new RuntimeException("Unreachable code point");
		}
	}
	
	/**
	 * Compute the mean SSIM between two sequences
	 * @throws IllegalArgumentException if both sequences do not have the same size
	 */
	public double computeMean(Sequence in1, Sequence in2)
	{
		try {
			return computeMean(in1, in2, new Controller());
		}
		catch(Controller.CanceledByUser err) {
			throw new RuntimeException("Unreachable code point");
		}
	}
	
	/**
	 * Compute the mean SSIM between two 2D images (represented as double arrays)
	 * @throws IllegalArgumentException if one of the arrays has a length different from sizeX*sizeY
	 */
	public double computeMean(int sizeX, int sizeY, double[] in1, double[] in2)
	{
		try {
			return computeMean(sizeX, sizeY, in1, in2, new Controller());
		}
		catch(Controller.CanceledByUser err) {
			throw new RuntimeException("Unreachable code point");
		}
	}
	
	/**
	 * Compute the SSIM map between two sequences
	 * @throws IllegalArgumentException if both sequences do not have the same size
	 */
	public Sequence computeMap(Sequence in1, Sequence in2)
	{
		try {
			return computeMap(in1, in2, new Controller());
		}
		catch(Controller.CanceledByUser err) {
			throw new RuntimeException("Unreachable code point");
		}
	}
	
	/**
	 * Compute the SSIM map between two 2D images (represented as double arrays)
	 * @throws IllegalArgumentException if one of the arrays has a length different from sizeX*sizeY
	 */
	public double[] computeMap(int sizeX, int sizeY, double[] in1, double[] in2)
	{
		try {
			return computeMap(sizeX, sizeY, in1, in2, new Controller());
		}
		catch(Controller.CanceledByUser err) {
			throw new RuntimeException("Unreachable code point");
		}
	}
	
	/**
	 * Compute the SSIM map between two sequences, and set the result in sequence 'out'
	 * 
	 * If 'out' is null, a new Sequence is allocated.
	 * 
	 * @throws IllegalArgumentException if all the sequences do not have the same size
	 */
	public Sequence computeMap(Sequence in1, Sequence in2, Sequence out)
	{
		try {
			return computeMap(in1, in2, out, new Controller());
		}
		catch(Controller.CanceledByUser err) {
			throw new RuntimeException("Unreachable code point");
		}
	}
	
	/**
	 * Compute the SSIM map between two 2D images (represented as double arrays),
	 * and set the result in the double array 'out'
	 * 
	 * If 'out' is null, a new array is allocated.
	 * 
	 * @throws IllegalArgumentException if one of the arrays has a length different from sizeX*sizeY
	 */
	public double[] computeMap(int sizeX, int sizeY, double[] in1, double[] in2, double[] out)
	{	
		try {
			return computeMap(sizeX, sizeY, in1, in2, out, new Controller());
		}
		catch(Controller.CanceledByUser err) {
			throw new RuntimeException("Unreachable code point");
		}
	}
	
	/**
	 * Compute both the SSIM map and the mean SSIM between two sequences
	 * @throws IllegalArgumentException if all the sequences do not have the same size
	 * @throws Controller.CanceledByUser if the operation is interrupted
	 *         through the method 'cancelComputation' of the controller object
	 */
	public Result compute(Sequence in1, Sequence in2, Controller controller)
		throws Controller.CanceledByUser
	{
		Result retVal = new Result();
		retVal.map    = computeMap(in1, in2, controller);
		retVal.mean   = aggregateMap(retVal.map); ///TODO: allocation of raw-results
		return retVal;
	}
	
	/**
	 * Compute both the SSIM map and the mean SSIM between two 2D images (given as double arrays)
	 * @throws IllegalArgumentException if one of the arrays has a length different from sizeX*sizeY
	 * @throws Controller.CanceledByUser if the operation is interrupted
	 *         through the method 'cancelComputation' of the controller object
	 */
	public RawResult compute(int sizeX, int sizeY, double[] in1, double[] in2, Controller controller)
		throws Controller.CanceledByUser
	{
		RawResult retVal = new RawResult();
		retVal.map       = computeMap(sizeX, sizeY, in1, in2, controller);
		retVal.mean      = aggregateMap(retVal.map);
		return retVal;
	}
	
	/**
	 * Compute the mean SSIM between two sequences
	 * @throws IllegalArgumentException if all the sequences do not have the same size
	 * @throws Controller.CanceledByUser if the operation is interrupted
	 *         through the method 'cancelComputation' of the controller object
	 */
	public double computeMean(Sequence in1, Sequence in2, Controller controller)
		throws Controller.CanceledByUser
	{
		Sequence ssimMap = computeMap(in1, in2, controller);
		return aggregateMap(ssimMap);
	}
	
	/**
	 * Compute the mean SSIM between two 2D images (given as double arrays)
	 * @throws IllegalArgumentException if one of the arrays has a length different from sizeX*sizeY
	 * @throws Controller.CanceledByUser if the operation is interrupted
	 *         through the method 'cancelComputation' of the controller object
	 */
	public double computeMean(int sizeX, int sizeY, double[] in1, double[] in2, Controller controller)
		throws Controller.CanceledByUser
	{
		double[] ssimMap = computeMap(sizeX, sizeY, in1, in2, controller);
		return aggregateMap(ssimMap);
	}
	
	/**
	 * Compute the SSIM map between two sequences
	 * @throws IllegalArgumentException if all the sequences do not have the same size
	 * @throws Controller.CanceledByUser if the operation is interrupted
	 *         through the method 'cancelComputation' of the controller object
	 */
	public Sequence computeMap(Sequence in1, Sequence in2, Controller controller)
		throws Controller.CanceledByUser
	{
		return computeMap(in1, in2, null, controller);
	}
	
	/**
	 * Compute the SSIM map between two 2D images (represented as double arrays)
	 * @throws IllegalArgumentException if one of the arrays has a length different from sizeX*sizeY
	 * @throws Controller.CanceledByUser if the operation is interrupted
	 *         through the method 'cancelComputation' of the controller object
	 */
	public double[] computeMap(int sizeX, int sizeY, double[] in1, double[] in2, Controller controller)
		throws Controller.CanceledByUser
	{	
		return computeMap(sizeX, sizeY, in1, in2, null, controller);
	}
	
	/**
	 * Compute the SSIM map between two sequences, and set the result in sequence 'out'
	 * @throws IllegalArgumentException if all the sequences do not have the same size
	 * @throws Controller.CanceledByUser if the operation is interrupted
	 *         through the method 'cancelComputation' of the controller object
	 */
	public Sequence computeMap(Sequence in1, Sequence in2, Sequence out, Controller controller)
		throws Controller.CanceledByUser
	{	
		// Check the inputs
		if(in1==null || in2==null) {
			throw new IllegalArgumentException("The input sequences cannot be null.");
		}
		if(!haveSameSize(in1, in2)) {
			throw new IllegalArgumentException(
				"All the sequences involved in a SSIM calculation must have the same size."
			);
		}
		int sizeX = in1.getSizeX();
		int sizeY = in1.getSizeY();
		int sizeZ = in1.getSizeZ();
		int sizeT = in1.getSizeT();
		int sizeC = in1.getSizeC();
		
		// Allocate the output if necessary
		if(out==null) {
			out = new Sequence();
			out.setName("SSIM(" + in1.getName() + "," + in2.getName() + ")");
		}
		SequenceBuilder builder = new SequenceBuilder(sizeX, sizeY, sizeZ, sizeT, sizeC, DataType.DOUBLE, out);
		if(!builder.isPreAllocated() && (out==in1 || out==in2)) {
			throw new IllegalArgumentException(
				"Cannot use a non double-valued sequence as input and output at the same time in a SSIM calculation."
			);
		}
		
		// Allocate the jobs
		JobCommons      jc   = new JobCommons(sizeX, sizeY, controller);
		LinkedList<Job> jobs = new LinkedList<Job>();
		for(int t=0; t<sizeT; ++t) {
			for(int z=0; z<sizeZ; ++z) {
				for(int c=0; c<sizeC; ++c) {
					jobs.add(new Job(t, z, c, in1, in2, builder, jc));
				}
			}
		}
		
		// Run the jobs
		builder.beginUpdate();
		try {
			if(jobs.size()==0) {
				// Nothing to do
			}
			else if(jobs.size()==1) {
				jobs.getFirst().doTheJob();
			}
			else {
				Processor processor = new Processor(jobs.size(), SystemUtil.getAvailableProcessors());
				for(Job job : jobs) {
					processor.submit(job);
				}
				ThreadUtil.sleep(50);
				processor.waitAll();
				controller.checkPoint();
			}
		}
		finally {
			builder.endUpdate();
		}
		
		// Return the result
		return out;
	}
	
	/**
	 * Compute the SSIM map between two 2D images (represented as double arrays),
	 * and set the result in the double array 'out'
	 * @throws IllegalArgumentException if one of the arrays has a length different from sizeX*sizeY
	 * @throws Controller.CanceledByUser if the operation is interrupted
	 *         through the method 'cancelComputation' of the controller object
	 */
	public double[] computeMap(int sizeX, int sizeY, double[] in1, double[] in2, double[] out, Controller controller)
		throws Controller.CanceledByUser
	{
		// Initialization
		if(in1==null || in2==null) {
			throw new IllegalArgumentException("The input arrays cannot be null");
		}
		int sizeXY = sizeX * sizeY;
		if(out==null) {
			out = new double[sizeXY];
		}
		if(out.length!=sizeXY || in1.length!=sizeXY || in2.length!=sizeXY) {
			throw new IllegalArgumentException(
				"One of the array involved in the SSIM calculation has an inconsistent size (expected length: " + sizeXY + ")."
			);
		}
		
		// Computation
		JobCommons jc = new JobCommons(sizeX, sizeY, controller);
		computeSSIM(in1, in2, out, jc);
		return out;
	}
	
	/**
	 * Parameters common to all jobs
	 */
	private class JobCommons
	{
		public Controller _controller;
		public int        _sizeXY;
		public int        _sizeX;
		public int        _sizeY;
		public double     _C1;
		public double     _C2;
		public Sequence   _kernelWindow;
		public double[]   _kernelGaussianX;
		public double[]   _kernelGaussianY;
		
		// Constructor
		public JobCommons(int sizeX, int sizeY, Controller controller)
		{
			_controller = controller;
			_sizeX      = sizeX;
			_sizeY      = sizeY;
			_sizeXY     = sizeX*sizeY;
			synchronized (_mutex) {
				_C1 = _L*_L*_K1*_K1;
				_C2 = _L*_L*_K2*_K2;
				if(_useGaussianKernel) {
					_kernelGaussianX = createGaussianKernel(_sigmaX);
					_kernelGaussianY = createGaussianKernel(_sigmaY);
				}
				else {
					_kernelWindow = _window.toSequence();
				}
			}
		}
	}
	
	/**
	 * Core function
	 */
	private static class Job implements Runnable
	{
		private int             _t  ;
		private int             _z  ;
		private int             _c  ;
		private Sequence        _in1;
		private Sequence        _in2;
		private SequenceBuilder _out;
		private JobCommons      _jc ;
		
		// Each job computes the SSIM map over a XY plane
		public Job(int t, int z, int c, Sequence in1, Sequence in2, SequenceBuilder out, JobCommons jc)
		{
			_t   = t  ;
			_z   = z  ;
			_c   = c  ;
			_in1 = in1;
			_in2 = in2;
			_out = out;
			_jc  = jc ;
		}
		
		// Core routine, that does not intercept SSIMThreadController.CanceledByUser
		public void doTheJob() throws Controller.CanceledByUser
		{
			_jc._controller.checkPoint();
			double[] out = _out.getDataAsDouble(_t, _z, _c);
			double[] in1 = getDataInAsDouble(_in1);
			double[] in2 = getDataInAsDouble(_in2);
			computeSSIM(in1, in2, out, _jc);
			_jc._controller.checkPoint();
			_out.validateData(_t, _z, _c);
		}
		
		// Conversion routine for input data
		private double[] getDataInAsDouble(Sequence in)
		{
			DataType datatype = in.getDataType_();
			if(datatype==DataType.DOUBLE) {
				return in.getDataXYAsDouble(_t, _z, _c);
			}
			else {
				return Array1DUtil.arrayToDoubleArray(in.getDataXY(_t, _z, _c), datatype.isSigned());
			}
		}
		
		@Override
		public void run()
		{
			try {
				doTheJob();
			}
			catch(Controller.CanceledByUser err) {}
		}
	}
	
	/**
	 * Compute the SSIM between two arrays, which are supposed to contain the
	 * samples corresponding to one XY plane
	 */
	private static void computeSSIM(double[] in1, double[] in2, double[] out, JobCommons jc)
		throws Controller.CanceledByUser
	{
		// Mean, variance and covariance
		jc._controller.checkPoint();
		double[] mu1   = computeConvolution(in1, jc);
		jc._controller.checkPoint();
		double[] mu2   = computeConvolution(in2, jc);
		jc._controller.checkPoint();
		double[] v1NC  = computeConvolution(computeProduct(in1, in1, jc), jc);
		jc._controller.checkPoint();
		double[] v2NC  = computeConvolution(computeProduct(in2, in2, jc), jc);
		jc._controller.checkPoint();
		double[] covNC = computeConvolution(computeProduct(in1, in2, jc), jc);
		jc._controller.checkPoint();
		
		// Final ratio
		int sizeXY = jc._sizeXY;
		for(int xy=0; xy<sizeXY; ++xy) {
			double currMu1 = mu1[xy];
			double currMu2 = mu2[xy];
			double mu1_sq  = currMu1*currMu1;
			double mu2_sq  = currMu2*currMu2;
			double mu1_mu2 = currMu1*currMu2;
			double v1  = v1NC [xy] - mu1_sq ;
			double v2  = v2NC [xy] - mu2_sq ;
			double cov = covNC[xy] - mu1_mu2;
			double numerator   = (2*mu1_mu2 + jc._C1) * (2*cov + jc._C2);
			double denominator = (mu1_sq + mu2_sq + jc._C1) * (v1 + v2 + jc._C2);
			out[xy] = numerator / denominator;
		}
	}
	
	/**
	 * Compute the product pointwise between two arrays
	 */
	private static double[] computeProduct(double[] in1, double[] in2, JobCommons jc)
	{
		int sizeXY = jc._sizeXY;
		double[] retVal = new double[sizeXY];
		for(int xy=0; xy<sizeXY; ++xy) {
			retVal[xy] = in1[xy]*in2[xy];
		}
		return retVal;
	}
	
	/**
	 * Compute the convolution against the selected kernel
	 */
	private static double[] computeConvolution(double[] in, JobCommons jc)
	{
		int sizeXY = jc._sizeXY;
		double[][] buffer = new double[1][sizeXY];
		for(int xy=0; xy<sizeXY; ++xy) {
			buffer[0][xy] = in[xy];
		}
		Sequence seq = new Sequence(new IcyBufferedImage(jc._sizeX, jc._sizeY, buffer));
		if(jc._kernelWindow==null) {
			try {
				Convolution1D.convolve(seq, jc._kernelGaussianX, jc._kernelGaussianY, null);
			}
			catch (ConvolutionException e) {
				throw new RuntimeException(e);
			}
		}
		else {
			Convolution.convolve(seq, jc._kernelWindow, true, 1, new VarBoolean("stop", false));
		}
		return seq.getDataXYAsDouble(0, 0, 0);
	}
	
	/**
	 * Compute a mean SSIM value over the whole map
	 */
	private static double aggregateMap(Sequence ssimMap)
	{
		int sizeXY = ssimMap.getSizeX() * ssimMap.getSizeY();
		int sizeZ = ssimMap.getSizeZ();
		int sizeT = ssimMap.getSizeT();
		int sizeC = ssimMap.getSizeC();
		double accu = 0;
		for(int t=0; t<sizeT; ++t) {
			for(int z=0; z<sizeZ; ++z) {
				for(int c=0; c<sizeC; ++c) {
					double[] values = ssimMap.getDataXYAsDouble(t, z, c);
					for(int xy=0; xy<sizeXY; ++xy) {
						accu += values[xy];
					}
				}
			}
		}
		return accu / (double)(sizeXY * sizeZ * sizeT * sizeC);
	}
	
	/**
	 * Compute a mean SSIM value over the whole map (given as a double array)
	 */
	private static double aggregateMap(double[] ssimMap)
	{
		int sizeXY = ssimMap.length;
		double accu = 0;
		for(int xy=0; xy<sizeXY; ++xy) {
			accu += ssimMap[xy];
		}
		return accu / (double)(sizeXY);
	}
	
	/**
	 * Return the coefficient of a 1D Gaussian kernel with given standard deviation
	 */
	private static double[] createGaussianKernel(double sigma)
	{
		return Kernels1D.CUSTOM_GAUSSIAN.createGaussianKernel1D(sigma).getData();
	}
}
