package plugins.tprovoost.painting;

import icy.gui.component.button.ColorChooserButton;
import icy.gui.component.button.ColorChooserButton.ColorChangeListener;
import icy.gui.component.button.IcyButton;
import icy.gui.frame.IcyFrame;
import icy.gui.main.ActiveSequenceListener;
import icy.image.ImageUtil;
import icy.main.Icy;
import icy.painter.Overlay;
import icy.plugin.PluginLoader;
import icy.resource.ResourceUtil;
import icy.resource.icon.IcyIcon;
import icy.sequence.Sequence;
import icy.sequence.SequenceEvent;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.Image;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.border.Border;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

public class PaintingTools extends IcyFrame
{
    public enum PaintingTool
    {
        NONE, SELECT, PEN, RECTANGLE, PLAIN_RECTANGLE, OVAL, PLAIN_OVAL, LINE, CURVE, ARROW
    };

    private static final int ICON_SIZE = 24;
    public static PaintingTools instance = null;

    PaintingTool currentTool;

    // pre-loaded images icons
    private Image imageOvalFill;
    private Image imageRectFill;
    private Image imageInvertArrows;
    private Image imageCurve;
    private Image imageDrawOnPainter;
    private Image imageDrawOnImage;

    // different tools
    IcyButton btnDrawOnPainter;
    IcyButton btnDrawOnImage;
    IcyButton btnPen;
    IcyButton btnRect;
    IcyButton btnOval;
    IcyButton btnRectFill;
    IcyButton btnOvalFill;
    IcyButton btnLine;
    IcyButton btnCurve;
    IcyButton btnArrow;
    JButton btnInvertColors;
    JButton btnSelected;

    // other GUI options
    JSlider sliderThickness;
    ColorChooserButton btnColorFg;
    ColorChooserButton btnColorBg;
    CheckBoxPanel panelModifyData;
    JComboBox comboChannelAffected;

    // TODO
    private Color previousFg;
    private Color previousBg;

    protected Sequence currentSequence;
    protected final ActiveSequenceListener activeSequenceListener;

    PaintingTools()
    {
        super("Tools", false, true, false, true);

        currentTool = PaintingTool.PEN;
        previousFg = Color.CYAN;
        previousBg = Color.YELLOW;

        initialize();

        activeSequenceListener = new ActiveSequenceListener()
        {
            @Override
            public void sequenceActivated(Sequence sequence)
            {
                setCurrentSequence(sequence);
            }

            @Override
            public void sequenceDeactivated(Sequence sequence)
            {
                // nothing here
            }

            @Override
            public void activeSequenceChanged(SequenceEvent event)
            {
                // nothing here
            }
        };

        setCurrentSequence(Icy.getMainInterface().getActiveSequence());

        Icy.getMainInterface().addActiveSequenceListener(activeSequenceListener);
    }

    private void initialize()
    {
        // get all images for icons
        imageOvalFill = ImageUtil.load(PluginLoader
                .getResourceAsStream("plugins/tprovoost/painting/icons/roi_oval_fill.png"));
        imageRectFill = ImageUtil.load(PluginLoader
                .getResourceAsStream("plugins/tprovoost/painting/icons/roi_rectangl_filled.png"));
        imageInvertArrows = ImageUtil.load(PluginLoader
                .getResourceAsStream("plugins/tprovoost/painting/icons/invert_arrows.png"));
        imageCurve = ImageUtil.load(PluginLoader.getResourceAsStream("plugins/tprovoost/painting/icons/roi_curve.png"));
        imageDrawOnPainter = ImageUtil.load(PluginLoader
                .getResourceAsStream("plugins/tprovoost/painting/icons/painting_drawOnPainter.png"));
        imageDrawOnImage = ImageUtil.load(PluginLoader
                .getResourceAsStream("plugins/tprovoost/painting/icons/painting_drawOnImage.png"));

        final JPanel mainPanel = new JPanel();
        mainPanel.setBorder(new EmptyBorder(4, 4, 4, 4));
        mainPanel.setLayout(new BoxLayout(mainPanel, BoxLayout.Y_AXIS));

        // create buttons
        btnDrawOnPainter = new IcyButton(new IcyIcon(imageDrawOnPainter));
        btnDrawOnPainter.setToolTipText("Draw on the drawing painter. Will not alter data, undo is possible.");
        btnDrawOnPainter.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent e)
            {
                btnDrawOnImage.setSelected(false);
                btnDrawOnPainter.setSelected(true);

                // panel affect only
                if (!panelModifyData.isShowing())
                    return;
                mainPanel.remove(panelModifyData);
                updateChannels();
                mainPanel.validate();
                setSize(getWidth(), getHeight() - panelModifyData.getHeight());
            }
        });

        btnDrawOnImage = new IcyButton(new IcyIcon(imageDrawOnImage));
        btnDrawOnImage.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent e)
            {
                btnDrawOnImage.setSelected(true);
                btnDrawOnPainter.setSelected(false);

                boolean toAdd = !panelModifyData.isShowing() && Icy.getMainInterface().getActiveSequence() != null;

                // panel affect only
                if (toAdd)
                    mainPanel.add(panelModifyData, 2);
                updateChannels();
                mainPanel.validate();
                if (toAdd)
                    setSize(getWidth(), getHeight() + panelModifyData.getHeight());
            }
        });
        btnDrawOnImage
                .setToolTipText("Draw on the image directly. Be careful, data will be altered permanently (no undo)");

        // always draw on painter by default
        btnDrawOnPainter.setSelected(true);
        btnDrawOnImage.setSelected(false);

        JPanel panelDrawTarget = new JPanel();
        panelDrawTarget.setLayout(new GridLayout(1, 2));
        panelDrawTarget.setBorder(BorderFactory.createEmptyBorder(0, 0, 8, 0));
        panelDrawTarget.add(btnDrawOnPainter);
        panelDrawTarget.add(btnDrawOnImage);

        // -------------
        // DRAW BUTTONS
        // -------------
        btnPen = new IcyButton(new IcyIcon(ResourceUtil.ICON_PENCIL, ICON_SIZE - 4));
        btnPen.addActionListener(new ActionListener()
        {

            @Override
            public void actionPerformed(ActionEvent e)
            {
                currentTool = PaintingTool.PEN;
                setSelectedObject(btnPen);
            }
        });

        btnRect = new IcyButton(new IcyIcon(ResourceUtil.ICON_ROI_RECTANGLE, ICON_SIZE));
        btnRect.addActionListener(new ActionListener()
        {

            @Override
            public void actionPerformed(ActionEvent e)
            {
                currentTool = PaintingTool.RECTANGLE;
                setSelectedObject(btnRect);
            }
        });

        btnOval = new IcyButton(new IcyIcon(ResourceUtil.ICON_ROI_OVAL, ICON_SIZE));
        btnOval.addActionListener(new ActionListener()
        {

            @Override
            public void actionPerformed(ActionEvent e)
            {
                currentTool = PaintingTool.OVAL;
                setSelectedObject(btnOval);
            }
        });

        btnRectFill = new IcyButton(new IcyIcon(imageRectFill));
        btnRectFill.addActionListener(new ActionListener()
        {

            @Override
            public void actionPerformed(ActionEvent e)
            {
                currentTool = PaintingTool.PLAIN_RECTANGLE;
                setSelectedObject(btnRectFill);
            }
        });

        btnOvalFill = new IcyButton(new IcyIcon(imageOvalFill));
        btnOvalFill.addActionListener(new ActionListener()
        {

            @Override
            public void actionPerformed(ActionEvent e)
            {
                currentTool = PaintingTool.PLAIN_OVAL;
                setSelectedObject(btnOvalFill);
            }
        });

        btnLine = new IcyButton(new IcyIcon(ResourceUtil.ICON_ROI_LINE, ICON_SIZE));
        btnLine.addActionListener(new ActionListener()
        {

            @Override
            public void actionPerformed(ActionEvent e)
            {
                currentTool = PaintingTool.LINE;
                setSelectedObject(btnLine);
            }
        });

        btnCurve = new IcyButton(new IcyIcon(imageCurve));
        btnCurve.addActionListener(new ActionListener()
        {

            @Override
            public void actionPerformed(ActionEvent e)
            {
                currentTool = PaintingTool.CURVE;
                setSelectedObject(btnCurve);
            }
        });

        btnArrow = new IcyButton(new IcyIcon(ResourceUtil.ICON_ARROW_TOP_RIGHT, ICON_SIZE));
        btnArrow.addActionListener(new ActionListener()
        {

            @Override
            public void actionPerformed(ActionEvent e)
            {
                currentTool = PaintingTool.ARROW;
                setSelectedObject(btnArrow);
            }
        });

        JPanel panelItems = new JPanel(new GridLayout(4, 2));
        panelItems.add(btnPen);
        panelItems.add(btnLine);
        panelItems.add(btnCurve);
        panelItems.add(btnArrow);
        panelItems.add(btnRect);
        panelItems.add(btnOval);
        panelItems.add(btnRectFill);
        panelItems.add(btnOvalFill);

        // ----------------
        // DRAW THICKNESS
        // ----------------
        final JLabel lblLineWidth = new JLabel("Line Width: 20");
        JPanel panelThickness = new JPanel();
        panelThickness.setLayout(new BoxLayout(panelThickness, BoxLayout.X_AXIS));
        panelThickness.add(lblLineWidth);
        panelThickness.add(Box.createHorizontalGlue());

        sliderThickness = new JSlider(1, 100, 20);
        sliderThickness.addChangeListener(new ChangeListener()
        {

            @Override
            public void stateChanged(ChangeEvent e)
            {
                lblLineWidth.setText("Line Width: " + sliderThickness.getValue());
            }
        });

        JPanel panelLineWidth = new JPanel();
        panelLineWidth.setLayout(new BoxLayout(panelLineWidth, BoxLayout.Y_AXIS));
        panelLineWidth.add(panelThickness);
        panelLineWidth.add(sliderThickness);

        // --------
        // COLORS
        // --------
        JPanel panelFG = new JPanel(new BorderLayout());
        btnColorFg = new ColorChooserButton(Color.CYAN);
        btnColorFg.addColorChangeListener(new ColorChangeListener()
        {

            @Override
            public void colorChanged(ColorChooserButton source)
            {
                Sequence s = Icy.getMainInterface().getActiveSequence();
                if ((s != null && s.getSizeC() == 1) || getChannelAffected() != -1)
                {
                    int c = toGray(source.getColor());
                    Color newColor = new Color(c, c, c);
                    if (source.getColor().getRGB() != newColor.getRGB())
                        btnColorFg.setColor(newColor);
                }
            }
        });

        btnInvertColors = new JButton(new ImageIcon(imageInvertArrows));
        btnInvertColors.setSize(10, 10);
        btnInvertColors.addActionListener(new ActionListener()
        {

            @Override
            public void actionPerformed(ActionEvent e)
            {
                Color tmp1 = btnColorFg.getColor();
                Color tmp2 = btnColorBg.getColor();

                Sequence s = Icy.getMainInterface().getActiveSequence();
                if ((s != null && s.getSizeC() == 1) || getChannelAffected() != -1)
                {
                    int c = toGray(tmp1);
                    tmp1 = new Color(c, c, c);

                    c = toGray(tmp2);
                    tmp2 = new Color(c, c, c);
                }
                btnColorFg.setColor(tmp2);
                btnColorBg.setColor(tmp1);
            }
        });

        // Foreground and invert buttons
        panelFG.add(btnColorFg, BorderLayout.CENTER);
        panelFG.add(btnInvertColors, BorderLayout.EAST);

        // Background button
        btnColorBg = new ColorChooserButton(Color.YELLOW);
        btnColorBg.addColorChangeListener(new ColorChangeListener()
        {

            @Override
            public void colorChanged(ColorChooserButton source)
            {
                Sequence s = Icy.getMainInterface().getActiveSequence();
                if ((s != null && s.getSizeC() == 1) || getChannelAffected() != -1)
                {
                    int c = toGray(source.getColor());
                    Color newColor = new Color(c, c, c);
                    if (source.getColor().getRGB() != newColor.getRGB())
                        btnColorBg.setColor(newColor);
                }
            }
        });

        // add to color panel
        JPanel panelColor = new JPanel(new GridLayout(2, 1));
        panelColor.add(panelFG);
        panelColor.add(btnColorBg);

        // ------------------
        // o MAIN GUI SETUP o
        // ------------------
        btnPen.doClick();
        mainPanel.add(panelDrawTarget);
        mainPanel.add(panelItems);
        panelModifyData = new CheckBoxPanel("Only affect:");
        panelModifyData.addActionListener(new ActionListener()
        {

            @Override
            public void actionPerformed(ActionEvent e)
            {
                updateColors();
            }
        });
        panelModifyData.setSelected(false);
        // mainPanel.add(panelModifyData);
        mainPanel.add(panelLineWidth);
        mainPanel.add(panelColor);
        add(mainPanel);
        setSize(new Dimension(120, 310));
    }

    void setCurrentSequence(Sequence sequence)
    {
        currentSequence = sequence;

        if (currentSequence != null)
        {
            addOverlay(sequence);

            if (!getDrawOnPainter())
                btnDrawOnImage.doClick();

            updateColors();
        }
    }

    void updateColors()
    {
        Color tmp1 = btnColorFg.getColor();
        Color tmp2 = btnColorBg.getColor();

        if ((currentSequence != null && currentSequence.getSizeC() == 1) || getChannelAffected() != -1)
        {
            previousFg = tmp1;
            previousBg = tmp2;

            int c = toGray(tmp1);
            tmp1 = new Color(c, c, c);

            c = toGray(tmp2);
            tmp2 = new Color(c, c, c);
        }
        else
        {
            tmp1 = previousBg;
            tmp2 = previousFg;
        }

        btnColorFg.setColor(tmp1);
        btnColorBg.setColor(tmp2);
    }

    public static int toGray(Color c)
    {
        int r = c.getRed();
        int g = c.getGreen();
        int b = c.getBlue();
        return Math.max(r, Math.max(g, b));
    }

    protected void updateChannels()
    {
        panelModifyData.removeAll();
        // panelModifyData.setLayout(new BoxLayout(panelModifyData,
        // BoxLayout.X_AXIS));
        panelModifyData.add(new JLabel("ch. "));
        Sequence s = Icy.getMainInterface().getActiveSequence();
        String[] channels = null;
        if (s != null)
        {
            int sizeC = s.getSizeC();
            channels = new String[sizeC];
            for (int i = 0; i < sizeC; ++i)
            {
                channels[i] = "" + i;
            }
        }
        else
        {
            channels = new String[0];
        }
        comboChannelAffected = new JComboBox(channels);
        panelModifyData.add(comboChannelAffected);
        panelModifyData.add(Box.createHorizontalGlue());
        panelModifyData.setSelected(panelModifyData.isSelected());
        panelModifyData.validate();
    }

    @Override
    public void onClosed()
    {
        super.onClosed();
        
        // unselect current tool
        if (btnSelected != null)
        {
            btnSelected.setSelected(false);
            btnSelected = null;
            currentTool = PaintingTool.NONE;
        }

        // remove listener
        Icy.getMainInterface().removeActiveSequenceListener(activeSequenceListener);
        // release instance
        instance = null;
    }

    public Color getForegroundColor()
    {
        return btnColorFg.getColor();
    }

    public Color getBackgroundColor()
    {
        return btnColorBg.getColor();
    }

    public PaintingTool getTool()
    {
        return currentTool;
    }

    public int getThickness()
    {
        return sliderThickness.getValue();
    }

    /**
     * Returns if wether or not the user wants to draw on a painter or change
     * the image data.
     */
    public boolean getDrawOnPainter()
    {
        return btnDrawOnPainter.isSelected();
    }

    public int getChannelAffected()
    {
        if (panelModifyData.isSelected())
            return comboChannelAffected.getSelectedIndex();

        return -1;
    }

    // /**
    // * Singleton pattern.
    // */
    // public static PaintingTools getInstance()
    // {
    // return singleton;
    // }

    /**
     * Set which button/object is used by the user.
     * 
     * @param button
     */
    void setSelectedObject(JButton button)
    {
        if (btnSelected != button)
        {
            if (btnSelected != null)
                btnSelected.setSelected(false);
            btnSelected = button;
            if (button != null)
                button.setSelected(true);
        }
        else
        {
            btnSelected.setSelected(false);
            btnSelected = null;
            currentTool = PaintingTool.NONE;
        }
    }

    PaintingOverlay getOverlay(Sequence sequence)
    {
        final List<Overlay> overlays = sequence.getOverlays(PaintingOverlay.class);

        if (overlays.size() > 0)
            return (PaintingOverlay) overlays.get(0);

        return null;
    }

    void addOverlay(Sequence sequence)
    {
        if (getOverlay(sequence) == null)
            sequence.addOverlay(new PaintingOverlay("Painting"));
    }

    public class CheckBoxPanel extends ComponentTitledPanel
    {
        /**
         * 
         */
        private static final long serialVersionUID = 8425456904326837378L;
        JCheckBox checkBox;

        CheckBoxPanel(String title)
        {
            super();
            titleComponent = new JCheckBox(title);
            checkBox = (JCheckBox) titleComponent;

            compTitledBorder = new ComponentTitledBorder(checkBox, this, BorderFactory.createEtchedBorder());
            this.setBorder(compTitledBorder);
            borderSet_ = true;

            final CheckBoxPanel thisPanel = this;

            checkBox.addActionListener(new ActionListener()
            {

                @Override
                public void actionPerformed(ActionEvent e)
                {
                    boolean enable = checkBox.isSelected();
                    thisPanel.setChildrenEnabled(enable);
                }
            });

        }

        public void setChildrenEnabled(boolean enabled)
        {

            Component comp[] = this.getComponents();
            for (int i = 0; i < comp.length; i++)
            {
                if (comp[i].getClass().equals(JPanel.class))
                {
                    Component subComp[] = ((JPanel) comp[i]).getComponents();
                    for (int c = 0; c < subComp.length; c++)
                    {
                        subComp[c].setEnabled(enabled);
                    }
                }
                else
                {
                    comp[i].setEnabled(enabled);
                }
            }
        }

        public boolean isSelected()
        {
            return checkBox.isSelected();
        }

        public void setSelected(boolean selected)
        {
            checkBox.setSelected(selected);
            setChildrenEnabled(selected);
        }

        public void addActionListener(ActionListener actionListener)
        {
            checkBox.addActionListener(actionListener);
        }

        public void removeActionListeners()
        {
            for (ActionListener l : checkBox.getActionListeners())
            {
                checkBox.removeActionListener(l);
            }
        }
    }

    public class ComponentTitledPanel extends JPanel
    {

        /**
         * 
         */
        private static final long serialVersionUID = 1L;
        public ComponentTitledBorder compTitledBorder;
        public boolean borderSet_ = false;
        public Component titleComponent;

        @Override
        public void setBorder(Border border)
        {
            if (compTitledBorder != null && borderSet_)
            {
                compTitledBorder.setBorder(border);
            }
            else
            {
                super.setBorder(border);
            }
        }

        @Override
        public Border getBorder()
        {
            return compTitledBorder;
        }

        public void setTitleFont(Font font)
        {
            titleComponent.setFont(font);
        }
    }
}
