package plugins.ylemontag.sequencecomparator;

import icy.sequence.Sequence;

import java.lang.ref.WeakReference;
import java.util.HashSet;
import java.util.Set;
import java.util.WeakHashMap;

/**
 * 
 * @author Yoann Le Montagner
 *
 * Comparison between one reference sequence and any number of test videos
 */
public class LocalComparisonPool extends ComparisonPool<Sequence, LocalComparator>
{	
	private ErrorMapPool _errorMaps;
	
	/**
	 * Constructor
	 */
	public LocalComparisonPool()
	{
		super();
		_errorMaps = new ErrorMapPool();
	}
	
	/**
	 * Return the sequence associated to the given error map, or null if the given
	 * sequence is not an error map
	 */
	public Sequence getErrorMapSource(Sequence errorMap)
	{
		if(errorMap!=null) {
			for(Sequence s : getComparedSequences()) {
				if(getResult(s)==errorMap) {
					return s;
				}
			}
		}
		return null;
	}
	
	/**
	 * Return all the existing error maps (in all local map pools)
	 */
	public static Set<Sequence> getAllErrorMaps()
	{
		return ErrorMapPool.getAllErrorMaps();
	}

	@Override
	protected Sequence allocateResult(Sequence seq)
	{
		return _errorMaps.getErrorMap(seq);
	}

	/**
	 * Keep track of the sequences that are allocated as error maps by the
	 * current LocalComparisonPool and also by all the others 
	 */
	private static class ErrorMapPool
	{	
		private static Set<WeakReference<ErrorMapPool> > _errorMapPools = new HashSet<WeakReference<ErrorMapPool>>();
		private WeakHashMap<Sequence, Sequence> _errorMaps;
		
		/**
		 * Constructor
		 */
		public ErrorMapPool()
		{
			_errorMaps = new WeakHashMap<Sequence, Sequence>();
			synchronized (_errorMapPools) {
				_errorMapPools.add(new WeakReference<ErrorMapPool>(this));
			}
		}
		
		/**
		 * Retrieve the error map associated to given sequence in this pool
		 * @remark A new error map is allocated if necessary
		 */
		public Sequence getErrorMap(Sequence seq)
		{
			synchronized (_errorMaps)
			{
				Sequence retVal = _errorMaps.get(seq);
				if(retVal==null) {
					retVal = new Sequence();
					_errorMaps.put(seq, retVal);
				}
				return retVal;
			}
		}
		
		/**
		 * Return all the error maps contained in all the error map pools
		 */
		public static Set<Sequence> getAllErrorMaps()
		{
			synchronized (_errorMapPools)
			{
				Set<Sequence> retVal = new HashSet<Sequence>();
				for(WeakReference<ErrorMapPool> wr : _errorMapPools) {
					ErrorMapPool currentPool = wr.get();
					if(currentPool==null) {
						continue;
					}
					synchronized (currentPool._errorMaps)
					{
						for(Sequence s : currentPool._errorMaps.keySet()) {
							Sequence currentErrorMap = currentPool._errorMaps.get(s);
							if(currentErrorMap!=null) {
								retVal.add(currentErrorMap);
							}
						}
					}
				}
				return retVal;
			}
		}
	}
}
