package plugins.ylemontag.sequencecomparator.gui;

import icy.gui.main.MainAdapter;
import icy.gui.main.MainEvent;
import icy.main.Icy;
import icy.sequence.Sequence;
import icy.sequence.SequenceAdapter;
import icy.sequence.SequenceEvent;
import icy.sequence.SequenceListener;
import icy.util.StringUtil;

import java.awt.Component;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.WeakHashMap;

import javax.swing.JButton;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.UIManager;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableModel;
import javax.swing.table.TableRowSorter;

import plugins.ylemontag.sequencecomparator.CompareUtil;
import plugins.ylemontag.sequencecomparator.ComparisonPool.StateChangeListener;
import plugins.ylemontag.sequencecomparator.ComparisonState;
import plugins.ylemontag.sequencecomparator.ErrorMeasure;
import plugins.ylemontag.sequencecomparator.GlobalComparisonPool;
import plugins.ylemontag.sequencecomparator.LocalComparisonPool;

/**
 * 
 * @author Yoann Le Montagner
 *
 * Component displaying the list of all comparable sequences with respect to
 * a given reference sequence
 */
public class SampleSequenceComponent extends JTable
{	
	private static final long serialVersionUID = 1L;
	
	/**
	 * Listener for tracking clicks on button renderer
	 */
	public static interface ComparisonListener
	{
		/**
		 * Event handler
		 * @param seq Targeted sequence  
		 */
		public void sequenceSelected(Sequence seq);
	};
	
	private WeakHashMap<Sequence, SequenceListener> _sequenceChangeListeners;
	private Sequence _reference;
	private boolean  _extendZ  ;
	private boolean  _extendT  ;
	private GlobalComparisonPool _globalComparisonPool;
	private LocalComparisonPool  _localComparisonPool ;
	private DefaultTableModel _model;
	private LinkedList<ComparisonListener> _globalComparisonListeners;
	private LinkedList<ComparisonListener> _localComparisonListeners ;
	
	/**
	 * Constructor
	 */
	public SampleSequenceComponent(GlobalComparisonPool globalComparisonPool,
		LocalComparisonPool localComparisonPool)
	{
		super();
		_sequenceChangeListeners = new WeakHashMap<Sequence, SequenceListener>();
		_extendZ = false;
		_extendT = false;
		
		// Comparison pools
		_globalComparisonPool = globalComparisonPool;
		_globalComparisonPool.addStateChangeListener(new StateChangeListener()
		{
			@Override
			public void stateChanged(Sequence seq, ComparisonState state, Object result) {
				int targetRow = searchSequence(seq);
				if(targetRow>=0) {
					Double overallError = state==ComparisonState.READY ? ((ErrorMeasure)result).errorAll : null; 
					_model.setValueAt(overallError, targetRow, OVERALL_ERROR_COLUMN      );
					_model.setValueAt(state       , targetRow, GLOBAL_ERROR_BUTTON_COLUMN);
				}
			}
		});
		_localComparisonPool = localComparisonPool;
		_localComparisonPool.addStateChangeListener(new StateChangeListener()
		{
			@Override
			public void stateChanged(Sequence seq, ComparisonState state, Object result) {
				int targetRow = searchSequence(seq);
				if(targetRow>=0) {
					_model.setValueAt(state, targetRow, LOCAL_ERROR_BUTTON_COLUMN);
				}
			}
		});
		
		// Model
		_model = new SampleTableModel();
		setModel(_model);
		TableRowSorter<TableModel> sorter = new TableRowSorter<TableModel>(_model);
		setRowSorter(sorter);
		Icy.getMainInterface().addListener(new MainAdapter()
		{
			@Override
			public void sequenceOpened(MainEvent event) {
				Sequence seq = (Sequence)event.getSource();
				registerSequenceChangedHandler(seq);
				registerSequence(seq);
			}
			
			@Override
			public void sequenceClosed(MainEvent event) {
				Sequence seq = (Sequence)event.getSource();
				unregisterSequenceChangedHandler(seq);
				unregisterSequence(seq);
			}
		});
		for(Sequence seq : Icy.getMainInterface().getSequences()) {
			registerSequenceChangedHandler(seq);
		}
		
		// Renderer and listeners
		_globalComparisonListeners = new LinkedList<ComparisonListener>();
		_localComparisonListeners  = new LinkedList<ComparisonListener>();
		getColumn(getColumnName(OVERALL_ERROR_COLUMN      )).setCellRenderer(new OverallErrorRenderer());
		getColumn(getColumnName(GLOBAL_ERROR_BUTTON_COLUMN)).setCellRenderer(new ButtonRenderer      ());
		getColumn(getColumnName(LOCAL_ERROR_BUTTON_COLUMN )).setCellRenderer(new ButtonRenderer      ());
		addMouseListener(new MouseAdapter()
		{
			@Override
			public void mouseClicked(MouseEvent e) {
				int c = columnAtPoint(e.getPoint());
				LinkedList<ComparisonListener> listeners;
				switch(c) {
					case GLOBAL_ERROR_BUTTON_COLUMN: listeners=_globalComparisonListeners; break;
					case LOCAL_ERROR_BUTTON_COLUMN : listeners=_localComparisonListeners ; break;
					default: return;
				}
				int r = rowAtPoint(e.getPoint());
				if(r<0) {
					return;
				}
				Sequence seq = getSequenceAtRow(r); 
				for(ComparisonListener l : listeners) {
					l.sequenceSelected(seq);
				}
			}
		});
	}
	
	/**
	 * Add a new listener for action performed on the global comparison button
	 */
	public void addGlobalComparisonListener(ComparisonListener l)
	{
		_globalComparisonListeners.add(l);
	}
	
	/**
	 * Add a new listener for action performed on the global comparison button
	 */
	public void addLocalComparisonListener(ComparisonListener l)
	{
		_localComparisonListeners.add(l);
	}
	
	/**
	 * Return the sequence corresponding to a given row
	 */
	public Sequence getSequenceAtRow(int r)
	{
		return (Sequence)getValueAt(r, SEQUENCE_COLUMN);
	}
	
	/**
	 * Change the reference
	 */
	public void changeReference(Sequence newReference, boolean extendZ, boolean extendT)
	{
		if(_reference==newReference && _extendZ==extendZ && _extendT==extendT) {
			return;
		}
		_reference = newReference;
		_extendZ   = extendZ     ;
		_extendT   = extendT     ;
		_model.setRowCount(0);
		ArrayList<Sequence> sequences = Icy.getMainInterface().getSequences();
		for(Sequence seq : sequences) {
			registerSequence(seq);
		}
	}
	
	/**
	 * Register the sequence changed handler for the given sequence
	 */
	private void registerSequenceChangedHandler(Sequence seq)
	{
		if(_sequenceChangeListeners.containsKey(seq)) {
			return;
		}
		SequenceListener l = new SequenceAdapter()
		{
			@Override
			public void sequenceChanged(SequenceEvent sequenceEvent) {
				onSequenceChanged(sequenceEvent.getSequence());
			}
		};
		seq.addListener(l);
		_sequenceChangeListeners.put(seq, l);
	}
	
	/**
	 * Unregister the sequence changed handler associated to a given sequence
	 */
	private void unregisterSequenceChangedHandler(Sequence seq)
	{
		SequenceListener l = _sequenceChangeListeners.get(seq);
		if(l!=null) {
			seq.removeListener(l);
			_sequenceChangeListeners.remove(seq);
		}
	}
	
	/**
	 * Sequence changed handler
	 */
	private void onSequenceChanged(Sequence seq)
	{
		if(seq==_reference) {
			_model.setRowCount(0);
			ArrayList<Sequence> sequences = Icy.getMainInterface().getSequences();
			for(Sequence s : sequences) {
				registerSequence(s);
			}
		}
		else {
			if(searchSequence(seq)>=0) {
				if(shouldBeInTheList(seq)) {
					repaint();
				}
				else {
					unregisterSequence(seq);
				}
			}
			else if(shouldBeInTheList(seq)) {
				registerSequence(seq);
			}
		}
	}
	
	/**
	 * Append a sequence to the list if it can be used as a reference
	 */
	private void registerSequence(Sequence seq)
	{
		if(!shouldBeInTheList(seq) || searchSequence(seq)>=0) {
			return;
		}
		ComparisonState currentGlobalState  = _globalComparisonPool.getState (seq);
		ComparisonState currentLocalState   = _localComparisonPool .getState (seq);
		ErrorMeasure    currentErrorMeasure = _globalComparisonPool.getResult(seq);
		_model.addRow(new Object[] {
			seq,
			currentGlobalState==ComparisonState.READY ? currentErrorMeasure.errorAll : null,
			currentGlobalState,
			currentLocalState 
		});
	}
	
	/**
	 * Remove a sequence from the list
	 */
	private void unregisterSequence(Sequence seq)
	{
		int targetRow = searchSequence(seq);
		if(targetRow>=0) {
			_model.removeRow(targetRow);
		}
	}
	
	/**
	 * Return the row index corresponding to the given sequence, or -1 if such
	 * a sequence is not present in the model
	 */
	private int searchSequence(Sequence seq)
	{
		int rowCount = _model.getRowCount();
		for(int k=0; k<rowCount; ++k) {
			if(_model.getValueAt(k, SEQUENCE_COLUMN)==seq) {
				return k;
			}
		}
		return -1;
	}
	
	/**
	 * Check if the given sequence should be in the list
	 */
	private boolean shouldBeInTheList(Sequence seq)
	{
		return _reference!=null && seq!=_reference &&
			CompareUtil.areComparable(_reference, seq, _extendZ, _extendT) &&
			!LocalComparisonPool.getAllErrorMaps().contains(seq);
	}
	
	/**
	 * Index of the sequence name column
	 */
	private static final int SEQUENCE_COLUMN = 0;
	
	/**
	 * Index of the overall error column
	 */
	private static final int OVERALL_ERROR_COLUMN = 1;
	
	/**
	 * Index of the global error button column
	 */
	private static final int GLOBAL_ERROR_BUTTON_COLUMN = 2;
	
	/**
	 * Index of the global error button column
	 */
	private static final int LOCAL_ERROR_BUTTON_COLUMN = 3;
	
	/**
	 * Model used in the component
	 */
	private static class SampleTableModel extends DefaultTableModel
	{
		private static final long serialVersionUID = 1L;
		
		private Class<?>[] _columnClass;
		private String  [] _columnName;
		
		/**
		 * Constructor
		 */
		public SampleTableModel()
		{
			super();
			_columnClass = new Class<?>[] {
				Sequence       .class,
				Double         .class,
				ComparisonState.class,
				ComparisonState.class
			};
			_columnName = new String[] {
					"Comparable sequences",
					"Overall error"       ,
					"Error measure"       ,
					"Error map"
			};
			for(int k=0; k<getColumnCount(); ++k) {
				addColumn(_columnName[k]);
			}
		}
		
		@Override
		public boolean isCellEditable(int row, int column)
		{
			return false;
		}
		
		@Override
		public int getColumnCount()
		{
			return _columnName.length;
		}
		
		@Override
		public Class<?> getColumnClass(int columnIndex)
		{
			return _columnClass[columnIndex];
		}
		
		@Override
		public String getColumnName(int columnIndex)
		{
			return _columnName[columnIndex];
		}
	}
	
	/**
	 * Renderer for the state columns
	 */
	private static class ButtonRenderer extends JButton implements TableCellRenderer
	{
		private static final long serialVersionUID = 1L;

		@Override
		public Component getTableCellRendererComponent(JTable table, Object value,
			boolean isSelected, boolean hasFocus, int row, int column)
		{
			if(isSelected) {
				setForeground(table.getSelectionForeground());
				setBackground(table.getSelectionBackground());
			}
			else {
				setForeground(table.getForeground());
				setBackground(UIManager.getColor("Button.background"));
			}
			ComparisonState state = (ComparisonState)value;
			switch(state) {
				case NOT_WATCHED: setText("Disabled"    ); break;
				case COMPUTING  : setText("Computing..."); break;
				case READY      : setText("Enabled"     ); break;
			}
			return this;
		}
	}
	
	/**
	 * Renderer for the overall error column
	 */
	private static class OverallErrorRenderer extends DefaultTableCellRenderer
	{
		private static final long serialVersionUID = 1L;
		
		@Override
		protected void setValue(Object value)
		{
			if(value==null) {
				super.setValue("--");
				setHorizontalAlignment(SwingConstants.CENTER);
			}
			else {
				double overallError = (Double)value;
				super.setValue(StringUtil.toString(overallError));
				setHorizontalAlignment(SwingConstants.RIGHT);
			}
		}
	}
}
