/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * Contributors:
 * Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * Nicolas Chenouard (nicolas.chenouard@gmail.com)
 * Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 * Emrah Bostan (emrah.bostan@gmail.com)
 * Ulugbek S. Kamilov (kamilov@gmail.com)
 * Ramtin Madani (ramtin_madani@yahoo.com)
 * Masih Nilchian (masih_n85@yahoo.com)
 * C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 * Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 * Cl&#233;ment Marti (clement.marti@epfl.ch)
 * Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnake.roi;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.geom.Area;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Line2D;
import java.awt.geom.Path2D;
import java.awt.geom.PathIterator;
import java.awt.geom.Point2D;
import java.util.ArrayList;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

import icy.canvas.Canvas2D;
import icy.canvas.IcyCanvas;
import icy.canvas.IcyCanvas2D;
import icy.painter.Anchor2D;
import icy.painter.PathAnchor2D;
import icy.sequence.Sequence;
import icy.type.point.Point5D;
import icy.util.ShapeUtil;
import icy.util.XMLUtil;
import plugins.big.bigsnake.keeper.SnakeKeeper;
import plugins.big.bigsnake.snake.ESnake;
import plugins.big.bigsnakeutils.icy.snake2D.Snake2D;
import plugins.big.bigsnakeutils.icy.snake2D.Snake2DNode;
import plugins.big.bigsnakeutils.icy.snake2D.Snake2DScale;
import plugins.kernel.canvas.VtkCanvas;
import plugins.kernel.roi.roi2d.ROI2DPath;

/**
 * Class that handles the interaction and the display of the snake.
 * 
 * @version May 24, 2015
 * @author Nicolas Chenouard (nicolas.chenouard@gmail.com)
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * @author Ulugbek S. Kamilov (kamilov@gmail.com)
 */
public class ROI2DSnake extends ROI2DPath
{

    /** Snake associated to the ROI. */
    private Snake2D snake_ = null;
    /** Keeper associated to the ROI */
    private SnakeKeeper keeper_ = null;
    /** Group of all polylines forming the skin of the snake. */
    private Snake2DScale[] scales_ = null;

    // ----------------------------------------------------------------------------
    // STATUS FIELDS

    /** Denotes the current state of the snake for mouse interaction. */
    private SnakeEditMode snakeEditMode_ = SnakeEditMode.MOVE_SNAKE;
    /** Flag that denotes if the snake can be modified. */
    private boolean isEditable_;

    // ----------------------------------------------------------------------------
    // COLORS

    /** Default ROI color for the snake. */
    private static final Color DEFAULT_NORMAL_COLOR = Color.RED;
    /** Default ROI color for the snake when selected. */
    private static final Color DEFAULT_SELECTED_COLOR = Color.GREEN;

    // ----------------------------------------------------------------------------
    // CURSORS

    /** Cursor shown on the screen when moving a control point. */
    private final Cursor moveControlPointCursor_ = Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR);
    /** Cursor shown on the screen when moving the snake. */
    private final Cursor moveSnakeCursor_ = Cursor.getPredefinedCursor(Cursor.HAND_CURSOR);
    /** Cursor shown on the screen when dragging the whole snake. */
    private final Cursor dragSnakeCursor_ = Cursor.getPredefinedCursor(Cursor.HAND_CURSOR);
    /** Cursor shown on the screen when rotating the whole snake. */
    private final Cursor rotateSnakeCursor_ = Cursor.getPredefinedCursor(Cursor.HAND_CURSOR);
    /** Cursor shown on the screen when dilating the whole snake. */
    private final Cursor dilateSnakeCursor_ = Cursor.getPredefinedCursor(Cursor.HAND_CURSOR);

    // ----------------------------------------------------------------------------
    // XML TAGS

    /** XML tag for the snake parameters. */
    public static final String ID_SNAKE_PARAMETERS = "snake_parameters";

    // ============================================================================
    // SNAKE2DROI METHODS

    // ----------------------------------------------------------------------------
    // PUBLIC METHODS

    /** Default constructor. */
    public ROI2DSnake()
    {
        super(new Path2D.Double());
    }

    // ----------------------------------------------------------------------------

    /** Constructor. */
    public ROI2DSnake(ESnake snake, SnakeKeeper keeper)
    {
        super(new Path2D.Double());
        snake_ = snake;
        scales_ = snake.getScales();
        isEditable_ = true;
        keeper_ = keeper;
        refreshROI();
        setEditMode(SnakeEditMode.MOVE_SNAKE);
        setSelected(true);
    }

    // ----------------------------------------------------------------------------

    public void changePosition(double[] xPosition, double[] yPosition, Snake2DScale[] scales)
    {
//        if (xPosition.length != controlPoints.size() || yPosition.length != controlPoints.size())
//        {
//            throw new IllegalArgumentException(
//                    "the number of modified controlPoints should be the same as the number of original control points");
//        }
        for (int i = 0; i < xPosition.length; i++)
        {
            controlPoints.get(i).setPosition(xPosition[i], yPosition[i]);
        }
        scales_ = scales;
    }

    // ----------------------------------------------------------------------------

    public Point2D getControlPoint(int i)
    {
        if (i < controlPoints.size())
        {
            return new Point2D.Double(controlPoints.get(i).getX(), controlPoints.get(i).getY());
        }
        else
        {
            return null;
        }
    }

    // ----------------------------------------------------------------------------

    public void refreshROI()
    {
        refreshScales();
        removeAllPoint();

        Path2D path = getPath();

        synchronized (path)
        {
            path.reset();
            Snake2DNode[] nodes = snake_.getNodes();
            for (int i = 0; i < nodes.length; i++)
            {
                if (i == 0)
                {
                    path.moveTo(nodes[i].getX(), nodes[i].getY());
                }
                else
                {
                    path.lineTo(nodes[i].getX(), nodes[i].getY());
                }
            }
            path.closePath();
        }

        closedArea = new Area(ShapeUtil.getClosedPath(path));
        openPath = ShapeUtil.getOpenPath(path);

        for (Anchor2D pt : getAnchorsFromShape(path))
        {
            addPoint(pt);
        }
    }

    // ----------------------------------------------------------------------------

    public void refreshScales()
    {
        scales_ = snake_.getScales();
    }

    // ----------------------------------------------------------------------------

    public void setEditMode(SnakeEditMode editMode)
    {
        snakeEditMode_ = editMode;
        for (Anchor2D pt : controlPoints)
        {
            pt.setSelected(false);
        }
    }

    // ----------------------------------------------------------------------------
    // PRIVATE METHODS

    private ArrayList<Anchor2D> getAnchorsFromShape(Shape shape)
    {
        final PathIterator pathIt = shape.getPathIterator(null);
        final ArrayList<Anchor2D> result = new ArrayList<Anchor2D>();
        final double crd[] = new double[6];
        while (!pathIt.isDone())
        {
            final int segType = pathIt.currentSegment(crd);
            PathAnchor2D pt = (PathAnchor2D) createAnchor(new Point2D.Double(crd[0], crd[1]));
            pt.setType(segType);
            pt.setVisible(segType != PathIterator.SEG_CLOSE);
            result.add(pt);
            pathIt.next();
        }
        return result;
    }

    // ============================================================================
    // ROI2DPATH METHODS

    // ----------------------------------------------------------------------------
    // PUBLIC METHODS

    /**
     * Method that creates a copy of the contour of the snake as an
     * <code>ROI2DPath</code>.
     */
    @Override
    public ROI2DPath getCopy()
    {
        if (snake_ == null)
        {
            System.err.println("ROI contains no valid snake.");
            return null;
        }
        Snake2DScale[] skin = snake_.getScales();
        if (skin == null)
        {
            System.err.println("The snake is corrupt. Skin not found.");
            return null;
        }
        else if (skin.length == 0)
        {
            System.err.println("The snake is corrupt. Invalid contour size.");
            return null;
        }
        else if (skin[0] == null)
        {
            System.err.println("The snake is corrupt. No contour is availiable.");
            return null;
        }
        return new ROI2DPath(skin[0]);
    }

    // ----------------------------------------------------------------------------

    /** Gets the bounding rectangle of the first polyline of the snake skin. */
    @Override
    public Rectangle computeBounds2D()
    {
        if (scales_ != null)
        {
            if (scales_.length >= 1)
            {
                return scales_[0].getBounds2D().getBounds();
            }
        }
        return null;
    }

    // ----------------------------------------------------------------------------

    /** Get the ROI as a boolean bitmap mask for the specified rectangular area. */
    @Override
    public boolean[] getAsBooleanMask(int x, int y, int w, int h, boolean inclusive)
    {
        if (scales_ != null)
        {
            if (scales_.length >= 1)
            {
                Snake2DScale scale = scales_[0];
                final boolean[] result = new boolean[w * h];
                int offset = 0;
                for (int j = 0; j < h; j++)
                {
                    for (int i = 0; i < w; i++)
                    {
                        result[offset] = scale.contains(x + i, y + j);
                        if (inclusive)
                        {
                            result[offset] |= scale.intersects(x + i, y + j, 1, 1);
                        }
                        offset++;
                    }
                }
                return result;
            }
        }
        return null;
    }

    // ----------------------------------------------------------------------------

    /** Gets the color of the first polyline forming the skin of the snake. */
    @Override
    public Color getColor()
    {
        if (scales_ != null)
        {
            if (scales_.length >= 1)
            {
                return scales_[0].getColor();
            }
        }
        return DEFAULT_NORMAL_COLOR;
    }

    // ----------------------------------------------------------------------------

    /** Saves a snake from an XML node. */
    @Override
    public boolean saveToXML(Node node)
    {
        if (snake_ != null)
        {
            Element snakeParametersElement = XMLUtil.setElement(node, ID_SNAKE_PARAMETERS);
            snake_.saveToXML(snakeParametersElement);
        }
        return true;
    }

    // ----------------------------------------------------------------------------

    /** Sets the edit status of the ROI. */
    @Override
    public void setEditable(boolean isEditable)
    {
        isEditable_ = isEditable;
    }

    // ----------------------------------------------------------------------------
    // PROTECTED METHODS

    @Override
    protected Anchor2D createAnchor(Point2D pos)
    {
        return new ControlPointPainter(pos, DEFAULT_NORMAL_COLOR, DEFAULT_SELECTED_COLOR);
    }

    // ----------------------------------------------------------------------------

    @Override
    protected ROI2DShapePainter createPainter()
    {
        return new Snake2DPainter();
    }

    // ============================================================================
    // INNER CLASSES

    /**
     * Class that handles the interaction and the display of the control points
     * of the snake.
     * 
     * @version October 30, 2013
     * @author Nicolas Chenouard (nicolas.chenouard@gmail.com)
     * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
     * @author Ulugbek S. Kamilov (kamilov@gmail.com)
     */
    private class ControlPointPainter extends PathAnchor2D
    {

        /** Size of the control point on the display. */
        private static final int CONTROL_POINT_SIZE = 8;

        // ============================================================================
        // PUBLIC METHODS

        /** Constructor. */
        public ControlPointPainter(Point2D pos, Color defaultSelectedColor, Color overColor)
        {
            super(pos.getX(), pos.getY(), CONTROL_POINT_SIZE, defaultSelectedColor, overColor);
        }

        // ----------------------------------------------------------------------------

        /** Draws the cross of the control point on the canvas. */
        @Override
        public void paint(Graphics2D g, Sequence sequence, IcyCanvas canvas)
        {
            if (!isVisible())
                return;

            if (canvas instanceof IcyCanvas2D)
            {
                final double adjRayX = canvas.canvasToImageLogDeltaX(ray);
                final double adjRayY = canvas.canvasToImageLogDeltaY(ray);
                double stroke = getAdjustedStroke(canvas);
                g.setStroke(new BasicStroke((float) stroke));
                if (selected)
                {
                    g.setColor(selectedColor);
                }
                else
                {
                    g.setColor(color);
                }
                g.draw(new Line2D.Double(position.x - adjRayX, position.y, position.x + adjRayX, position.y));
                g.draw(new Line2D.Double(position.x, position.y - adjRayY, position.x, position.y + adjRayY));
            }
            else
            {
                System.err.println("Unrecognized IcyCanvas.");
            }
        }
    }

    // ----------------------------------------------------------------------------

    /**
     * Class that draws the ROI forming the snake.
     * 
     * @version October 30, 2013
     * @author Nicolas Chenouard (nicolas.chenouard@gmail.com)
     * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
     * @author Ulugbek S. Kamilov (kamilov@gmail.com)
     */
    private class Snake2DPainter extends ROI2DShapePainter
    {

        /** Point in image coordinates where the is clicking clicked. */
        private Point5D.Double pressedImagePoint_ = null;
        /** Point in image coordinates where the is clicking clicked. */
        private Point2D firstControlPointInitPosition_ = null;
        /** Point with the center of gravity of the snake curve. */
        private Point2D snakeMassCenter_ = null;

        /** Flag to determine if the mouse pointer is within the snake curve. */
        private boolean isMouseInROI_ = false;

        // ============================================================================
        // PUBLIC METHODS

        /**
         * Invoked when a key is pressed. This method is used to block the key
         * events from Icy.
         */
        @Override
        public void keyPressed(KeyEvent e, Point5D.Double imagePoint, IcyCanvas canvas)
        {
            e.consume();
        }

        // ----------------------------------------------------------------------------

        /**
         * Invoked when the mouse cursor has been moved onto the Icy canvas but
         * no buttons have been pushed.
         */
        @Override
        public void mouseMove(MouseEvent e, Point5D.Double imagePoint, IcyCanvas canvas)
        {
            if (isEditable_)
            {
                switch (snakeEditMode_)
                {
                    case DILATE_SNAKE:
                        mouseMoveDilate(e, imagePoint, canvas);
                        break;
                    case ROTATE_SNAKE:
                        mouseMoveRotate(e, imagePoint, canvas);
                        break;
                    default:
                        mouseMoveDefault(e, imagePoint, canvas);
                        break;
                }
            }
        }

        // ----------------------------------------------------------------------------

        /** Invoked when a mouse button has been pressed on the Icy canvas. */
        @Override
        public void mousePressed(MouseEvent e, Point5D.Double imagePoint, IcyCanvas canvas)
        {
            pressedImagePoint_ = imagePoint;
            if (isEditable_)
            {
                boolean isMouseOnControlPoint = false;
                for (Anchor2D pt : controlPoints)
                {
                    if (pt.isOver(canvas, imagePoint.getX(), imagePoint.getY()))
                    {
                        isMouseOnControlPoint = true;
                    }
                }
                isMouseInROI_ = ROI2DSnake.this.contains(imagePoint);
                if (!(isMouseInROI_ && imagePoint.equals(pressedImagePoint_)) && !isMouseOnControlPoint)
                {
                    keeper_.deactivateSnake();
                    setSelected(false);
                }
                switch (snakeEditMode_)
                {
                    case DILATE_SNAKE:
                        mousePressedDilate(e, imagePoint, canvas);
                        break;
                    case ROTATE_SNAKE:
                        mousePressedRotate(e, imagePoint, canvas);
                        break;
                    default:
                        mousePressedDefault(e, imagePoint, canvas);
                        break;
                }
            }
        }

        // ----------------------------------------------------------------------------

        /**
         * Invoked when a mouse button is pressed on the Icy canvas and then
         * dragged.
         */
        @Override
        public void mouseDrag(MouseEvent e, Point5D.Double imagePoint, IcyCanvas canvas)
        {
            if (isEditable_)
            {
                switch (snakeEditMode_)
                {
                    case DILATE_SNAKE:
                        mouseDragDilate(e, imagePoint, canvas);
                        break;
                    case ROTATE_SNAKE:
                        mouseDragRotate(e, imagePoint, canvas);
                        break;
                    default:
                        mouseDragDefault(e, imagePoint, canvas);
                        break;
                }
            }
        }

        // ----------------------------------------------------------------------------

        /** Invoked when a mouse button has been pressed on the Icy canvas. */
        @Override
        public void mouseReleased(MouseEvent e, Point5D.Double imagePoint, IcyCanvas canvas)
        {
            if (!isEditable_)
            {
                if (!isActiveFor(canvas))
                {
                    return;
                }
                ROI2DSnake.this.beginUpdate();
                try
                {
                    // First check if the mouse is inside the ROI
                    isMouseInROI_ = ROI2DSnake.this.contains(imagePoint);
                    if (isMouseInROI_ && imagePoint.equals(pressedImagePoint_))
                    {
                        keeper_.activateSnake();
                        setSelected(true);
                    }
                }
                finally
                {
                    ROI2DSnake.this.endUpdate();
                }
            }
        }

        // ============================================================================
        // PROTECTED METHODS

        /**
         * Draws on the image canvas all the visual representation of the snake
         * (polylines, control points, etc.).
         */
        @Override
        public void drawROI(Graphics2D g, Sequence sequence, IcyCanvas canvas)
        {
            if (canvas instanceof IcyCanvas2D)
            {
                final Graphics2D g2 = (Graphics2D) g.create();
                if (scales_ != null)
                {
                    double stroke = getAdjustedStroke(canvas);
                    g2.setStroke(new BasicStroke((float) (stroke / 2)));
                    int imax = isEditable_ ? scales_.length : 1;
                    for (int i = 0; i < imax; i++)
                    {
                        g2.setColor(scales_[i].getColor());
                        g2.draw(scales_[i]);
                    }
                }
                if ((ROI2DSnake.this.snakeEditMode_ == SnakeEditMode.DILATE_SNAKE
                        || ROI2DSnake.this.snakeEditMode_ == SnakeEditMode.ROTATE_SNAKE) && isMouseInROI_
                        && (snakeMassCenter_ != null) && (firstControlPointInitPosition_ != null))
                {
                    g2.setColor(Color.GREEN);
                    final double sizeX = canvas.canvasToImageLogDeltaX(5);
                    final double sizeY = canvas.canvasToImageLogDeltaX(5);
                    g2.fill(new Ellipse2D.Double(snakeMassCenter_.getX() - sizeX / 2,
                            snakeMassCenter_.getY() - sizeY / 2, sizeX, sizeY));
                }
                if (isEditable_)
                {
                    for (Anchor2D pt : controlPoints)
                    {
                        pt.paint(g2, sequence, canvas);
                    }
                }

                // draw (i.e. display) the IDs of the snake close to the contour
                /*
                 * if (keeper_ != null) { String id = keeper_.getID(); if (id !=
                 * null) { g2.drawString(id, (int) controlPoints.get(0).getX(),
                 * (int) controlPoints.get(0).getY()); } }
                 */

                g2.dispose();
            }
            else if (canvas instanceof VtkCanvas)
            {
                super.drawROI(g, sequence, canvas);
            }
        }

        // ============================================================================
        // PRIVATE METHODS

        /** Default action when mouse cursor has been moved onto the Icy canvas. */
        private void mouseMoveDefault(MouseEvent e, Point5D.Double imagePoint, IcyCanvas canvas)
        {
            if (!isActiveFor(canvas))
            {
                return;
            }
            ROI2DSnake.this.beginUpdate();
            try
            {
                isMouseInROI_ = ROI2DSnake.this.contains(imagePoint);
                if (!isMouseInROI_)
                {
                    for (Snake2DScale scale : ROI2DSnake.this.scales_)
                    {
                        isMouseInROI_ = scale.contains(imagePoint.getX(), imagePoint.getY());
                        if (isMouseInROI_)
                        {
                            break;
                        }
                    }
                }
                for (Anchor2D pt : controlPoints)
                {
                    pt.setSelected(false);
                    if (pt.isOver(canvas, imagePoint.getX(), imagePoint.getY()))
                    {
                        ((Canvas2D) canvas).getViewComponent().setCursor(moveControlPointCursor_);
                        isMouseInROI_ = false;
                    }
                }
                if (isMouseInROI_)
                {
                    ((Canvas2D) canvas).getViewComponent().setCursor(moveSnakeCursor_);
                    for (Anchor2D pt : controlPoints)
                    {
                        pt.setSelected(true);
                    }
                }
                else
                {
                    for (Anchor2D pt : controlPoints)
                    {
                        pt.mouseMove(e, imagePoint, canvas);
                    }
                }
            }
            finally
            {
                ROI2DSnake.this.endUpdate();
            }
        }

        // ----------------------------------------------------------------------------

        private void mouseMoveDilate(MouseEvent e, Point5D.Double imagePoint, IcyCanvas canvas)
        {
            if (!isActiveFor(canvas))
            {
                return;
            }
            ROI2DSnake.this.beginUpdate();
            try
            {
                isMouseInROI_ = ROI2DSnake.this.contains(imagePoint);
                if (!isMouseInROI_)
                {
                    for (Snake2DScale scale : ROI2DSnake.this.scales_)
                    {
                        isMouseInROI_ = scale.contains(imagePoint.getX(), imagePoint.getY());
                        if (isMouseInROI_)
                        {
                            break;
                        }
                    }
                }
                if (isMouseInROI_)
                {
                    ((Canvas2D) canvas).getViewComponent().setCursor(dilateSnakeCursor_);
                    for (Anchor2D pt : controlPoints)
                    {
                        pt.setSelected(true);
                    }
                }
                else
                {
                    for (Anchor2D pt : controlPoints)
                    {
                        pt.setSelected(false);
                    }
                    snakeMassCenter_ = null;
                    firstControlPointInitPosition_ = null;
                }
            }
            finally
            {
                ROI2DSnake.this.endUpdate();
            }
        }

        // ----------------------------------------------------------------------------

        private void mouseMoveRotate(MouseEvent e, Point5D.Double imagePoint, IcyCanvas canvas)
        {
            if (!isActiveFor(canvas))
            {
                return;
            }
            ROI2DSnake.this.beginUpdate();
            try
            {
                isMouseInROI_ = ROI2DSnake.this.contains(imagePoint);
                if (!isMouseInROI_)
                {
                    for (Snake2DScale scale : ROI2DSnake.this.scales_)
                    {
                        isMouseInROI_ = scale.contains(imagePoint.getX(), imagePoint.getY());
                        if (isMouseInROI_)
                        {
                            break;
                        }
                    }
                }
                if (isMouseInROI_)
                {
                    ((Canvas2D) canvas).getViewComponent().setCursor(rotateSnakeCursor_);
                    for (Anchor2D pt : controlPoints)
                    {
                        pt.setSelected(true);
                    }
                }
                else
                {
                    for (Anchor2D pt : controlPoints)
                    {
                        pt.setSelected(false);
                    }
                    snakeMassCenter_ = null;
                    firstControlPointInitPosition_ = null;
                }
            }
            finally
            {
                ROI2DSnake.this.endUpdate();
            }
        }

        // ----------------------------------------------------------------------------

        /**
         * Default actions when a mouse button has been pressed on the Icy
         * canvas.
         */
        private void mousePressedDefault(MouseEvent e, Point5D.Double imagePoint, IcyCanvas canvas)
        {
            if (!isActiveFor(canvas))
            {
                return;
            }
            ROI2DSnake.this.beginUpdate();
            try
            {
                if (isMouseInROI_)
                {
                    ((Canvas2D) canvas).getViewComponent().setCursor(dragSnakeCursor_);
                    e.consume();
                }
                else
                {
                    for (Anchor2D pt : controlPoints)
                    {
                        pt.mousePressed(e, imagePoint, canvas);
                    }
                }
            }
            finally
            {
                ROI2DSnake.this.endUpdate();
            }
        }

        // ----------------------------------------------------------------------------

        private void mousePressedDilate(MouseEvent e, Point5D.Double imagePoint, IcyCanvas canvas)
        {
            if (!isActiveFor(canvas))
            {
                return;
            }
            ROI2DSnake.this.beginUpdate();
            try
            {
                if (isMouseInROI_)
                {
                    ((Canvas2D) canvas).getViewComponent().setCursor(dilateSnakeCursor_);
                    snakeMassCenter_ = snake_.getCentroid();
                    firstControlPointInitPosition_ = (Point2D) ROI2DSnake.this.getControlPoint(0).clone();
                    e.consume();
                }
                else
                {
                    snakeMassCenter_ = null;
                    firstControlPointInitPosition_ = null;
                }
            }
            finally
            {
                ROI2DSnake.this.endUpdate();
            }
        }

        // ----------------------------------------------------------------------------

        private void mousePressedRotate(MouseEvent e, Point5D.Double imagePoint, IcyCanvas canvas)
        {
            if (!isActiveFor(canvas))
            {
                return;
            }
            ROI2DSnake.this.beginUpdate();
            try
            {
                if (isMouseInROI_)
                {
                    ((Canvas2D) canvas).getViewComponent().setCursor(rotateSnakeCursor_);
                    snakeMassCenter_ = snake_.getCentroid();
                    firstControlPointInitPosition_ = (Point2D) ROI2DSnake.this.getControlPoint(0).clone();
                    e.consume();
                }
                else
                {
                    snakeMassCenter_ = null;
                    firstControlPointInitPosition_ = null;
                }
            }
            finally
            {
                ROI2DSnake.this.endUpdate();
            }
        }

        // ----------------------------------------------------------------------------

        /**
         * Default action when a mouse button is pressed on the Icy canvas and
         * then dragged.
         */
        private void mouseDragDefault(MouseEvent e, Point5D.Double imagePoint, IcyCanvas canvas)
        {
            if (!isActiveFor(canvas))
            {
                return;
            }
            ROI2DSnake.this.beginUpdate();
            try
            {
                for (Anchor2D pt : controlPoints)
                {
                    if (pt.isOver(canvas, imagePoint.getX(), imagePoint.getY()))
                    {
                        ((Canvas2D) canvas).getViewComponent().setCursor(moveControlPointCursor_);
                    }
                }
                if (isMouseInROI_ && pressedImagePoint_ != null)
                {
                    ((Canvas2D) canvas).getViewComponent().setCursor(dragSnakeCursor_);
                    e.consume();
                    double dx = imagePoint.getX() - pressedImagePoint_.getX();
                    double dy = imagePoint.getY() - pressedImagePoint_.getY();
                    pressedImagePoint_ = imagePoint;
                    ROI2DSnake.this.translate(dx, dy);
                }
                else
                {
                    for (Anchor2D pt : controlPoints)
                    {
                        pt.mouseDrag(e, imagePoint, canvas);
                    }
                }
            }
            finally
            {
                ROI2DSnake.this.endUpdate();
            }
        }

        // ----------------------------------------------------------------------------

        private void mouseDragDilate(MouseEvent e, Point5D.Double imagePoint, IcyCanvas canvas)
        {
            if (!isActiveFor(canvas))
            {
                return;
            }
            ROI2DSnake.this.beginUpdate();
            try
            {
                if (isMouseInROI_ && pressedImagePoint_ != null && snakeMassCenter_ != null && !controlPoints.isEmpty())
                {
                    ((Canvas2D) canvas).getViewComponent().setCursor(dilateSnakeCursor_);
                    double distInit = (snakeMassCenter_.getX() - pressedImagePoint_.getX())
                            * (snakeMassCenter_.getX() - pressedImagePoint_.getX())
                            + (snakeMassCenter_.getY() - pressedImagePoint_.getY())
                                    * (snakeMassCenter_.getY() - pressedImagePoint_.getY());
                    double distCurrent = (snakeMassCenter_.getX() - imagePoint.getX())
                            * (snakeMassCenter_.getX() - imagePoint.getX())
                            + (snakeMassCenter_.getY() - imagePoint.getY())
                                    * (snakeMassCenter_.getY() - imagePoint.getY());
                    if (distInit == 0)
                    {
                        return;
                    }
                    double mouseDilationFactor = distCurrent / distInit;
                    Point2D firstPoint = ROI2DSnake.this.getControlPoint(0);
                    double snakeDilationFactor = ((snakeMassCenter_.getX() - firstPoint.getX())
                            * (snakeMassCenter_.getX() - firstPoint.getX())
                            + (snakeMassCenter_.getY() - firstPoint.getY())
                                    * (snakeMassCenter_.getY() - firstPoint.getY()))
                            / ((snakeMassCenter_.getX() - firstControlPointInitPosition_.getX())
                                    * (snakeMassCenter_.getX() - firstControlPointInitPosition_.getX())
                                    + (snakeMassCenter_.getY() - firstControlPointInitPosition_.getY())
                                            * (snakeMassCenter_.getY() - firstControlPointInitPosition_.getY()));
                    double dilationFactor = Math.sqrt(mouseDilationFactor / snakeDilationFactor);
                    for (Anchor2D anchor : controlPoints)
                    {
                        Point2D p = anchor.getPosition();
                        anchor.setPosition(
                                snakeMassCenter_.getX() + dilationFactor * (p.getX() - snakeMassCenter_.getX()),
                                snakeMassCenter_.getY() + dilationFactor * (p.getY() - snakeMassCenter_.getY()));
                    }
                    e.consume();
                }
            }
            finally
            {
                ROI2DSnake.this.endUpdate();
            }
        }

        // ----------------------------------------------------------------------------

        private void mouseDragRotate(MouseEvent e, Point5D.Double imagePoint, IcyCanvas canvas)
        {
            if (!isActiveFor(canvas))
            {
                return;
            }
            ROI2DSnake.this.beginUpdate();
            try
            {
                if (isMouseInROI_ && (pressedImagePoint_ != null) && (snakeMassCenter_ != null)
                        && !controlPoints.isEmpty())
                {
                    ((Canvas2D) canvas).getViewComponent().setCursor(rotateSnakeCursor_);
                    double distMouse = Math.sqrt((snakeMassCenter_.getX() - imagePoint.getX())
                            * (snakeMassCenter_.getX() - imagePoint.getX())
                            + (snakeMassCenter_.getY() - imagePoint.getY())
                                    * (snakeMassCenter_.getY() - imagePoint.getY()));
                    if (distMouse == 0)
                    {
                        return;
                    }
                    double cosAngMouse = (imagePoint.getX() - snakeMassCenter_.getX()) / distMouse;
                    double sinAngMouse = (imagePoint.getY() - snakeMassCenter_.getY()) / distMouse;
                    double distMouseInit = Math.sqrt((snakeMassCenter_.getX() - pressedImagePoint_.getX())
                            * (snakeMassCenter_.getX() - pressedImagePoint_.getX())
                            + (snakeMassCenter_.getY() - pressedImagePoint_.getY())
                                    * (snakeMassCenter_.getY() - pressedImagePoint_.getY()));
                    if (distMouseInit == 0)
                    {
                        return;
                    }
                    double cosAngMouseInit = (pressedImagePoint_.getX() - snakeMassCenter_.getX()) / distMouseInit;
                    double sinAngMouseInit = (pressedImagePoint_.getY() - snakeMassCenter_.getY()) / distMouseInit;
                    double cosDeltaAngMouse = cosAngMouse * cosAngMouseInit + sinAngMouseInit * sinAngMouse;
                    double sinDeltaAngMouse = sinAngMouse * cosAngMouseInit - sinAngMouseInit * cosAngMouse;
                    double distFirstPointInit = Math
                            .sqrt((snakeMassCenter_.getX() - firstControlPointInitPosition_.getX())
                                    * (snakeMassCenter_.getX() - firstControlPointInitPosition_.getX())
                                    + (snakeMassCenter_.getY() - firstControlPointInitPosition_.getY())
                                            * (snakeMassCenter_.getY() - firstControlPointInitPosition_.getY()));
                    if (distFirstPointInit == 0)
                    {
                        return;
                    }
                    double cosAngPointInit = (firstControlPointInitPosition_.getX() - snakeMassCenter_.getX())
                            / distFirstPointInit;
                    double sinAngPointInit = (firstControlPointInitPosition_.getY() - snakeMassCenter_.getY())
                            / distFirstPointInit;
                    Point2D firstPoint = controlPoints.get(0).getPosition();
                    double distFirstPoint = Math.sqrt((snakeMassCenter_.getX() - firstPoint.getX())
                            * (snakeMassCenter_.getX() - firstPoint.getX())
                            + (snakeMassCenter_.getY() - firstPoint.getY())
                                    * (snakeMassCenter_.getY() - firstPoint.getY()));
                    if (distFirstPoint == 0)
                    {
                        return;
                    }
                    double cosAngPoint = (firstPoint.getX() - snakeMassCenter_.getX()) / distFirstPoint;
                    double sinAngPoint = (firstPoint.getY() - snakeMassCenter_.getY()) / distFirstPoint;
                    double cosDeltaAngPoint = cosAngPoint * cosAngPointInit + sinAngPointInit * sinAngPoint;
                    double sinDeltaAngPoint = sinAngPoint * cosAngPointInit - sinAngPointInit * cosAngPoint;
                    double cosRotAngle = cosDeltaAngMouse * cosDeltaAngPoint + sinDeltaAngMouse * sinDeltaAngPoint;
                    double sinRotAngle = sinDeltaAngMouse * cosDeltaAngPoint - sinDeltaAngPoint * cosDeltaAngMouse;
                    for (Anchor2D anchor : controlPoints)
                    {
                        Point2D p = anchor.getPosition();
                        anchor.setPosition(
                                snakeMassCenter_.getX() + cosRotAngle * (p.getX() - snakeMassCenter_.getX())
                                        - sinRotAngle * (p.getY() - snakeMassCenter_.getY()),
                                snakeMassCenter_.getY() + sinRotAngle * (p.getX() - snakeMassCenter_.getX())
                                        + cosRotAngle * (p.getY() - snakeMassCenter_.getY()));
                    }
                    e.consume();
                }
            }
            finally
            {
                ROI2DSnake.this.endUpdate();
            }
        }
    }
}
