package plugins.ylemontag.complex;

import java.util.LinkedList;

/**
 * 
 * @author Yoann Le Montagner
 * 
 * Test function for complex conversion
 */
public class Test
{	
	/**
	 * Ground truth 
	 */
	public static class GT
	{
		public double a;
		public double b;
		public double r;
		public double theta;
		public GT(double real, double imaginary, double modulus, double argument)
		{
			a = real;
			b = imaginary;
			r = modulus;
			theta = argument;
		}
	}
	
	/**
	 * Interface for test functions
	 */
	public static interface TestFunction
	{
		public double out(GT gt);
		public double in1(GT gt);
		public double in2(GT gt);
		public double compute(double d1, double d2);
		public String getName();
	}
	
	/**
	 * Error tolerance
	 */
	private static final double ERROR_TOLERANCE = 1e-15;
	
	/**
	 * Entry point
	 */
	public static void main(String[] args)
	{
		// Argument
		runTest(new TestFunction() {
			public double out(GT gt) { return gt.theta; }
			public double in1(GT gt) { return gt.a    ; }
			public double in2(GT gt) { return gt.b    ; }
			public String getName() { return "argument"; }
			public double compute(double d1, double d2) { return ComplexUtil.argument(d1, d2); }
		});
		
		// Modulus
		runTest(new TestFunction() {
			public double out(GT gt) { return gt.r    ; }
			public double in1(GT gt) { return gt.a    ; }
			public double in2(GT gt) { return gt.b    ; }
			public String getName() { return "modulus"; }
			public double compute(double d1, double d2) { return ComplexUtil.modulus(d1, d2); }
		});
		
		// Real part
		runTest(new TestFunction() {
			public double out(GT gt) { return gt.a    ; }
			public double in1(GT gt) { return gt.r    ; }
			public double in2(GT gt) { return gt.theta; }
			public String getName() { return "real"; }
			public double compute(double d1, double d2) { return ComplexUtil.real(d1, d2); }
		});
		
		// Imaginary part
		runTest(new TestFunction() {
			public double out(GT gt) { return gt.b    ; }
			public double in1(GT gt) { return gt.r    ; }
			public double in2(GT gt) { return gt.theta; }
			public String getName() { return "imaginary"; }
			public double compute(double d1, double d2) { return ComplexUtil.imaginary(d1, d2); }
		});
	}
	
	/**
	 * Run a bench
	 */
	private static void runTest(TestFunction fun)
	{
		// Values on the unit circle
		LinkedList<GT> unit = new LinkedList<Test.GT>();
		unit.add(new GT(                1.0,                0.0, 1,              0.0));
		unit.add(new GT( Math.sqrt(3.0)/2.0,            1.0/2.0, 1,      Math.PI/6.0));
		unit.add(new GT( Math.sqrt(2.0)/2.0, Math.sqrt(2.0)/2.0, 1,      Math.PI/4.0));
		unit.add(new GT(            1.0/2.0, Math.sqrt(3.0)/2.0, 1,      Math.PI/3.0));
		unit.add(new GT(                0.0,                1.0, 1,      Math.PI/2.0));
		unit.add(new GT(           -1.0/2.0, Math.sqrt(3.0)/2.0, 1,  2.0*Math.PI/3.0));
		unit.add(new GT(-Math.sqrt(2.0)/2.0, Math.sqrt(2.0)/2.0, 1,  3.0*Math.PI/4.0));
		unit.add(new GT(-Math.sqrt(3.0)/2.0,            1.0/2.0, 1,  5.0*Math.PI/6.0));
		unit.add(new GT(               -1.0,                0.0, 1,      Math.PI    ));
		unit.add(new GT(-Math.sqrt(3.0)/2.0,           -1.0/2.0, 1, -5.0*Math.PI/6.0));
		unit.add(new GT(-Math.sqrt(2.0)/2.0,-Math.sqrt(2.0)/2.0, 1, -3.0*Math.PI/4.0));
		unit.add(new GT(           -1.0/2.0,-Math.sqrt(3.0)/2.0, 1, -2.0*Math.PI/3.0));
		unit.add(new GT(                0.0,               -1.0, 1,     -Math.PI/2.0));
		unit.add(new GT(            1.0/2.0,-Math.sqrt(3.0)/2.0, 1,     -Math.PI/3.0));
		unit.add(new GT( Math.sqrt(2.0)/2.0,-Math.sqrt(2.0)/2.0, 1,     -Math.PI/4.0));
		unit.add(new GT( Math.sqrt(3.0)/2.0,           -1.0/2.0, 1,     -Math.PI/6.0));
		
		// Zero point
		GT zero = new GT(0, 0, 0, 0);
		
		// Infinite directions
		double Inf = Double.POSITIVE_INFINITY;
		LinkedList<GT> infinity = new LinkedList<GT>();
		infinity.add(new GT( Inf,  0.0, Inf,              0.0));
		infinity.add(new GT( Inf,  Inf, Inf,      Math.PI/4.0));
		infinity.add(new GT( 0.0,  Inf, Inf,      Math.PI/2.0));
		infinity.add(new GT(-Inf,  Inf, Inf,  3.0*Math.PI/4.0));
		infinity.add(new GT(-Inf,  0.0, Inf,      Math.PI    ));
		infinity.add(new GT(-Inf, -Inf, Inf, -3.0*Math.PI/4.0));
		infinity.add(new GT( 0.0, -Inf, Inf,     -Math.PI/2.0));
		infinity.add(new GT( Inf, -Inf, Inf,     -Math.PI/4.0));
		
		// Actual computations
		System.out.println("===== TESTING FUNCTION " + fun.getName() + " =====");
		System.out.println("  Unit circle:");
		for(GT gt : unit) {
			testReport(fun, gt);
		}
		System.out.println("  Zero:");
		testReport(fun, zero);
		System.out.println("  Infinity:");
		for(GT gt : infinity) {
			testReport(fun, gt);
		}
	}
	
	/**
	 * Test report
	 */
	private static void testReport(TestFunction fun, GT gt)
	{
		String testName  = fun.getName() + "(" + fun.in1(gt) + "," + fun.in2(gt) + ")";
		double thValue   = fun.out(gt);
		double calcValue = fun.compute(fun.in1(gt), fun.in2(gt));
		if(equals(thValue, calcValue)) {
			System.out.println("    [ OK ] " + testName);
		}
		else {
			System.out.println("    [Fail] " + testName + " (th=" + thValue + " calc=" + calcValue + ")");
		}
	}
	
	/**
	 * Comparison with error thresholding
	 */
	private static boolean equals(double d1, double d2)
	{
		if(Double.isInfinite(d1)) {
			return Double.isInfinite(d2) && d1*d2>0;
		}
		else if(Double.isNaN(d1)) {
			return Double.isNaN(d2);
		}
		else {
			return Math.abs(d1-d2) < ERROR_TOLERANCE;
		}
	}
}
