/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Zsuzsanna Puspoki (zsuzsanna.puspoki@epfl.ch)
 ******************************************************************************/
package plugins.big.steerablej.process;

import icy.image.IcyBufferedImage;
import icy.image.IcyBufferedImageUtil;
import icy.sequence.Sequence;
import icy.sequence.SequenceUtil;
import icy.type.DataType;

/**
 * Class that encapsulated conversion tools between <code>Sequence</code>
 * objects and rasterized image stacks.
 * 
 * @version April 22, 2013
 * 
 * @author Zsuzsanna Puspoki (zsuzsanna.puspoki@epfl.ch)
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 */
public class Image2ArrayConverter {

	// ============================================================================
	// PUBLIC METHODS

	/** Transforms a Sequence object to a rasterized <code>double</code> array. */
	public static double[] seqToDoubleArray(Sequence sequence, int channel) {

		int sizeX = sequence.getSizeX();
		int sizeY = sequence.getSizeY();
		int sizeT = sequence.getSizeT();

		double[] pixelArray = new double[sizeX * sizeY * sizeT];
		double[] dataBuffer = new double[sizeX * sizeY];

		Sequence singleChannelSequence = SequenceUtil.extractChannel(sequence,
				channel);
		for (int t = 0; t < sizeT; t++) {
			IcyBufferedImage inputImage = singleChannelSequence.getImage(t, 0);
			IcyBufferedImage inputCopy = IcyBufferedImageUtil.convertToType(
					inputImage, DataType.DOUBLE, false);
			dataBuffer = inputCopy.getDataXYAsDouble(0);
			for (int x = 0; x < sizeX; ++x) {
				for (int y = 0; y < sizeY; ++y) {
					pixelArray[x + sizeX * y + t * sizeX * sizeY] = dataBuffer[x
							+ sizeX * y];
				}
			}
		}
		return pixelArray;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Transforms a rasterized <code>double</code> array to a
	 * <code>Sequence</code> object.
	 */
	public static Sequence doubleArrToSeq(double[] array, int nx, int ny, int nz) {

		Sequence seq = new Sequence();
		for (int z = 0; z < nz; z++) {
			IcyBufferedImage outImage = new IcyBufferedImage(nx, ny, 1,
					DataType.DOUBLE);
			double[] dataBuffer = outImage.getDataXYAsDouble(0);
			for (int y = 0; y < ny; y++) {
				for (int x = 0; x < nx; x++) {
					dataBuffer[x + nx * y] = array[x + nx * y + z * nx * ny];
				}
			}
			outImage.dataChanged();
			seq.addImage(z, outImage);
		}
		return seq;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Encodes three rasterized <code>int</code> arrays to an RGB
	 * <code>Sequence</code> object.
	 */
	public static Sequence arraysToRGBSeq(int[] R, int[] G, int[] B, int nx,
			int ny, int nz) {

		Sequence seq = new Sequence();
		for (int z = 0; z < nz; z++) {
			IcyBufferedImage outImage = new IcyBufferedImage(nx, ny, 3,
					DataType.INT);
			int[] dataBufferR = outImage.getDataXYAsInt(0);
			int[] dataBufferG = outImage.getDataXYAsInt(1);
			int[] dataBufferB = outImage.getDataXYAsInt(2);

			for (int y = 0; y < ny; y++) {
				for (int x = 0; x < nx; x++) {
					dataBufferR[x + nx * y] = R[x + nx * y + z * nx * ny];
					dataBufferG[x + nx * y] = G[x + nx * y + z * nx * ny];
					dataBufferB[x + nx * y] = B[x + nx * y + z * nx * ny];
				}
			}
			outImage.dataChanged();
			seq.addImage(z, outImage);
		}
		return seq;
	}
}
