package plugins.ylemontag.sequencecomparator.gui;

import icy.gui.dialog.MessageDialog;
import icy.gui.frame.IcyFrame;
import icy.gui.main.MainAdapter;
import icy.gui.main.MainEvent;
import icy.image.lut.LUT;
import icy.main.Icy;
import icy.plugin.abstract_.PluginActionable;
import icy.preferences.XMLPreferences;
import icy.sequence.Sequence;
import icy.system.thread.ThreadUtil;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import plugins.adufour.ezplug.EzComponent;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzVarDouble;
import plugins.ylemontag.sequencecomparator.ComparisonPool;
import plugins.ylemontag.sequencecomparator.ComparisonPool.StateChangeListener;
import plugins.ylemontag.sequencecomparator.ComparisonState;
import plugins.ylemontag.sequencecomparator.ErrorMeasure;
import plugins.ylemontag.sequencecomparator.GlobalComparator;
import plugins.ylemontag.sequencecomparator.GlobalComparatorFactory;
import plugins.ylemontag.sequencecomparator.GlobalComparisonPool;
import plugins.ylemontag.sequencecomparator.LocalComparator;
import plugins.ylemontag.sequencecomparator.LocalComparatorFactory;
import plugins.ylemontag.sequencecomparator.LocalComparisonPool;
import plugins.ylemontag.sequencecomparator.ParameterProvider;
import plugins.ylemontag.sequencecomparator.comparators.GlobalPSNR;
import plugins.ylemontag.sequencecomparator.gui.ErrorMapChooserComponent.ErrorMapLUTListener;
import plugins.ylemontag.sequencecomparator.gui.OptionButton.OptionButtonListener;
import plugins.ylemontag.sequencecomparator.gui.SampleSequenceComponent.ComparisonListener;
import plugins.ylemontag.ssim.IllegalSSIMParameterException;
import plugins.ylemontag.ssim.SSIMCalculator;
import plugins.ylemontag.ssim.SSIMParameterComponent;

/**
 * 
 * @author Yoann Le Montagner
 * 
 * Plugin to visualize differences between multi-channel 2D+T or 3D+T sequences
 */
public class SequenceComparator extends PluginActionable
{
	private GlobalComparisonPool _globalComparisonPool;
	private LocalComparisonPool _localComparisonPool;
	private double _psnrDynamicRange;
	private SSIMCalculator _ssimCalculator;
	private GUIParameterProvider _parameterProvider;
	private IcyFrame _frame;
	private ReferenceSequenceComponent _refComponent;
	private JCheckBox _extendZComponent;
	private JCheckBox _extendTComponent;
	private SampleSequenceComponent _sampleComponent;
	private GlobalComparatorComponent _globalComparatorComponent;
	private LocalComparatorComponent _localComparatorComponent;
	private JButton _enableSelectionButton;
	private JButton _disableSelectionButton;
	private ErrorMapChooserComponent _errorMapLUTButton;
	private ErrorMeasureComponent _errorMeasureComponent;
	private PSNRParameterFrame _psnrFrame;
	private SSIMParameterFrame _ssimFrame;
	
	@Override
	public void run()
	{
		// Comparison pools
		_globalComparisonPool = new GlobalComparisonPool();
		_localComparisonPool  = new LocalComparisonPool ();
		_psnrDynamicRange     = GlobalPSNR.DEFAULT_DYNAMIC_RANGE;
		_ssimCalculator       = new SSIMCalculator      ();
		_parameterProvider    = new GUIParameterProvider();
		
		// GUI
		_frame = new IcyFrame("Sequence comparator", true, true, true, true);
		_frame.setSize        (800, 700);
		_frame.setSizeExternal(800, 700);
		MainPanel mainPanel = new MainPanel(_globalComparisonPool, _localComparisonPool);
		_frame.add(mainPanel);
		mainPanel.getSplitPane().setDividerLocation(120);
		addIcyFrame(_frame);
		_frame.center();
		_frame.setVisible(true);
		_frame.requestFocus();
		_refComponent = mainPanel.getRefComponent();
		_extendZComponent = mainPanel.getExtendZComponent();
		_extendTComponent = mainPanel.getExtendTComponent();
		_sampleComponent = mainPanel.getSampleComponent();
		_globalComparatorComponent = mainPanel.getGlobalComparatorComponent();
		_localComparatorComponent = mainPanel.getLocalComparatorComponent();
		_enableSelectionButton = mainPanel.getEnableSelectionButton();
		_disableSelectionButton = mainPanel.getDisableSelectionButton();
		_errorMapLUTButton = mainPanel.getErrorMapLUTButton();
		_errorMeasureComponent = mainPanel.getErrorMeasureComponent();
		retrievePersistentGlobalComparator();
		retrievePersistentLocalComparator();
		retrievePersistentPSNRParameters();
		try {
			SSIMParameterComponent ssimComponent = makeSSIMComponent();
			_ssimCalculator.cloneParameters(ssimComponent.createCalculator());
		}
		catch(IllegalSSIMParameterException err) {
			// This exception can occur if the persistent SSIM parameters are corrupted
			// in the XML preference file. In that case, the IllegalSSIMParameterException
			// is caught, and the default SSIM parameters are used.
		}
		
		// Main listeners
		_refComponent.addActionListener(new ActionListener()
		{
			@Override
			public void actionPerformed(ActionEvent e) {
				onReferenceChanged();
			}
		});
		_extendZComponent.addActionListener(new ActionListener()
		{
			@Override
			public void actionPerformed(ActionEvent e) {
				onReferenceChanged();
			}
		});
		_extendTComponent.addActionListener(new ActionListener()
		{
			@Override
			public void actionPerformed(ActionEvent e) {
				onReferenceChanged();
			}
		});
		_globalComparatorComponent.addActionListener(new ActionListener()
		{
			@Override
			public void actionPerformed(ActionEvent e) {
				onGlobalComparatorChanged();
			}
		});
		_localComparatorComponent.addActionListener(new ActionListener()
		{
			@Override
			public void actionPerformed(ActionEvent e) {
				onLocalComparatorChanged();
			}
		});
		_sampleComponent.addGlobalComparisonListener(new ComparisonListener()
		{
			@Override
			public void sequenceSelected(Sequence seq) {
				onComparisonRequested(_globalComparisonPool, seq);
			}
		});
		_sampleComponent.addLocalComparisonListener(new ComparisonListener()
		{
			@Override
			public void sequenceSelected(Sequence seq) {
				onComparisonRequested(_localComparisonPool, seq);
			}
		});
		_sampleComponent.getSelectionModel().addListSelectionListener(new ListSelectionListener()
		{
			@Override
			public void valueChanged(ListSelectionEvent e) {
				refreshGlobalComparisonButtonStates();
			}
		});
		Icy.getMainInterface().addListener(new MainAdapter()
		{
			@Override
			public void sequenceClosed(MainEvent event) {
				Sequence seq = (Sequence)event.getSource();
				_globalComparisonPool.stopComparison(seq);
				_localComparisonPool .stopComparison(seq);
				Sequence masterSeq = _localComparisonPool.getErrorMapSource(seq);
				if(masterSeq!=null) {
					_localComparisonPool.stopComparison(masterSeq);
				}
			}
		});
		_globalComparisonPool.addStateChangeListener(new StateChangeListener()
		{
			@Override
			public void stateChanged(Sequence seq, ComparisonState state, Object result) {
				onGlobalComparisonStateChanged(seq, state, (ErrorMeasure)result);
			}
		});
		_localComparisonPool.addStateChangeListener(new StateChangeListener()
		{
			@Override
			public void stateChanged(Sequence seq, ComparisonState state, Object result) {
				onLocalComparisonStateChanged(seq, state, (Sequence)result);
			}
		});
		
		// Action performed on a click on the refresh LUT button
		mainPanel.getRefreshLUTButton().addActionListener(new ActionListener()
		{
			@Override
			public void actionPerformed(ActionEvent event) {
				synchronizeAllSequences(_globalComparisonPool);
				synchronizeAllSequences(_localComparisonPool );
			}
		});
		
		// Action performed on a click on the synchronize error map LUT button
		_errorMapLUTButton.addErrorMapLUTListener(new ErrorMapLUTListener()
		{
			@Override
			public void eventTriggered(Sequence seq) {
				synchronizeErrorMapLUTs(seq);
			}
		});
		
		// Action performed when the user clicks on the activate selection button
		_enableSelectionButton.addActionListener(new ActionListener()
		{
			@Override
			public void actionPerformed(ActionEvent e) {
				int nbRows = _sampleComponent.getRowCount();
				for(int k=0; k<nbRows; ++k) {
					if(!_sampleComponent.isRowSelected(k)) {
						continue;
					}
					Sequence seq = _sampleComponent.getSequenceAtRow(k);
					if(_globalComparisonPool.getState(seq)==ComparisonState.NOT_WATCHED) {
						_globalComparisonPool.startComparisonAsync(seq);
					}
				}
			}
		});
		
		// Action performed when the user clicks on the activate selection button
		_disableSelectionButton.addActionListener(new ActionListener()
		{
			@Override
			public void actionPerformed(ActionEvent e) {
				int nbRows = _sampleComponent.getRowCount();
				for(int k=0; k<nbRows; ++k) {
					if(!_sampleComponent.isRowSelected(k)) {
						continue;
					}
					Sequence seq = _sampleComponent.getSequenceAtRow(k);
					_globalComparisonPool.stopComparison(seq);
				}
			}
		});
		
		// Action performed when the user clicks on the SSIM options/info button
		mainPanel.getOptionButton().addListener(new OptionButtonListener()
		{
			@Override
			public void onPSNRClicked()
			{
				onEditPSNRParametersClicked();
			}
			
			@Override
			public void onSSIMClicked()
			{
				onEditSSIMParametersClicked();
			}
		});
		
		// Feeding GUI
		onGlobalComparatorChanged();
		onLocalComparatorChanged();
		onReferenceChanged();
		refreshGlobalComparisonButtonStates();
		refreshErrorMapLUTButtonState();
	}
	
	/**
	 * Action performed when the user clicks on the edit PSNR parameters button
	 */
	private void onEditPSNRParametersClicked()
	{
		if(_psnrFrame==null || _psnrFrame.getUI()==null) {
			_psnrFrame = new PSNRParameterFrame();
			_psnrFrame.compute();
		}
	}
	
	/**
	 * Action performed when the user clicks on the edit SSIM parameters button
	 */
	private void onEditSSIMParametersClicked()
	{
		if(_ssimFrame==null || _ssimFrame.getUI()==null) {
			_ssimFrame = new SSIMParameterFrame();
			_ssimFrame.compute();
		}
	}
	
	/**
	 * Action performed on reference video switch
	 */
	private void onReferenceChanged()
	{
		Sequence newRef = _refComponent.getSelectedObject();
		boolean canExtendZ = newRef!=null && newRef.getSizeZ()==1;
		boolean canExtendT = newRef!=null && newRef.getSizeT()==1;
		_extendZComponent.setEnabled(canExtendZ);
		_extendTComponent.setEnabled(canExtendT);
		boolean doExtendZ = canExtendZ && _extendZComponent.isSelected();
		boolean doExtendT = canExtendT && _extendTComponent.isSelected();
		_globalComparisonPool.setReference(newRef, doExtendZ, doExtendT);
		_localComparisonPool .setReference(newRef, doExtendZ, doExtendT);
		_sampleComponent  .changeReference(newRef, doExtendZ, doExtendT);
	}
	
	/**
	 * Action performed on global comparator switch
	 */
	private void onGlobalComparatorChanged()
	{
		updatePersistentGlobalComparator();
		GlobalComparatorFactory factory = _globalComparatorComponent.getSelectedObject();
		GlobalComparator newGlobalComparator = factory.createInstance(_parameterProvider);
		_globalComparisonPool.setComparator(newGlobalComparator);
		_errorMeasureComponent.setDistanceName(_globalComparisonPool.getDistanceName());
	}
	
	/**
	 * Action performed on local comparator switch
	 */
	private void onLocalComparatorChanged()
	{
		updatePersistentLocalComparator();
		LocalComparatorFactory factory = _localComparatorComponent.getSelectedObject();
		LocalComparator newLocalComparator = factory.createInstance(_parameterProvider);
		_localComparisonPool.setComparator(newLocalComparator);
	}
	
	/**
	 * Action performed when the user clicks on a global comparison button
	 */
	private void onComparisonRequested(ComparisonPool<?,?> comparisonPool, Sequence seq)
	{
		if(comparisonPool.getState(seq)==ComparisonState.NOT_WATCHED)
			comparisonPool.startComparisonAsync(seq);
		else
			comparisonPool.stopComparison(seq);
	}
	
	/**
	 * Action performed when the global comparison state of a sequence has changed
	 */
	private void onGlobalComparisonStateChanged(Sequence seq, ComparisonState state, ErrorMeasure result)
	{
		if(state==ComparisonState.READY) {
			result.label = seq.getName(); // Simplify the labeling used in the graph
			_errorMeasureComponent.addMeasure(result);
			synchronizeSequence(_globalComparisonPool.getReference(), seq);
		}
		else if(state==ComparisonState.NOT_WATCHED) {
			if(result!=null) {
				_errorMeasureComponent.removeMeasure(result);
			}
		}
	}
	
	/**
	 * Action performed when the local comparison state of a sequence has changed
	 */
	private void onLocalComparisonStateChanged(Sequence seq, ComparisonState state, Sequence result)
	{
		refreshErrorMapLUTButtonState();
		if(state==ComparisonState.READY) {
			synchronizeSequence(_localComparisonPool.getReference(), seq);
			synchronizeErrorMap(seq, result);
		}
		else if(state==ComparisonState.NOT_WATCHED) {
			if(result!=null) {
				result.close();
			}
		}
	}
	
	/**
	 * Synchronize the viewer and the LUT of two sequences
	 */
	private void synchronizeSequence(Sequence master, Sequence slave)
	{
		synchronizeLUT    (master, slave);
		synchronizeViewers(master, slave);
	}
	
	/**
	 * Synchronize all the compared sequences within a given comparison pool
	 */
	private void synchronizeAllSequences(ComparisonPool<?,?> comparisonPool)
	{
		Sequence master = comparisonPool.getReference();
		if(master==null) {
			return;
		}
		for(Sequence s : comparisonPool.getComparedSequences()) {
			synchronizeSequence(master, s);
		}
	}
	
	/**
	 * Synchronize an error map to its corresponding sequence
	 */
	private void synchronizeErrorMap(final Sequence seq, final Sequence errorMap)
	{
		if(errorMap.getFirstViewer()==null) {
			Icy.getMainInterface().addSequence(errorMap);
		}
		ThreadUtil.invokeLater(new Runnable()
		{	
			@Override
			public void run() {
				synchronizeViewers(seq, errorMap);
			}
		}, true);
	}
	
	/**
	 * Use the same look-up table for the tested sequence than the one used for
	 * the reference
	 */
	public static void synchronizeLUT(Sequence master, Sequence slave)
	{
		LUT refLUT = master.getFirstViewer().getLut();
		slave.getFirstViewer().getLut().copyScalers(refLUT);
	}
	
	/**
	 * Synchronize the viewers
	 */
	public static void synchronizeViewers(Sequence master, Sequence slave)
	{
		master.getFirstViewer().getCanvas().setSyncId(0);
		slave .getFirstViewer().getCanvas().setSyncId(1);
		master.getFirstViewer().getCanvas().setSyncId(1);
	}
	
	/**
	 * Synchronize all the error map LUTs belonging to the current local comparison
	 * pool to the given sequence's error map
	 */
	private void synchronizeErrorMapLUTs(Sequence seq)
	{
		Sequence refErrorMap = _localComparisonPool.getResult(seq);
		if(refErrorMap==null) {
			return;
		}
		for(Sequence s : _localComparisonPool.getComparedSequences()) {
			if(s==seq) {
				continue;
			}
			Sequence currentErrorMap = _localComparisonPool.getResult(s);
			synchronizeLUT(refErrorMap, currentErrorMap);
		}
	}
	
	/**
	 * Change the activation state of enable/disable error measure buttons according
	 * to the selected sample sequence list
	 */
	private void refreshGlobalComparisonButtonStates()
	{
		boolean selectionNotEmpty = !_sampleComponent.getSelectionModel().isSelectionEmpty();
		_enableSelectionButton.setEnabled(selectionNotEmpty);
		_disableSelectionButton.setEnabled(selectionNotEmpty);
	}
	
	/**
	 * Change the activation state of the synchronize error map LUTs button
	 * according to the number of opened error maps
	 */
	private void refreshErrorMapLUTButtonState()
	{
		_errorMapLUTButton.setEnabled(_localComparisonPool.getComparedSequences().size()>=2);
	}
	
	/**
	 * Retrieve the global comparator previously chosen
	 */
	private void retrievePersistentGlobalComparator()
	{
		retrievePersistentComparator(_globalComparatorComponent, "GlobalComparator");
	}
	
	/**
	 * Retrieve the local comparator previously chosen
	 */
	private void retrievePersistentLocalComparator()
	{
		retrievePersistentComparator(_localComparatorComponent, "LocalComparator");
	}
	
	/**
	 * Base routine for retrieving persistent comparators 
	 */
	private void retrievePersistentComparator(ComparatorComponent<?> component, String XMLKey)
	{
		XMLPreferences prefs = getPreferences(XMLKey);
		String newName = prefs.get("Name", component.getSelectedObject().getDistanceName());
		int nbRows = component.getItemCount();
		for(int k=0; k<nbRows; ++k) {
			if(newName.equals(component.getObjectAt(k).getDistanceName())) {
				component.setSelectedIndex(k);
				return;
			}
		}
	}
	
	/**
	 * Updates the preferred global comparator
	 */
	private void updatePersistentGlobalComparator()
	{
		updatePersistentComparator(_globalComparatorComponent, "GlobalComparator");
	}
	
	/**
	 * Updates the preferred local comparator
	 */
	private void updatePersistentLocalComparator()
	{
		updatePersistentComparator(_localComparatorComponent, "LocalComparator");
	}
	
	/**
	 * Base routine for updating persistent comparators
	 */
	private void updatePersistentComparator(ComparatorComponent<?> component, String XMLKey)
	{
		XMLPreferences prefs = getPreferences(XMLKey);
		prefs.put("Name", component.getSelectedObject().getDistanceName());
	}
	
	/**
	 * Loads the preferred PSNR parameters
	 */
	private void retrievePersistentPSNRParameters()
	{
		XMLPreferences prefs = getPreferences("PSNROptions");
		_psnrDynamicRange = prefs.getDouble("L", GlobalPSNR.DEFAULT_DYNAMIC_RANGE);
		if(_psnrDynamicRange<=0) {
			_psnrDynamicRange = GlobalPSNR.DEFAULT_DYNAMIC_RANGE;
		}
	}
	
	/**
	 * Updates the preferred PSNR parameters
	 */
	private void updatePersistentPSNRParameters()
	{
		XMLPreferences prefs = getPreferences("PSNROptions");
		prefs.putDouble("L", _psnrDynamicRange);
	}
	
	/**
	 * Singleton pattern for the SSIM component
	 */
	private SSIMParameterComponent makeSSIMComponent()
	{
		SSIMParameterComponent retval = new SSIMParameterComponent();
		retval.load(getSSIMPersistentNode());
		return retval;
	}
	
	/**
	 * Node used to save the preferences relative to SSIM
	 */
	private XMLPreferences getSSIMPersistentNode()
	{
		return getPreferences("SSIMOptions");
	}
	
	/**
	 * Implements the parameter provider interface
	 */
	private class GUIParameterProvider implements ParameterProvider
	{
		@Override
		public double getPSNRDynamicRange()
		{
			return _psnrDynamicRange;
		}
		
		@Override
		public SSIMCalculator getSSIMCalculator()
		{
			return _ssimCalculator;
		}
	}
	
	
	
	/**
	 * Frame for changing PSNR options
	 */
	private class PSNRParameterFrame extends EzPlug
	{
		private EzVarDouble _dynamicRange;
		
		@Override
		public String getName()
		{
			return "Sequence comparator - PSNR options";
		}
		
		@Override
		protected void initialize()
		{
			_dynamicRange = new EzVarDouble("Dynamic range");
			_dynamicRange.setValue(_psnrDynamicRange);
			_dynamicRange.setToolTipText(
				"Dynamic range of the pixel values of the sequences " +
				"(for example: might be 255 if the sequences are 8 bit encoded)"
			);
			addEzComponent(_dynamicRange);
			getUI().setRunButtonText("Update");
		}

		@Override
		protected void execute()
		{
			ThreadUtil.invokeLater(new Runnable()
			{	
				@Override
				public void run() {
					double newValue = _dynamicRange.getValue();
					if(newValue>0) {
						_psnrDynamicRange = newValue;
						updatePersistentPSNRParameters();
						if(_globalComparatorComponent.getSelectedObject()==GlobalComparatorFactory.PSNR) {
							onGlobalComparatorChanged();
						}
					}
					else {
						MessageDialog.showDialog(
							"Wrong PSNR parameters",
							"The dynamic range must be positive (>0).",
							MessageDialog.ERROR_MESSAGE
						);
					}
				}
			});
		}
		
		@Override
		public void clean() {}
	}
	
	
	
	/**
	 * Frame for changing SSIM options
	 */
	private class SSIMParameterFrame extends EzPlug
	{	
		private SSIMParameterComponent _ssimComponent;
		
		@Override
		public String getName()
		{
			return "Sequence comparator - SSIM options";
		}
		
		@Override
		protected void initialize()
		{
			_ssimComponent = makeSSIMComponent();
			for(EzComponent c : _ssimComponent.getComponents()) {
				addEzComponent(c);
			}
			getUI().setRunButtonText("Update");
		}

		@Override
		protected void execute()
		{
			ThreadUtil.invokeLater(new Runnable()
			{	
				@Override
				public void run()
				{
					try {
						_ssimCalculator.cloneParameters(_ssimComponent.createCalculator());
						_ssimComponent.save(getSSIMPersistentNode());
						if(_globalComparatorComponent.getSelectedObject()==GlobalComparatorFactory.SSIM) {
							onGlobalComparatorChanged();
						}
						if(_localComparatorComponent.getSelectedObject()==LocalComparatorFactory.SSIM) {
							onLocalComparatorChanged();
						}
					}
					catch(IllegalSSIMParameterException err) {
						MessageDialog.showDialog("Bad SSIM parameters", err.getMessage(), MessageDialog.ERROR_MESSAGE);
					}
				}
			});
		}
		
		@Override
		public void clean() {}
	}
}
