/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.icy.ellipsoid;

import icy.canvas.IcyCanvas;
import icy.canvas.IcyCanvas2D;
import icy.roi.ROI;
import icy.roi.ROI2D;
import icy.sequence.Sequence;

import java.awt.BasicStroke;
import java.awt.Graphics2D;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Rectangle2D;

/**
 * Class describing a ROI for ellipses.
 * 
 * @version May 3, 2014
 * 
 * @author Julien Jacquemot
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 */
public class EllipsoidROI2D extends ROI2D {

	private final Ellipsoid2D ellipse_;

	// ============================================================================
	// PUBLIC METHODS

	/**
	 * Default constructor.
	 * 
	 * @param ellipse
	 *            Ellipse descriptor associated to this ROI
	 */
	public EllipsoidROI2D(Ellipsoid2D ellipse) {
		ellipse_ = ellipse;
		setName(ellipse_.toString());
	}

	/** Returns the descriptor associated to this ROI. */
	public Ellipsoid2D getDescriptor() {
		return ellipse_;
	}

	// ============================================================================
	// ROI3D METHODS

	@Override
	public Rectangle2D computeBounds2D() {
		double c = Math.cos(ellipse_.alpha);
		double s = Math.sin(ellipse_.alpha);

		double A1 = ellipse_.a * c, B1 = -ellipse_.b * s;
		double A2 = ellipse_.a * s, B2 = ellipse_.b * c;

		double beta1 = Math.atan2(-A1, B1);
		double beta2 = Math.atan2(-A2, B2);

		double x1 = ellipse_.x0 + A1 * Math.cos(beta1) - B1 * Math.sin(beta1);
		double x2 = ellipse_.x0 - A1 * Math.cos(beta1) + B1 * Math.sin(beta1);

		double y1 = ellipse_.y0 + A2 * Math.cos(beta2) - B2 * Math.sin(beta2);
		double y2 = ellipse_.y0 - A2 * Math.cos(beta2) + B2 * Math.sin(beta2);

		return new Rectangle2D.Double(Math.min(x1, x2), Math.min(y2, y1),
				Math.abs(x2 - x1), Math.abs(y2 - y1));
	}

	@Override
	public boolean contains(double x, double y) {
		return ellipse_.contains(x, y);
	}

	@Override
	public boolean contains(double x, double y, double w, double h) {
		return ellipse_.contains(x, y) && ellipse_.contains(x + w, y + h);
	}

	@Override
	protected ROIPainter createPainter() {
		return new EllipsoidPainter();
	}

	@Override
	public ROI getCopy() {
		return new EllipsoidROI2D((Ellipsoid2D) ellipse_.clone());
	}

	@Override
	public boolean hasSelectedPoint() {
		return false;
	}

	@Override
	public double getArea() {
		return ellipse_.getVolume();
	}

	@Override
	public double getPerimeter() {
		return ellipse_.getPerimeter();
	}

	@Override
	public int getT() {
		return ellipse_.getT();
	}

	@Override
	public boolean intersects(double x, double y, double w, double h) {
		return getBounds2D().intersects(x, y, w, h);
	}

	@Override
	public boolean isOver(IcyCanvas canvas, double x, double y) {
		return canvas.getPositionT() == ellipse_.getT()
				&& ellipse_.isOver(x, y);
	}

	@Override
	public boolean isOverEdge(IcyCanvas canvas, double x, double y) {
		return false;
	}

	@Override
	public void translate(double dx, double dy) {
		ellipse_.x0 += dx;
		ellipse_.y0 += dy;
	}

	// ============================================================================
	// PAINTER

	protected class EllipsoidPainter extends ROIPainter {
		@Override
		public void paint(Graphics2D g2, Sequence sequence, IcyCanvas canvas) {
			if (ellipse_ == null || ellipse_.getT() != canvas.getPositionT()) {
				return;
			}

			if (canvas instanceof IcyCanvas2D) {
				g2.setStroke(new BasicStroke((float) stroke));
				g2.setColor(getColor());

				Shape baseShape = new Ellipse2D.Double(
						ellipse_.x0 - ellipse_.a, ellipse_.y0 - ellipse_.b,
						2 * ellipse_.a, 2 * ellipse_.b);
				Shape rotatedShape = AffineTransform.getRotateInstance(
						ellipse_.alpha, ellipse_.x0, ellipse_.y0)
						.createTransformedShape(baseShape);
				g2.draw(rotatedShape);
			}
		}
	}
}
