/*
 * Copyright 2012 Institut Pasteur.
 * 
 * This file is part of ICY.
 * 
 * ICY is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * ICY is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with ICY. If not, see <http://www.gnu.org/licenses/>.
 */

package plugins.tlecomte.contourPlot;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.font.FontRenderContext;
import java.awt.font.TextLayout;
import java.awt.geom.AffineTransform;
import java.awt.geom.Path2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;

import icy.canvas.IcyCanvas;
import icy.gui.viewer.Viewer;
import icy.painter.Painter;
import icy.sequence.Sequence;

/**
 * 
 * @author Timothee Lecomte
 *
 * This plugin implements a painter that displays contour lines.
 *
 * TODO:
 * 
 */

public class ContourPainter implements Painter {
	ArrayList<ArrayList<Contour>> globalContourList; // one list of contours per time point
	boolean drawLabels = false;
	int fontsize = 5;
	int nDecimals = 0;
	double linewidth = 1.0;
	
	// one list of Path2D per time point
//	ArrayList<ArrayList<Path2D>> globalPathList;
	
	ContourPainter(ArrayList<ArrayList<Contour>> globalContourList) {
		this.globalContourList = globalContourList;

//		globalPathList = new ArrayList<ArrayList<Path2D>>();
//		
//		// populate the path list
//		for (ArrayList<Contour> contourList: globalContourList)
//		{
//			ArrayList<Path2D> pathList = new ArrayList<Path2D>();
//			
//			// loop over the contour lines
//			for (Contour contour : contourList) {     
//		         Path2D.Double path = new Path2D.Double();
//		         path.moveTo(contour.x[0] + 0.5, contour.y[0] + 0.5);
//		         for (int i=1; i<contour.x.length; i++) {
//		        	 path.lineTo(contour.x[i] + 0.5, contour.y[i] + 0.5);
//		         }
//		         pathList.add(path);
//			}
//			globalPathList.add(pathList);
//		}
	}
	
	void enableLabels() {
		drawLabels = true;
	}

	void disableLabels() {
		drawLabels = false;
	}
	
	void setFontsize(int fontsize) {
		this.fontsize = fontsize;
	}
	
	void setDecimals(int nDecimals) {
		this.nDecimals = nDecimals;
	}
	
	void setLinewidth(double linewidth) {
		this.linewidth = linewidth;
	}
	
   	@Override
   	public void paint(Graphics2D g2, Sequence sequence, IcyCanvas canvas) {  		
   		//float maxWidth= 4;
   		//float minWidth= 1;
   		
   		if (g2 == null) {
   			// on a VTK canvas, the paint is called with g2==null
   			return;
   		}

   		//if ( !isEnabled() ) return;
   		//if ( !affectSequenceButton.isSelected() ) return;

   		Viewer viewer = sequence.getFirstViewer();
   		int t = viewer.getT();
   		
   		g2.setColor( Color.black );
   		//g2.setXORMode( Color.yellow );

   		g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
   		//g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_STROKE_PURE);
   		
   		// enable the fractional metrics so that text is not hinted to pixel values, so that
   		// the outline shape actually matches the main text
   		g2.setRenderingHint(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_ON);
   		
   		//BasicStroke stroke = new BasicStroke((float) (minWidth + (maxWidth - minWidth)*flowarrow.norme/resolution));
   		
   		//float width = 
   		
		BasicStroke stroke = new BasicStroke((float) linewidth, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND);
		BasicStroke outlineStroke = new BasicStroke((float) (fontsize*1.5/5.0), BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND);
		
        g2.setStroke(stroke);
        AffineTransform at = g2.getTransform();
		
        
        
		// loop over the contour lines
		for (Contour contour : globalContourList.get(t)) {
	         //System.out.println(line);
	         //System.out.println(line.getLevel());
	         //System.out.println(n);
        
	         Path2D.Double path = new Path2D.Double();
	         path.moveTo(contour.x[0] + 0.5, contour.y[0] + 0.5);
	         for (int i=1; i<contour.x.length; i++) {
	        	 path.lineTo(contour.x[i] + 0.5, contour.y[i] + 0.5);
	         }

	         g2.setStroke(stroke);
	         g2.draw(path);
	         
	         if (drawLabels) {
	        	 Font font = new Font("SansSerif", Font.PLAIN, fontsize);
	        	 g2.setFont(font);
	        	 
	        	 String format = String.format("%%.%df", nDecimals);
	        	 String s = String.format(format, contour.level);
	        	 
	        	 FontMetrics fm = g2.getFontMetrics();
	             Rectangle2D textBounds = fm.getStringBounds(s, g2);
	        	 
	             // text width should be much smaller than contour size
	             Rectangle2D contourBounds = contour.getBounds();
	             
	             if (contourBounds.getWidth() > 2.*textBounds.getWidth()
	            		 || contourBounds.getHeight() > 2.*textBounds.getHeight()) {
		        	 int l = contour.x.length;
		        	 double x0 = contour.x[l/2] - textBounds.getWidth()/2.;
		        	 double y0 = contour.y[l/2] + textBounds.getHeight()/2.;
		        	 
		        	 FontRenderContext frc = g2.getFontRenderContext();
		        	 TextLayout textTl = new TextLayout(s, font, frc);
		        	 Shape outline = textTl.getOutline(null);

		        	 AffineTransform transform = (AffineTransform) at.clone();
		        	 transform.translate(x0, y0);
		        	 g2.setTransform(transform);
		        	 
		        	 g2.setColor(Color.white);
		        	 g2.setStroke(outlineStroke);
		        	 g2.draw(outline);
		        	 
		        	 g2.setColor(Color.black);
		        	 g2.fill(outline);
		        	 
		        	 g2.setTransform(at); 
	             }
	         }
		}
		
        g2.setTransform(at);
   	}
   	
   	@Override
   	public void keyPressed(KeyEvent e, Point2D imagePoint, IcyCanvas canvas) {
   	}

   	@Override
   	public void keyReleased(KeyEvent e, Point2D imagePoint, IcyCanvas canvas) {
   	}

   	@Override
   	public void mouseClick(MouseEvent e, Point2D imagePoint, IcyCanvas canvas) {
   	}

   	@Override
   	public void mouseDrag(MouseEvent e, Point2D imagePoint, IcyCanvas canvas) {
   	}

   	@Override
   	public void mouseMove(MouseEvent e, Point2D imagePoint, IcyCanvas canvas) {
   	}

   	@Override
   	public void mousePressed(MouseEvent e, Point2D imagePoint, IcyCanvas canvas) {
   	}

   	@Override
   	public void mouseReleased(MouseEvent e, Point2D imagePoint, IcyCanvas canvas) {
   	}
}