package plugins.tlecomte.contourPlot;

import java.util.ArrayList;
import java.util.List;

import icy.gui.dialog.MessageDialog;
import icy.painter.Painter;
import icy.sequence.Sequence;
import icy.type.collection.array.Array1DUtil;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzVarBoolean;
import plugins.adufour.ezplug.EzVarDouble;
import plugins.adufour.ezplug.EzVarInteger;
import plugins.adufour.ezplug.EzVarSequence;

public class ContourPlot extends EzPlug {

	public EzVarSequence sequenceSelector = new EzVarSequence("Input Sequence");
	public EzVarSequence maskSelector = new EzVarSequence("Mask Sequence");
	public EzVarInteger	levelCountSelector = new EzVarInteger("Number of levels", 10, 1, Integer.MAX_VALUE, 1);
	public EzVarDouble	minSelector = new EzVarDouble("Min level", 0, -Double.MAX_VALUE, Double.MAX_VALUE, 1);
	public EzVarDouble	maxSelector = new EzVarDouble("Max level", 1, -Double.MAX_VALUE, Double.MAX_VALUE, 1);
	public EzVarBoolean removePreviousPaintersSelector = new EzVarBoolean("Remove previous contour painters", true);
	public EzVarDouble	linewidthSelector = new EzVarDouble("Contour linewidth", 1, 0, Double.MAX_VALUE, 1);
	public EzVarBoolean drawLabelsSelector = new EzVarBoolean("Print levels labels", true);
	public EzVarInteger	fontsizeSelector = new EzVarInteger("Label fontsize", 5, 1, Integer.MAX_VALUE, 1);
	
	@Override
	protected void initialize() {
		maskSelector.setNoSequenceSelection(); // no mask by default
		
		addEzComponent(sequenceSelector);
		addEzComponent(maskSelector);
		addEzComponent(levelCountSelector);
		addEzComponent(minSelector);
		addEzComponent(maxSelector);
		addEzComponent(removePreviousPaintersSelector);
		removePreviousPaintersSelector.setToolTipText(  "<html>If checked, the previous contour painters that are present on the sequence<br>"
				  + "will be removed. Uncheck if you want to preserve them.</html>");
		addEzComponent(linewidthSelector);
		addEzComponent(drawLabelsSelector);
		addEzComponent(fontsizeSelector);
	}
	
	@Override
	protected void execute() {
		Sequence sequence = sequenceSelector.getValue();
		Sequence maskSequence = maskSelector.getValue();
		
        // Check if sequence exists.
        if ( sequence == null )
        {
    		MessageDialog.showDialog("Please open a sequence to use this plugin.", MessageDialog.ERROR_MESSAGE );
    		return;
        }
        
		if (removePreviousPaintersSelector.getValue()) {
	    	// remove previous painters
    		List<Painter> painters = sequence.getPainters();
    		//List<Painter> painters = coverSequence.getPainters(ContourPainter2.class);
    		for (Painter painter : painters) {
    			if (painter.getClass().isAssignableFrom(ContourPainter.class)) {
    				sequence.removePainter(painter);
    				sequence.painterChanged(painter);    				
    			}
    		}		
		}
        
		int w = sequence.getSizeX();
    	int h = sequence.getSizeY();
		int numT = sequence.getSizeT();
		int z = 0;
		int channel = 0;
		
		double[] xArray = new double[w*h];
		double[] yArray = new double[w*h];
		
		for (int i=0; i<w; i++) {
			for (int j=0; j<h; j++) {
				xArray[i + j*w] = (double)i;
				yArray[i + j*w] = (double)j;
			}
		}
		
		ArrayList<ArrayList<Contour>> globalContourList = new ArrayList<ArrayList<Contour>>();

		// stores the level values
		double min = minSelector.getValue(); // ArrayMath.min(zArray);
		double max = maxSelector.getValue(); // ArrayMath.max(zArray);
		int levelCount = levelCountSelector.getValue();
		
		// ignore levelCount when min==max
		if (min==max) {
			levelCount = 1;
		}
		
		for (int t = 0; t<numT; t++) {
			// Get a a direct reference to the data as doubles.
			double[] zArray = Array1DUtil.arrayToDoubleArray(sequence.getDataXY(t, z, channel), sequence.isSignedDataType());
					
			int[] maskArray = null;
			if (maskSequence != null) {
				// Get a a direct reference to the data as doubles.
				maskArray = Array1DUtil.arrayToIntArray(maskSequence.getDataXY(t, z, channel), maskSequence.isSignedDataType());
			}
			
			ArrayList<Contour> contourList = new ArrayList<Contour>();
			
			for (int i=0; i<levelCount; i++) {
				double level;
				if (levelCount == 1) {
					level = min;
				}
				else
				{
					level = min + (max-min)*(double)i/(double)(levelCount-1);
				}
				
			    double[] levels = {level, -Double.MAX_VALUE};
			    int nlevels = 1;
			    int nchunk = 0;
			    
			    ContourTracer site = new ContourTracer(w, h, xArray, yArray, zArray, maskArray);
			    
			    try {
			    	contourList.addAll(cntr_trace(site, levels, nlevels, nchunk));			    	
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
			
			globalContourList.add(contourList);
		}
		
		int size = 0;
		for (ArrayList<Contour> list : globalContourList) {
			size += list.size();
		}
		if (size == 0) {
			MessageDialog.showDialog("No contour found ! Please check the level values.", MessageDialog.ERROR_MESSAGE );
    		return;
		}
	
    	ContourPainter painter = new ContourPainter(globalContourList);
    	painter.setLinewidth(linewidthSelector.getValue());
    	if (drawLabelsSelector.getValue()) {
    		painter.enableLabels();
    		painter.setFontsize(fontsizeSelector.getValue());   		
    		if (levelCount==1) {
    			int n = 0;
    			while (Math.abs(min*Math.pow(10, n) - (double) ((int) (min*Math.pow(10, n))))>1e-5 && n<5) {
    				n++;
    			}
    			painter.setDecimals(n);
    		}
    		else
    		{
    			double delta = (max-min)/(double)(levelCount-1);
    			double n = Math.log10(delta);
        		n = -n;
        		n = Math.max(n, 0);
        		n = Math.ceil(n);
        		n = Math.min(n, 5); // limit the number of decimals to 5
        		painter.setDecimals((int) n);
    		}
    	}
    	
        // add a painter to the sequence to draw the arrows
		sequence.addPainter(painter);
	}
	
	/* cntr_trace is called once per contour level or level pair.
	 * If nlevels is 1, a set of contour lines will be returned; if nlevels
	 * is 2, the set of polygons bounded by the levels will be returned.
	 * If points is True, the lines will be returned as a list of list
	 * of points; otherwise, as a list of tuples of vectors.
	 */
	List<Contour> cntr_trace(ContourTracer site, double levels[], int nlevels, int nchunk) throws Exception
	{
		site.zlevel[0] = levels[0];
		site.zlevel[1] = levels[0];
		if (nlevels == 2)
		{
			site.zlevel[1] = levels[1];
		}
		site.n = 0;
		site.count = 0;
		site.data_init (nchunk);
		
		/* make first pass to compute required sizes for second pass
		 * curve_tracer trace the next disjoint curve
		 * curve_tracer returns 0 when there is no more contour line */
		int n;
		int ntotal = 0;
		int nparts = 0;
		
		//System.out.println("First pass");
		while ((n = site.curve_tracer (false)) != 0)
		{
			if (n > 0)
			{ 	// normal case, n is the number of points of the curve
				nparts++;
				ntotal += n;
				
			}
			else
			{ 	// incomplete open curve, returns n=-(number of points)
				//System.out.println("Open curve, n = " + (-n));
				nparts++;
				ntotal += (-n);
			}
		}
		
		double[] x0 = new double[ntotal];
		double[] y0 = new double[ntotal];
		int[] k0 = new int[ntotal];
		
		List<Contour> contourList = new ArrayList<Contour>();
		
		//System.out.println("ntotal = " + ntotal + ", nparts = " + nparts);

		//System.out.println("Second pass");
		/* second pass */	
		for (int iseg = 0; iseg<nparts; iseg++)
		{
			site.xcp = x0;
			site.ycp = y0;
			site.kcp = k0;
			
			n = site.curve_tracer (true);
			
			if (n == 0) {
				System.out.println("curve_tracer returned 0, and iseg = " + iseg + ", nparts = " + nparts);
				break;
			}

			if (n <= 0)
			{
				throw new Exception("Negative or zero n from curve_tracer in pass 2");
			}
			
			double[] x1 = new double[n];
			double[] y1 = new double[n];
			double[] k1 = new double[n];
			
			for (int i=0; i<n; i++) {
				x1[i] = x0[i];
				y1[i] = y0[i];
				k1[i] = k0[i];
			}
			
			contourList.add(new Contour(x1, y1, levels[0]));
		}

		site.xcp = null;
		site.ycp = null;
		site.kcp = null;
		
		return contourList;
	}
	
	@Override
	public void clean() {
		// nothing to clean
	}
}