package plugins.adufour.morphology;

import icy.image.IcyBufferedImage;
import icy.sequence.Sequence;
import icy.sequence.SequenceUtil;
import icy.type.DataType;
import icy.type.collection.array.Array1DUtil;
import plugins.adufour.blocks.lang.Block;
import plugins.adufour.blocks.util.VarList;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzVarDouble;
import plugins.adufour.ezplug.EzVarSequence;
import plugins.adufour.vars.lang.VarSequence;

public class FillHolesInSequence extends EzPlug implements Block
{
    EzVarSequence input     = new EzVarSequence("input sequence");
    
    EzVarDouble   threshold = new EzVarDouble("threshold / fill value");
    
    VarSequence   output    = new VarSequence("output sequence", null);
    
    @Override
    protected void initialize()
    {
        addEzComponent(input);
        addEzComponent(threshold);
    }
    
    @Override
    public void declareInput(VarList inputMap)
    {
        inputMap.add("input", input.getVariable());
        inputMap.add("value", threshold.getVariable());
    }
    
    @Override
    public void declareOutput(VarList outputMap)
    {
        outputMap.add("output", output);
    }
    
    @Override
    public void clean()
    {
        output.setValue(null);
    }
    
    @Override
    protected void execute()
    {
        Sequence outputSequence = SequenceUtil.getCopy(input.getValue(true));
        
        for (IcyBufferedImage image : outputSequence.getAllImage())
            fillHoles(image, threshold.getValue());
        
        output.setValue(outputSequence);
        
        if (getUI() != null) addSequence(outputSequence);
    }
    
    /**
     * Fills the holes in the given image with the specified value.<br/>
     * The algorithm first extracts the image background using a flooding approach until
     * <code>threshold</code> is reached. The remainder of the image is then filled with
     * <code>threshold</code>.
     * 
     * @param bin
     * @param threshold
     * @param fillValue
     */
    public void fillHoles(IcyBufferedImage bin, double threshold)
    {
        int width = bin.getWidth();
        int height = bin.getHeight();
        int slice = width * height;
        int channels = bin.getSizeC();
        
        DataType type = bin.getDataType_();
        
        final byte FREE = 0;
        final byte BACKGROUND = 1;
        final byte VISITED = 2;
        
        for (int c = 0; c < channels; c++)
        {
            byte[] labels = new byte[slice];
            int[] backgroundNeighbors = new int[slice];
            Object in = bin.getDataXY(c);
            int n = 0;
            
            // 1) extract background pixels on the image edges
            
            // 1.a) top + bottom edges
            for (int top = 0, bottom = slice - width; top < width; top++, bottom++)
            {
                if (Array1DUtil.getValue(in, top, type) < threshold)
                {
                    labels[top] = BACKGROUND;
                    backgroundNeighbors[n++] = top + width;
                }
                else
                {
                    labels[top] = VISITED;
                }
                
                if (Array1DUtil.getValue(in, bottom, type) < threshold)
                {
                    labels[bottom] = BACKGROUND;
                    backgroundNeighbors[n++] = bottom - width;
                }
                else
                {
                    labels[bottom] = VISITED;
                }
            }
            
            // 1.b) left + right edges
            for (int left = 1, right = width - 1; left < slice; left += width, right += width)
            {
                if (Array1DUtil.getValue(in, left, type) < threshold)
                {
                    labels[left] = BACKGROUND;
                    backgroundNeighbors[n++] = left + 1;
                }
                else
                {
                    labels[left] = VISITED;
                }
                
                if (Array1DUtil.getValue(in, right, type) < threshold)
                {
                    labels[right] = BACKGROUND;
                    backgroundNeighbors[n++] = right - 1;
                }
                else
                {
                    labels[right] = VISITED;
                }
            }
            
            if (n == 0) System.err.println("fillHoles_2D was unable to find a background pixel for flooding");
            
            System.out.println("neighbors (first pass): " + n);
            
            // 2) flood the image from the list of border pixels
            
            for (int index = 0; index < n; index++)
            {
                int offset = backgroundNeighbors[index];
                
                if (labels[offset] != BACKGROUND)
                {
                    if (Array1DUtil.getValue(in, offset, type) < threshold)
                    {
                        labels[offset] = BACKGROUND;
                        
                        for (int neighbor : new int[] { offset - 1, offset + 1, offset - width, offset + width })
                            if (labels[neighbor] == FREE && Array1DUtil.getValue(in, neighbor, type) < threshold)
                            {
                                labels[neighbor] = VISITED;
                                backgroundNeighbors[n++] = neighbor;
                            }
                    }
                }
            }
            
            System.out.println("neighbors (second pass): " + n);
            
            // 3) fill holes by overwriting all "FREE" pixels
            
            for (int i = 0; i < slice; i++)
                if (labels[i] == FREE && Array1DUtil.getValue(in, i, type) < threshold) Array1DUtil.setValue(in, i, type, threshold);
        }
    }
}
