package plugins.ylemontag.matlabfunctioncaller;

import icy.file.FileUtil;
import icy.preferences.XMLPreferences;
import icy.system.thread.ThreadUtil;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.LinkedList;

/**
 * 
 * @author Yoann Le Montagner
 * 
 * Main classes to execute a Matlab function from Icy
 */
public class MatlabFunctionCaller
{
	/**
	 * Exception thrown when the execution of the function failed
	 */
	public static class ExecutionFailed extends Exception
	{
		private static final long serialVersionUID = 1L;
		
		private String _matlabMessage;
		
		public ExecutionFailed(String message)
		{
			this(message, null);
		}
		
		public ExecutionFailed(String message, String matlabMessage)
		{
			super(message);
			_matlabMessage = matlabMessage;
		}
		
		public String getMatlabMessage()
		{
			return _matlabMessage;
		}
	}
	
	private MatlabFunctionHeader _header    ;
	private File                 _inputFile ;
	private File                 _outputFile;
	private File                 _scriptFile;
	private ArgumentSetInput     _inputArgs ;
	private ArgumentSetOutput    _outputArgs;
	
	/**
	 * Constructor
	 */
	public MatlabFunctionCaller(String fileName) throws IOException
	{
		this(new File(fileName));
	}
	
	/**
	 * Constructor
	 */
	public MatlabFunctionCaller(File file) throws IOException
	{
		this(parseFunctionHeader(file));
	}
	
	/**
	 * Constructor
	 */
	public MatlabFunctionCaller(MatlabFunctionHeader header) throws IOException
	{
		_header     = header;
		_inputFile  = allocateTemporaryFile("mat");
		_outputFile = allocateTemporaryFile("mat");
		_scriptFile = allocateTemporaryFile("m"  );
		_inputArgs  = new ArgumentSetInput (_header.getIn ());
		_outputArgs = new ArgumentSetOutput(_header.getOut());
		createScript();
	}
	
	/**
	 * Function header
	 */
	public MatlabFunctionHeader getFunctionHeader()
	{
		return _header;
	}
	
	/**
	 * Input arguments
	 */
	public ArgumentSetInput getInputArguments()
	{
		return _inputArgs;
	}
	
	/**
	 * Output arguments
	 */
	public ArgumentSetOutput getOutputArguments()
	{
		return _outputArgs;
	}
	
	/**
	 * Execute the function call
	 */
	public void execute() throws IOException, ExecutionFailed
	{
		// All the input arguments must be binded before executing the function
		if(!_inputArgs.allBinded()) {
			throw new IllegalStateException("All the input arguments must be binded.");
		}
		
		// Pass the input arguments to the temporary file
		_inputArgs.save(_inputFile);
		
		// Call Matlab
		File   executionDirectory = _scriptFile.getParentFile();
		String scriptName         = FileUtil.getFileName(_scriptFile.getAbsolutePath(), false);
		runMatlab(executionDirectory, scriptName);
		
		// Retrieve the ouput arguments from the second temporary file
		_outputArgs.load(_outputFile);
	}
	
	/**
	 * Run Matlab in headless mode
	 */
	private void runMatlab(File executionDirectory, String scriptName)
		throws IOException, ExecutionFailed
	{
		// Retrieve the Matlab command
		Boolean isGlobalOptionsValid = MatlabFunctionCaller.getOptions().isValid();
		if(isGlobalOptionsValid==null || !isGlobalOptionsValid.booleanValue()) {
			throw new IllegalStateException("Invalid Matlab execution command");
		}
		String matlabCommand = getOptions().getMatlabCommand();
		
		// Run Matlab and starts the dedicated script
		String command = matlabCommand + " -nodisplay -nojvm -r " + scriptName;
		Process p = Runtime.getRuntime().exec(command, null, executionDirectory);
		writeOnStream(p.getOutputStream(), "pause(1); exit(1);\n");
		try {
			int exitCode = p.waitFor();
			if(exitCode!=0) {
				String matlabMessage = readFromStream(p.getErrorStream());
				throw new ExecutionFailed(
					"Non-zero exit code: " + exitCode, matlabMessage
				);
			}
		}
		catch(InterruptedException err) {
			throw new ExecutionFailed("Process interrupted");
		}
	}
	
	/**
	 * Write the script that should be run by Matlab
	 */
	private void createScript() throws IOException
	{
		Writer writer = new BufferedWriter(new FileWriter(_scriptFile, false));
		try {
			writer.write(
				"% This file is automatically generated from Icy by the Matlab function caller plugin.\n" +
				"% Do not edit manually.\n" + 
				instructionSetFolder() + "\n" +
				instructionLoad     () + "\n" +
				instructionExecute  () + "\n" +
				instructionSave     () + "\n" +
				"exit(0);\n"
			);
		}
		finally {
			writer.close();
		}
	}
	
	/**
	 * Return the Matlab instruction to use to switch to the working directory
	 * associated to this function
	 */
	public String instructionSetFolder()
	{
		return "cd(" + escape(_header.getFolder().getAbsolutePath()) + ");";
	}
	
	/**
	 * Return the Matlab instruction corresponding to the execution of this function
	 */
	public String instructionExecute()
	{
		String   name = _header.getName();
		String[] in   = _header.getIn  ();
		String[] out  = _header.getOut ();
		if(out.length==0) {
			return name + " (" + implode(in, false, false) + ");";
		}
		else {
			return "[" + implode(out, false, false) + "] = " + name + "(" + implode(in, false, false) + ");";
		}
	}
	
	/**
	 * Return the Matlab instruction to load the arguments for this function from the
	 * given file
	 */
	private String instructionLoad()
	{
		String[] in = _header.getIn();
		if(in.length==0) {
			return "";
		}
		else {
			return "load(" + escape(_inputFile.getAbsolutePath()) + implode(in, true, true) + ");";
		}
	}
	
	/**
	 * Return the Matlab instruction to save the arguments returned by this function
	 * to the given file
	 */
	private String instructionSave()
	{
		String[] out = _header.getOut();
		if(out.length==0) {
			return "tmp = struct(); save(" + escape(_outputFile.getAbsolutePath()) + ", '-struct', 'tmp');";
		}
		else {
			return "save(" + escape(_outputFile.getAbsolutePath()) + implode(out, true, true) + ");";
		}
	}
	
	/**
	 * Concatenate a list of argument names and separate them with commas
	 */
	private static String implode(String[] names, boolean prefix_with_comma, boolean quote_names)
	{
		String retVal = "";
		for(int k=0; k<names.length; ++k) {
			if(k>0 || prefix_with_comma) {
				retVal += ", ";
			}
			retVal += quote_names ? ("'" + names[k] + "'") : names[k];
		}
		return retVal;
	}
	
	/**
	 * Escape a string a return a Matlab litteral with special charaters properly handled
	 */
	private static String escape(String value)
	{
		return "'" + value.replaceAll("'", "''") + "'";
	}
	
	/**
	 * Read a text from a stream
	 */
	private static String readFromStream(InputStream stream) throws IOException
	{
		BufferedReader reader = new BufferedReader(new InputStreamReader(stream));
		String retVal = "";
		while(true) {
			String buffer = reader.readLine();
			if(buffer==null) {
				break;
			}
			retVal += buffer + "\n";
		}
		return retVal;
	}
	
	/**
	 * Write a text on the given output stream
	 */
	private static void writeOnStream(OutputStream stream, String text) throws IOException
	{
		Writer writer = new BufferedWriter(new OutputStreamWriter(stream));
		writer.write(text);
		writer.flush();
	}
	
	/**
	 * Allocate a new temporary file, ending with the given extension
	 */
	private static File allocateTemporaryFile(String extension) throws IOException
	{
		File temporaryFile = File.createTempFile("icy_", "." + extension);
		temporaryFile.deleteOnExit();
		return temporaryFile;
	}
	
	/**
	 * Try to parse a .m file, and convert the parsing exception into a IOException
	 */
	private static MatlabFunctionHeader parseFunctionHeader(File file) throws IOException
	{
		try {
			return MatlabFunctionHeader.parse(file);
		}
		catch(MatlabFunctionHeader.ParsingException err) {
			throw new IOException(err);
		}
	}
	
	/**
	 * General options and parameters
	 */
	public static class Options
	{
		/**
		 * Listen to option modifications
		 */
		public static interface Listener
		{
			/**
			 * Method fired when the validity state of the options changes
			 */
			void stateChanged(Boolean isValid, String stateMessage);
		}
		
		private LinkedList<Listener> _listeners    ;
		private XMLPreferences       _prefs        ;
		private Boolean              _isValid      ;
		private String               _stateMessage ;
		private String               _matlabCommand;
		
		/**
		 * Constructor
		 */
		public Options(XMLPreferences prefs)
		{
			_listeners     = new LinkedList<Listener>();
			_prefs         = prefs;
			_isValid       = null;
			_stateMessage  = null;
			_matlabCommand = _prefs.get("MatlabCommand", "matlab");
			checkValidity(false);
		}
		
		/**
		 * Add a new listener
		 */
		public void addListener(Listener l)
		{
			_listeners.add(l);
		}
		
		/**
		 * Remove a listener
		 */
		public void removeListener(Listener l)
		{
			_listeners.remove(l);
		}
		
		/**
		 * Save the options to the underlying persistent XML node
		 */
		public void refreshPersistentData()
		{
			_prefs.put("MatlabCommand", _matlabCommand);
		}
		
		/**
		 * Make sure that the current set of options is consistent, so that it
		 * can be use to run a function call
		 */
		public Boolean isValid()
		{
			return _isValid;
		}
		
		/**
		 * Return a message describing whether the set of options is valid or not,
		 * and what the problem is if any
		 */
		public String getStateMessage()
		{
			return _stateMessage;
		}
		
		/**
		 * Command to use to run Matlab
		 */
		public String getMatlabCommand()
		{
			return _matlabCommand;
		}
		
		/**
		 * Set the command to use to run Matlab
		 */
		public void setMatlabCommand(String command)
		{
			_matlabCommand = command;
			checkValidity(true);
		}
		
		/**
		 * Perform the option validity checks
		 */
		private void checkValidity(final boolean forceCheck)
		{
			ThreadUtil.bgRun(new Runnable()
			{
				@Override
				public void run()
				{
					// Reset the flag
					if(!forceCheck && _isValid!=null) {
						return;
					}
					updateState(null, "");
					
					// The Matlab command must be set
					if(_matlabCommand.isEmpty()) {
						updateState(false, "Matlab command option not defined yet.");
						return;
					}
					
					// Try to run Matlab
					if(!tryRunCommand(_matlabCommand + " -n")) {
						updateState(false, "Matlab command incorrectly set: it does not seems to launch Matlab properly.");
						return;
					}
					
					// At this point, everything is fine
					updateState(true, "Global options properly set.");
				}
			});
		}
		
		/**
		 * Change the values of the isValid flag and the state message
		 */
		private void updateState(final Boolean isValid, final String stateMessage)
		{
			_isValid      = isValid     ;
			_stateMessage = stateMessage;
			ThreadUtil.invokeLater(new Runnable()
			{
				@Override
				public void run()
				{
					for(Listener l : _listeners) {
						l.stateChanged(isValid, stateMessage);
					}
				}
			});
		}
		
		/**
		 * Try to run the given command, and make sure that it terminates normally
		 * (with an exit code equal to 0)
		 */
		private static boolean tryRunCommand(String command)
		{
			try {
				final Process p = Runtime.getRuntime().exec(command);
				ThreadUtil.bgRun(new Runnable()
				{
					@Override
					public void run() {
						ThreadUtil.sleep(1000);
						p.destroy();
					}
				});
				int exitCode = p.waitFor();
				return exitCode==0;
			}
			catch(InterruptedException err) {}
			catch(IOException err) {}
			return false;
		}
	}
	
	/**
	 * Singleton holding the global options associated to the Matlab function caller
	 */
	public static Options getOptions()
	{
		if(_options==null) {
			MatlabFunctionCallerPlugin plugin = new MatlabFunctionCallerPlugin(); /// TODO: remove instanciation
			_options = new Options(plugin.getMatlabFunctionCallerOptionsNode());
		}
		return _options;
	}
	
	/**
	 * Singleton holding the global options associated to the Matlab function caller
	 */
	private static Options _options;
}
