package plugins.ylemontag.matlabfunctioncaller;

import icy.file.FileUtil;
import icy.preferences.XMLPreferences;
import icy.sequence.Sequence;
import icy.system.IcyHandledException;
import icy.system.thread.ThreadUtil;

import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import plugins.adufour.ezplug.EzButton;
import plugins.adufour.ezplug.EzDialog;
import plugins.adufour.ezplug.EzGroup;
import plugins.adufour.ezplug.EzLabel;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzVar;
import plugins.adufour.ezplug.EzVarDouble;
import plugins.adufour.ezplug.EzVarDoubleArrayNative;
import plugins.adufour.ezplug.EzVarEnum;
import plugins.adufour.ezplug.EzVarFile;
import plugins.adufour.ezplug.EzVarInteger;
import plugins.adufour.ezplug.EzVarIntegerArrayNative;
import plugins.adufour.ezplug.EzVarListener;
import plugins.adufour.ezplug.EzVarSequence;
import plugins.adufour.ezplug.EzVarText;
import plugins.adufour.vars.gui.VarEditor;
import plugins.adufour.vars.lang.VarSequence;

/**
 * 
 * @author Yoann Le Montagner
 * 
 * EzPlug interface to call a Matlab function
 */
public class MatlabFunctionCallerPlugin extends EzPlug
{
	private EzButton                  _options    ;
	private EzVarFile                 _file       ;
	private EzLabel                   _description;
	private List<EzGroupInput>        _inputs     ;
	private Map<String, EzGroupInput> _allInputs  ;
	
	@Override
	protected void initialize()
	{
		// Option management
		_options = new EzButton("", new ActionListener()
		{
			@Override
			public void actionPerformed(ActionEvent e)
			{
				onOptionsClicked();
			}
		});
		addEzComponent(_options);
		MatlabFunctionCaller.getOptions().addListener(new MatlabFunctionCaller.Options.Listener()
		{	
			@Override
			public void stateChanged(Boolean isValid, String stateMessage)
			{
				onGlobalOptionsChanged(isValid);
			}
		});
		onGlobalOptionsChanged(MatlabFunctionCaller.getOptions().isValid());
		
		// Matlab function selection
		_file = new EzVarFile("Matlab function", loadPersistentDirectory());
		_file.setToolTipText("Matlab .m file to launch. This must be a function, not a script.");
		_file.addVarChangeListener(new EzVarListener<File>()
		{
			@Override
			public void variableChanged(EzVar<File> source, File newValue)
			{
				onFunctionChanged();
			}
		});
		_description = new EzLabel("");
		_inputs      = new LinkedList<EzGroupInput>();
		_allInputs   = new HashMap<String, EzGroupInput>();
		addEzComponent(_file       );
		addEzComponent(_description);
		onFunctionChanged();
	}

	@Override
	protected void execute()
	{
		// Check the global options
		Boolean isGlobalOptionsValid = MatlabFunctionCaller.getOptions().isValid();
		if(isGlobalOptionsValid==null || !isGlobalOptionsValid.booleanValue()) {
			throw new IcyHandledException("Invalid Matlab execution command");
		}
		
		// File to execute
		File file = _file.getValue();
		if(file==null || !file.exists()) {
			throw new IcyHandledException("You must select an existing Matlab .m function file.");
		}
		
		// Create the Matlab function caller object
		try {
			final MatlabFunctionCaller caller = new MatlabFunctionCaller(file);
			
			// Bind the inputs
			ArgumentSetInput inputs = caller.getInputArguments();
			for(String name : inputs.getArgumentNames()) {
				inputs.bind(name, _allInputs.get(name).getInput());
			}
			
			// Launch Matlab
			caller.execute();
			
			// Build the output frame
			ThreadUtil.invokeLater(new Runnable()
			{
				@Override
				public void run()
				{
					ResultFrame frame = new ResultFrame(caller.getFunctionHeader(), caller.getOutputArguments());
					frame.addToMainDesktopPane();
					frame.setVisible(true);
				}
			});
		}
		
		// Report execution failure
		catch(MatlabFunctionCaller.ExecutionFailed err) {
			String message = "The execution of the selected Matlab function has failed.";
			if(err.getMatlabMessage()!=null) {
				message += "Error message:\n" + err.getMatlabMessage();
			}
			throw new IcyHandledException(message);
		}
		
		// Report unexpected I/O errors
		catch(IOException err) {
			throw new IcyHandledException(
				"An error has occurred while executing the Matlab function "
				+ file.getAbsolutePath() + ". Here is the error message:\n"
				+ err.getMessage()
			);
		}
	}
	
	@Override
	public void clean() {}
	
	/**
	 * Open the options dialog
	 */
	private void onOptionsClicked()
	{
		MatlabFunctionCaller.Options options = MatlabFunctionCaller.getOptions();
		OptionDialog dialog = new OptionDialog(options);
		dialog.showDialog();
		options.refreshPersistentData();
	}
	
	/**
	 * Refresh the text in the option state label
	 */
	private void onGlobalOptionsChanged(Boolean isValid)
	{
		String label = "Options ";
		if(isValid==null) {
			label += " [Checking...]";
		}
		else {
			label += isValid.booleanValue() ? "[OK]" : "[Bad configuration]";
		}
		_options.setText(label);
	}
	
	/**
	 * Refresh the interface when the selected Matlab function changes
	 */
	private void onFunctionChanged()
	{
		// Reset the interface
		_description.setColor(Color.BLACK);
		clearInputs();
		
		// Retrieve new file, and abort if it does not exist
		File file = _file.getValue();
		savePersistentDirectory(file);
		if(file==null || !file.exists()) {
			_description.setText("<No function selected>");
			return;
		}
		
		// Try to parse the header of the selected file
		try {
			MatlabFunctionHeader header = MatlabFunctionHeader.parse(file);
			_description.setText(header.toString());
			
			// Rebuild the list of input variables, and exit
			for(String name : header.getIn()) {
				addInput(name);
			}
			getUI().repack(true);
			return;
		}
		
		// Catch the parsing errors
		catch(MatlabFunctionHeader.ParsingException err) {}
		catch(IOException err) {}
		_description.setColor(Color.RED);
		_description.setText("<Invalid Matlab function file>");
	}
	
	/**
	 * Remove all the currently existing input arguments
	 */
	private void clearInputs()
	{
		for(EzGroupInput group : _inputs) {
			group.setVisible(false);
		}
		_inputs.clear();
	}
	
	/**
	 * Add a new input argument
	 */
	private void addInput(String name)
	{
		EzGroupInput group = _allInputs.get(name);
		if(group==null) {
			group = new EzGroupInput(name);
			_allInputs.put(name, group);
			addEzComponent(group);
		}
		else {
			group.setVisible(true);
		}
		_inputs.add(group);
	}
	
	/**
	 * Save the directory of the currently selected file to the XML preference file
	 */
	private void savePersistentDirectory(File file)
	{
		if(file==null) {
			return;
		}
		if(!file.isDirectory()) {
			file = file.getParentFile();
		}
		getPreferencesRoot().put("Path", file.getAbsolutePath());
	}
	
	/**
	 * Set the current directory for the function selector from the one saved in
	 * the XML preference file
	 */
	private String loadPersistentDirectory()
	{
		return getPreferencesRoot().get("Path", FileUtil.getCurrentDirectory());
	}
	
	/**
	 * XML node holding the global preferences associated to the MatlabFunctionCaller objects
	 */
	public XMLPreferences getMatlabFunctionCallerOptionsNode()
	{
		return getPreferences("Options");
	}
	
	/**
	 * Dialog to set the global options
	 */
	private static class OptionDialog extends EzDialog
	{
		private MatlabFunctionCaller.Options _options;
		private EzVarText _matlabCommand;
		private EzLabel _stateLabel;
		
		/**
		 * Constructor
		 */
		public OptionDialog(MatlabFunctionCaller.Options options)
		{
			super("Matlab function caller options");
			_options = options;
			
			// Matlab command field
			_matlabCommand = new EzVarText("Matlab command");
			_matlabCommand.setValue(options.getMatlabCommand());
			addEzComponent(_matlabCommand);
			
			// State label
			_stateLabel = new EzLabel("");
			addEzComponent(_stateLabel);
			_options.addListener(new MatlabFunctionCaller.Options.Listener()
			{
				@Override
				public void stateChanged(Boolean isValid, String stateMessage)
				{
					onOptionsChanged(isValid, stateMessage);
				}
			});
			onOptionsChanged(_options.isValid(), _options.getStateMessage());
			
			// Validation button
			EzButton ok = new EzButton("Update options", new ActionListener()
			{
				@Override
				public void actionPerformed(ActionEvent e)
				{
					onOkClicked();
				}
			});
			addEzComponent(ok);
			repack(true);
		}
		
		/**
		 * Action performed when the button is clicked
		 */
		private void onOkClicked()
		{
			_options.setMatlabCommand(_matlabCommand.getValue());
			//hideDialog();
		}
		
		/**
		 * Refresh the state label
		 */
		private void onOptionsChanged(Boolean isValid, String stateMessage)
		{
			if(isValid==null) {
				_stateLabel.setColor(Color.BLACK);
				_stateLabel.setText("Checking...");
			}
			else {
				_stateLabel.setColor(isValid.booleanValue() ? new Color(0, 128, 0) : Color.RED);
				_stateLabel.setText(stateMessage);
			}
			repack(false);
		}
	}
	
	/**
	 * Recast request
	 */
	private static interface RecastRequestListener
	{
		void onRecastRequested(Variant.Type newType);
	}
	
	/**
	 * Result frame
	 */
	private static class ResultFrame extends EzDialog
	{
		private ArgumentSetOutput _data;
		private EzLabel _infoLabel;
		
		/**
		 * Constructor
		 */
		public ResultFrame(MatlabFunctionHeader header, ArgumentSetOutput data)
		{
			super("Result: " + header, false);
			_data = data;
			_infoLabel = new EzLabel("");
			_infoLabel.setVisible(false);
			for(String name : header.getOut()) {
				addOutput(name);
			}
			addEzComponent(_infoLabel);
			repack(true);
		}
		
		/**
		 * Create a new EzGroupOutput object for the given variable
		 */
		private void addOutput(final String name)
		{
			// Nothing to do if the variable is not binded or if it is not convertible
			// to any supported type
			if(!_data.isBinded(name)) {
				return;
			}
			Variant.Type[] recastableTypes = _data.getRecastableTypes(name);
			if(recastableTypes.length==0) {
				return;
			}
			
			// Otherwise, create the EzGroupOutput object to display the content
			// of the variable
			final EzGroupOutput group = new EzGroupOutput(name, recastableTypes);
			addEzComponent(group);
			group.setOutput(_data.getArgument(name));
			
			// Manage recast requests
			group.setRecastRequestListener(new RecastRequestListener()
			{
				@Override
				public void onRecastRequested(final Variant.Type newType)
				{
					group.setVisible(false);
					_infoLabel.setText("Converting " + name + " to " + newType + "...");
					_infoLabel.setVisible(true);
					repack(true);
					ThreadUtil.bgRun(new Runnable()
					{
						@Override
						public void run()
						{
							try {
								_data.recast(name, newType);
								ThreadUtil.invokeLater(new Runnable()
								{
									@Override
									public void run()
									{
										group.setOutput(_data.getArgument(name));
										group.setVisible(true);
										_infoLabel.setVisible(false);
										repack(true);
									}
								});
							}
							catch(IOException err) {
								ThreadUtil.invokeLater(new Runnable()
								{
									@Override
									public void run()
									{
										addEzComponent(new EzLabel("The variable " + name + " has been corrupted."));
										_infoLabel.setVisible(false);
										repack(true);
									}
								});
							}
						}
					});
				}
			});
		}
	}
	
	/**
	 * Group of EzComponent objects to define the value to bind to a Matlab function
	 * input argument
	 */
	private static class EzGroupInput extends EzGroup
	{
		private EzVarEnum<Variant.Type> _type    ;
		private EzVarInteger            _integer ;
		private EzVarIntegerArrayNative _integers;
		private EzVarDouble             _decimal ;
		private EzVarDoubleArrayNative  _decimals;
		private EzVarText               _text    ;
		private EzVarSequence           _sequence;
		
		/**
		 * Constructor
		 */
		public EzGroupInput(String label)
		{
			this(label,
				new EzVarEnum<Variant.Type>("Type", Variant.Type.values(), Variant.Type.SEQUENCE),
				new EzVarInteger           ("Integer" ),
				new EzVarIntegerArrayNative("Integers", null, true),
				new EzVarDouble            ("Decimal" ),
				new EzVarDoubleArrayNative ("Decimals", null, true),
				new EzVarText              ("Text"    ),
				new EzVarSequence          ("Sequence")
			);
		}
		
		/**
		 * Core constructor
		 */
		private EzGroupInput(String label,
			EzVarEnum<Variant.Type> type    ,
			EzVarInteger            integer ,
			EzVarIntegerArrayNative integers,
			EzVarDouble             decimal ,
			EzVarDoubleArrayNative  decimals,
			EzVarText               text    ,
			EzVarSequence           sequence)
		{
			super(label, type, integer, integers, decimal, decimals, text, sequence);
			_type     = type    ;
			_integer  = integer ;
			_integers = integers;
			_decimal  = decimal ;
			_decimals = decimals;
			_text     = text    ;
			_sequence = sequence;
			_type.addVisibilityTriggerTo(_integer , Variant.Type.INTEGER );
			_type.addVisibilityTriggerTo(_integers, Variant.Type.INTEGERS);
			_type.addVisibilityTriggerTo(_decimal , Variant.Type.DECIMAL );
			_type.addVisibilityTriggerTo(_decimals, Variant.Type.DECIMALS);
			_type.addVisibilityTriggerTo(_text    , Variant.Type.TEXT    );
			_type.addVisibilityTriggerTo(_sequence, Variant.Type.SEQUENCE);
		}
		
		/**
		 * Retrieve the selected value
		 */
		public Variant getInput()
		{
			try {
				switch(_type.getValue()) {
					case INTEGER : return new Variant(_integer .getValue());
					case INTEGERS: return new Variant(_integers.getValue());
					case DECIMAL : return new Variant(_decimal .getValue());
					case DECIMALS: return new Variant(_decimals.getValue());
					case TEXT    : return new Variant(_text    .getValue());
					case SEQUENCE: return new Variant(_sequence.getValue());
					default:
						throw new RuntimeException("Unreachable code point");
				}
			}
			catch(IllegalArgumentException err) {
				throw new IcyHandledException(err.getMessage());
			}
		}
	}
	
	/**
	 * Group of EzComponent objects to define the value to bind to a Matlab function
	 * input argument
	 */
	private static class EzGroupOutput extends EzGroup
	{
		private RecastRequestListener   _listener;
		private EzVarEnum<Variant.Type> _type    ;
		private EzVarInteger            _integer ;
		private EzVarIntegerView        _integers;
		private EzVarDouble             _decimal ;
		private EzVarDoubleView         _decimals;
		private EzVarText               _text    ;
		private EzVarSequenceView       _sequence;
		
		/**
		 * Constructor
		 */
		public EzGroupOutput(String label, Variant.Type[] recastableTypes)
		{
			this(label,
				new EzVarEnum<Variant.Type>("Type", recastableTypes),
				new EzVarInteger           ("Integer" ),
				new EzVarIntegerView       ("Integers"),
				new EzVarDouble            ("Decimal" ),
				new EzVarDoubleView        ("Decimals"),
				new EzVarText              ("Text"    ),
				new EzVarSequenceView      ("Sequence")
			);
		}
		
		/**
		 * Core constructor
		 */
		private EzGroupOutput(String label,
			EzVarEnum<Variant.Type> type    ,
			EzVarInteger            integer ,
			EzVarIntegerView        integers,
			EzVarDouble             decimal ,
			EzVarDoubleView         decimals,
			EzVarText               text    ,
			EzVarSequenceView       sequence)
		{
			super(label, type, integer, integers, decimal, decimals, text, sequence);
			_type     = type    ;
			_integer  = integer ;
			_integers = integers;
			_decimal  = decimal ;
			_decimals = decimals;
			_text     = text    ;
			_sequence = sequence;
			_type.addVisibilityTriggerTo(_integer , Variant.Type.INTEGER );
			_type.addVisibilityTriggerTo(_integers, Variant.Type.INTEGERS);
			_type.addVisibilityTriggerTo(_decimal , Variant.Type.DECIMAL );
			_type.addVisibilityTriggerTo(_decimals, Variant.Type.DECIMALS);
			_type.addVisibilityTriggerTo(_text    , Variant.Type.TEXT    );
			_type.addVisibilityTriggerTo(_sequence, Variant.Type.SEQUENCE);
			_type.addVarChangeListener(new EzVarListener<Variant.Type>()
			{
				@Override
				public void variableChanged(EzVar<Variant.Type> source, Variant.Type newValue)
				{
					if(_listener!=null) {
						_listener.onRecastRequested(newValue);
					}
				}
			});
		}
		
		/**
		 * Display the given value
		 */
		public void setOutput(Variant value)
		{
			Variant.Type type = value.getType();
			_type.setValue(type);
			switch(type) {
				case INTEGER : _integer .setValue(value.getAsInteger ()); break;
				case INTEGERS: _integers.setValue(value.getAsIntegers()); break;
				case DECIMAL : _decimal .setValue(value.getAsDecimal ()); break;
				case DECIMALS: _decimals.setValue(value.getAsDecimals()); break;
				case TEXT    : _text    .setValue(value.getAsText    ()); break;
				case SEQUENCE: _sequence.setValue(value.getAsSequence()); break;
				default:
					throw new RuntimeException("Unreachable code point");
			}
		}
		
		/**
		 * Define how recast requests should be handled
		 */
		public void setRecastRequestListener(RecastRequestListener l)
		{
			_listener = l;
		}
	}
	
	/**
	 * Display long integer arrays with proper ellipsis
	 */
	private static class EzVarIntegerView extends EzVarText
	{
		public EzVarIntegerView(String varName)
		{
			super(varName);
		}
		
		public void setValue(int[] v)
		{
			if(v.length>=6) {
				setValue("<Array of " + v.length + " integers>");
			}
			else {
				String text = "";
				for(int k=0; k<v.length; ++k) {
					if(k>0) {
						text += " ";
					}
					text += v[k];
				}
				setValue(text);
			}
		}
	}
	
	/**
	 * Display long double arrays with proper ellipsis
	 */
	private static class EzVarDoubleView extends EzVarText
	{
		public EzVarDoubleView(String varName)
		{
			super(varName);
			setEnabled(false);
		}
		
		public void setValue(double[] v)
		{
			if(v.length>=4) {
				setValue("<Array of " + v.length + " decimals>");
			}
			else {
				String text = "";
				for(int k=0; k<v.length; ++k) {
					if(k>0) {
						text += " ";
					}
					text += v[k];
				}
				setValue(text);
			}
		}
	}
	
	/**
	 * Extends EzVar<Sequence> to use the mini-display component as editor
	 */
	private static class EzVarSequenceView extends EzVar<Sequence>
	{
		public EzVarSequenceView(String varName)
		{
			super(new VarSequence(varName, null)
			{
				@Override
				public VarEditor<Sequence> createVarEditor()
				{
					return createVarViewer();
				}
			}, null);
		}
	}
}
