/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.blobplugin.misc;

import icy.sequence.Sequence;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import plugins.adufour.ezplug.EzComponent;
import plugins.adufour.ezplug.EzVarBoolean;
import plugins.adufour.ezplug.EzVarDouble;
import plugins.adufour.ezplug.EzVarEnum;
import plugins.adufour.ezplug.EzVarInteger;
import plugins.adufour.ezplug.EzVarNumeric;
import plugins.adufour.ezplug.EzVarSequence;
import plugins.adufour.vars.lang.Var;
import plugins.adufour.vars.util.VarListener;
import plugins.big.bigsnakeutils.icy.gui.pair.EzVarPair;
import plugins.big.bigsnakeutils.icy.gui.pair.Pair;
import plugins.big.bigsnakeutils.icy.gui.pair.RangeRangeModel;
import plugins.big.blobplugin.BlobPlugin;

/**
 * @class Parameters
 * 
 *        Class containing all the parameters used by the algorithm that the
 *        user can tune. It also initialize the graphical variables used by the
 *        plugins <a
 *        href="http://icy.bioimageanalysis.org/plugin/EzPlug_SDK"><b>
 *        EzPlug</b></a> and <a
 *        href="http://icy.bioimageanalysis.org/plugin/Protocols"
 *        ><b>Protocol</b></a> of <a
 *        href="http://www.bioimageanalysis.org/icy/doc/index.html"
 *        ><b>ICY</b></a>.
 * 
 */
public class Parameters {
	final private boolean _isHeadLess; // HeadLess means that the plugin is
										// launch through its normal version,
										// otherwise is through Protocol
	/**
	 * List of all the <a
	 * href="http://icy.bioimageanalysis.org/plugin/EzPlug_SDK"
	 * ><b>EzPlug</b></a> variable containers.
	 */
	final public HashMap<ParametersCategory, List<EzComponent>> ezVariables = new HashMap<ParametersCategory, List<EzComponent>>();

	/**
	 * List of all the <a
	 * href="http://icy.bioimageanalysis.org/plugin/Protocols"
	 * ><b>Protocol</b></a> variable containers.
	 */
	final public HashMap<ParametersCategory, List<Var<?>>> variables = new HashMap<ParametersCategory, List<Var<?>>>();

	// Global ezVariables
	private final EzVarSequence _ezVar_sequence;

	public Sequence sequence() {
		return _ezVar_sequence.getValue();
	}

	/**
	 * < <a href=
	 * "http://www.bioimageanalysis.org/icy/doc/classicy_1_1sequence_1_1_sequence.html"
	 * ><b>Sequence</b></a> used as input for the algorithm.
	 */

	private final EzVarPair.Integer _ezVar_rRange;

	public int rMin() {
		return (int) _ezVar_rRange.getValue().getFirst();
	}

	public int rMax() {
		return (int) _ezVar_rRange.getValue().getSecond();
	}

	/** < Minimal radius of a cell. */

	// Binarization ezVariables
	private final EzVarEnum<TargetBrightness> _ezVar_targetBrightness;

	public TargetBrightness targetBrightness() {
		return _ezVar_targetBrightness.getValue();
	}

	private final EzVarDouble _ezVar_threshold;

	public double threshold() {
		return _ezVar_threshold.getValue();
	}

	// Optimization ezVariables
	private final EzVarBoolean _ezVar_optimizeDescriptors;

	public boolean optimizeDescriptors() {
		return _ezVar_optimizeDescriptors.getValue();
	}

	private final EzVarInteger _ezVar_maxIter;

	/**
	 * Maximal number of iterations during the optimization step.
	 * 
	 * @see @ref Optimization::optimize
	 */
	public int maxIter() {
		return _ezVar_maxIter.getValue();
	}

	/**
	 * Constructor
	 * 
	 * @param isHeadLess
	 *            If it's true, the plug-in is called directly from <a
	 *            href="http://www.bioimageanalysis.org/icy/doc/index.html"
	 *            ><b>ICY</b></a>, else it's launch from <a
	 *            href="http://icy.bioimageanalysis.org/plugin/Protocols"
	 *            ><b>Protocol</b></a>.
	 */
	public Parameters(final BlobPlugin parent, boolean isHeadLess) {
		_isHeadLess = isHeadLess;
		for (ParametersCategory category : ParametersCategory.values()) {
			ezVariables.put(category, new ArrayList<EzComponent>());
			variables.put(category, new ArrayList<Var<?>>());
		}

		// Global Parameters
		_ezVar_sequence = newEzVarSequence(ParametersCategory.Global,
				"Sequence");
		_ezVar_rRange = newEzVarIntegerRange(ParametersCategory.Global,
				"Radius Range", new Pair.Integer(3, 9999), 1, 9999, 1,
				"Expected radius range of a cell.");

		// Binarization Parameters
		_ezVar_targetBrightness = newEzVarEnum(ParametersCategory.Binarization,
				"Target Brightness", TargetBrightness.Bright,
				TargetBrightness.values(), "Brightness of the cells");
		_ezVar_threshold = newEzVarDouble(ParametersCategory.Binarization,
				"Threshold", 0.25, 0.0, 1.0, 0.001,
				"Threshold used during the binarization step.");
		
		// Optimization Parameters
		_ezVar_optimizeDescriptors = new EzVarBoolean("Optimize Descriptors",
				true);
		if (_isHeadLess) {
			_ezVar_optimizeDescriptors
					.setToolTipText("Optimize the cell descriptors.");
		}
		ezVariables.get(ParametersCategory.Optimization).add(
				_ezVar_optimizeDescriptors);
		variables.get(ParametersCategory.Optimization).add(
				_ezVar_optimizeDescriptors.getVariable());

		_ezVar_maxIter = newEzVarInteger(ParametersCategory.Optimization,
				"Maximum number of iterations", 200, 1, null, 1,
				"Maximal number of iterations during the optimization step.");

		// Add listeners
		if (isHeadLess) {
			_ezVar_sequence.getVariable().addListener(
					new VarListener<Sequence>() {
						@Override
						public void valueChanged(Var<Sequence> source,
								Sequence oldValue, Sequence newValue) {
							parent.valueChanged(source);
						}

						@Override
						public void referenceChanged(Var<Sequence> source,
								Var<? extends Sequence> oldReference,
								Var<? extends Sequence> newReference) {
						}
					});
			_ezVar_targetBrightness.getVariable().addListener(
					new VarListener<TargetBrightness>() {
						@Override
						public void valueChanged(Var<TargetBrightness> source,
								TargetBrightness oldValue,
								TargetBrightness newValue) {
							parent.valueChanged(source);
						}

						@Override
						public void referenceChanged(
								Var<TargetBrightness> source,
								Var<? extends TargetBrightness> oldReference,
								Var<? extends TargetBrightness> newReference) {
						}
					});
			_ezVar_threshold.getVariable().addListener(
					new VarListener<Double>() {
						@Override
						public void valueChanged(Var<Double> source,
								Double oldValue, Double newValue) {
							parent.valueChanged(source);
						}

						@Override
						public void referenceChanged(Var<Double> source,
								Var<? extends Double> oldReference,
								Var<? extends Double> newReference) {
						}
					});
		}
	}

	private EzVarDouble newEzVarDouble(ParametersCategory category,
			final String name, final Double defaultValue,
			final Double minValue, final Double maxValue,
			final Double stepValue, final String tooltip) {
		EzVarDouble var = new EzVarDouble(name);
		initEzVarNumeric(var, category, defaultValue, minValue, maxValue,
				stepValue, tooltip);
		return var;
	}

	private EzVarInteger newEzVarInteger(ParametersCategory category,
			final String name, final Integer defaultValue,
			final Integer minValue, final Integer maxValue,
			final Integer stepValue, final String tooltip) {
		EzVarInteger var = new EzVarInteger(name);
		initEzVarNumeric(var, category, defaultValue, minValue, maxValue,
				stepValue, tooltip);
		return var;
	}

	private EzVarPair.Integer newEzVarIntegerRange(ParametersCategory category,
			final String name, final Pair.Integer defaultValue, int minValue,
			int maxValue, int stepValue, final String tooltip) {
		EzVarPair.Integer var = new EzVarPair.Integer(
				name,
				new RangeRangeModel(defaultValue, minValue, maxValue, stepValue));
		initEzVarRange(var, category, defaultValue, minValue, maxValue,
				stepValue, tooltip);
		return var;
	}

	private void initEzVarRange(EzVarPair var,
			final ParametersCategory category, final Pair defaultValue,
			double minValue, double maxValue, double stepValue,
			final String tooltip) {
		var.setToolTipText(tooltip);
		var.setValue(defaultValue);
		var.setLinkingWord("to");

		ezVariables.get(category).add(var);
		variables.get(category).add(var.getVariable());
	}

	private <T extends Number> void initEzVarNumeric(EzVarNumeric<T> var,
			final ParametersCategory category, final T defaultValue,
			final Comparable<T> minValue, final Comparable<T> maxValue,
			final T stepValue, final String tooltip) {
		if (_isHeadLess) {
			var.setValues(defaultValue, minValue, maxValue, stepValue);
			var.setToolTipText(tooltip);
		}
		var.setValue(defaultValue);

		ezVariables.get(category).add(var);
		variables.get(category).add(var.getVariable());
	}

	private <E extends Enum<E>> EzVarEnum<E> newEzVarEnum(
			final ParametersCategory category, final String name,
			E defaultValue, final E[] values, final String tooltip) {
		EzVarEnum<E> var = new EzVarEnum<E>(name, values);
		if (_isHeadLess) {
			var.setToolTipText(tooltip);
		}
		var.setValue(defaultValue);

		ezVariables.get(category).add(var);
		variables.get(category).add(var.getVariable());

		return var;
	}

	private EzVarSequence newEzVarSequence(final ParametersCategory category,
			final String name) {
		EzVarSequence var = new EzVarSequence(name);

		ezVariables.get(category).add(var);
		variables.get(category).add(var.getVariable());

		return var;
	}
}
