package plugins.fab.manualtnt.toremove;

import java.awt.Rectangle;
import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;

/**
 * high-performance multidimensional real-value array
 **/

public class MArray implements Types
{
//	byte[]   bdata;
	short[]  bdata;
	short[]  sdata;
	int[]    idata;
	float[]  fdata;
	double[] ddata;

	int    datatype; // type of the internal array
	int    maxsize;  // the physical size of the array
	int    nelem;    // the real size of the occupation of the array
	double extperc;  // array extension percentage (>= 100%) at each re-allocation
	int    dim;      // dimension of the array
	int[]  dimSize;  // the size of each dimension
	protected int[]  scanSize; // sequential scanning size of each dimension

	double[] elemSize; // size of the element in each dimension
	Object property;

	public MArray (int type, double extperc, int[] size, double elemSize[], Object prop)
	{
		datatype = type;
		this.extperc = extperc;

		this.elemSize = new double[size.length];
		for (int i=0; i<size.length; i++) 
		{
			if (elemSize != null)
			{
				if (i<elemSize.length)
				{
					this.elemSize[i] = elemSize[i];
				}
				else
					this.elemSize[i] = 1.;
			}
			else
				this.elemSize[i] = 1.;
		}
		resize(size);
		setProperty(prop);
	}
	public MArray (int type, double extperc, int[] size, Object prop)
	{
		this(type, extperc, size, null, prop);
	}

	public MArray(int type, int[] size)
	{
		this(type, 1., size, null);
	}

	public MArray (int type, int n1)
	{
		this(type, 1., new int[]{n1}, null);
	}
	public MArray (int type, int n1, int n2)
	{
		this(type, 1., new int[]{n1, n2}, null);
	}
	public MArray (int type, int n1, int n2, int n3)
	{
		this(type, 1., new int[]{n1, n2, n3}, null);
	}
	public MArray (int type, int n1, int n2, int n3, int n4)
	{
		this(type, 1., new int[]{n1, n2, n3, n4}, null);
	}
	public MArray (int type, int n1, int n2, int n3, int n4, Object prop)
	{
		this(type, 1., new int[]{n1, n2, n3, n4}, prop);
	}
	public MArray (int type, int n1, int n2, int n3, Object prop)
	{
		this(type, 1., new int[]{n1, n2, n3}, prop);
	}
	public MArray (int type, int n1, int n2, Object prop)
	{
		this(type, 1., new int[]{n1, n2}, prop);
	}
	public MArray (int type, int n1, Object prop)
	{
		this(type, 1., new int[]{n1}, prop);
	}
	public MArray (MArray ma)
	{
		setMA(ma);
	}

	public void abs()
	{
		for (int i=0; i<nelem; i++) set(i, Math.abs(get(i))); 
	}
	public void addConst(double c)
	{
		for (int i=0; i<nelem; i++) set(i, c+get(i));
	}
	public void addMA(MArray ma)
	{
		int len = (nelem <= ma.getNelem() ? nelem : ma.getNelem());

		for (int i=0; i<len; i++) set(i, get(i)+ma.get(i));
	}
	
	public MArray getRectangularCopy(int[] min, int[] max)
	{
		int numDim = min.length;
		if (numDim!=max.length) return null;
		int[] sizes = new int[numDim];
		for (int i = 0; i<numDim; i++)
		{
			sizes[i] = Math.max(0, max[i]-min[i]+1);

		}
		MArray rectArray = new MArray(this.getType(), this.extperc, sizes, null);
		int maxX = 0, minX = 0, maxY = 0, minY = 0, maxZ = 0, minZ = 0, maxT = 0, minT = 0;
		switch(numDim)
		{
		case 4:
			minT = min[3]; maxT = max[3]; 
		case 3:
			minZ = min[2]; maxZ = max[2];
		case 2:
			minY = min[1]; maxY = max[1];
		case 1:
			minX = min[0]; maxX = max[0];
		}
		int t1 = 0;
		for (int t = minT; t<=maxT; t++)
		{
			int z1 = 0;
			for (int z = minZ; z <= maxZ; z++)
			{
				int y1 = 0;
				for (int y = minY; y <= maxY; y++)
				{
					int x1 = 0;
					for (int x = minX; x <= maxX;x++)
					{	
						rectArray.set(x1, y1, z1, t1, this.get(x, y, z, t));
						x1++;
					}
					y1++;
				}
				z1++;
			}
			t1++;
		}
		return rectArray;
	}

	public MArray getRectangularCopy(Rectangle bounds)
	{
		Rectangle b = (Rectangle) bounds.clone();
		if (b.x < 0)
			b.x=0;
		if (b.y < 0)
			b.y=0;
		if (b.width + b.x > dimSize[0])
			b.width = dimSize[0] - b.width;
		if (b.height + b.y > dimSize[1])
			b.height = dimSize[1] - b.height;

		int[] sizes = new int[dim];
		sizes[0] = b.width;
		sizes[1] = b.height;
		for (int i = 2; i<dim; i++)
			sizes[i] = dimSize[i];

		MArray rectArray = new MArray(this.getType(), this.extperc, sizes, null);

		switch(dim)
		{
		case 4:
			int c4 = 0;
			for (int c = 0; c< dimSize[3]; c++)
			{
				int a4 = 0;
				for (int a = 0; a < dimSize[2]; a++)
				{
					int y4 = 0;
					for (int y = b.y; y< b.y + b.height; y++)
					{
						int x4 = 0;
						for (int x = b.x; x < b.x + b.width; x++)
						{
							rectArray.set(x4, y4, a4, c4, this.get(x, y,a,c));
							x4++;
						}
						y4++;
					}
					a4++;
				}
				c4++;
			}
			break;
		case 3:
			int a3 = 0;
			for (int a = 0; a < dimSize[2]; a++)
			{
				int y3 = 0;
				for (int y = b.y; y< b.y + b.height; y++)
				{
					int x3 = 0;
					for (int x = b.x; x < b.x + b.width; x++)
					{
						rectArray.set(x3, y3, a3, this.get(x, y,a));
						x3++;
					}
					x3 = 0;
					y3++;
				}
				a3++;
			}
			break;
		case 2:
			int y2 = 0;
			for (int y = b.y; y< b.y + b.height; y++)
			{
				int x2 = 0;
				for (int x = b.x; x < b.x + b.width; x++)
				{
					rectArray.set(x2, y2, this.get(x, y));
					x2++;
				}
				x2++;
			}
			break;
		default:
			return null;

		}
		return rectArray;
	}

	public void deoscillate()
	{
		for (int i=0; i<nelem; i++) if (get(i)<0)set(i,0); 
	}
	public void divConst(double c)
	{
		for (int i=0; i<nelem; i++)	set(i, get(i)/c);
	}
	public void divMA(MArray ma)
	{
		int len = (nelem <= ma.getNelem() ? getNelem() : ma.getNelem());

		for (int i=0; i<len; i++) set(i, get(i)/ma.get(i));
	}	
	public void divMA(MArray ma, double divzero)
	{
		double val;
		int len = (nelem <= ma.getNelem() ? nelem : ma.getNelem());

		for (int i=0; i<len; i++) 
		{
			val = ma.get(i);
			set(i, val == 0 ? divzero : get(i)/val);
		}
	}
	public double get(int x1)
	{
		switch(datatype)
		{
		case Types.BYTE   : return bdata[x1];
		case Types.SHORT  : return sdata[x1];
		case Types.INT    : return idata[x1];
		case Types.FLOAT  : return fdata[x1];
		case Types.DOUBLE : return ddata[x1];
		default : throw new IllegalArgumentException("MArray::get:Unknown datatype = " + datatype);
		}		
	}
	public double get(int x1, int x2)
	{
		switch(datatype)
		{
		case Types.BYTE   : return bdata[x1+x2*scanSize[0]];
		case Types.SHORT  : return sdata[x1+x2*scanSize[0]];
		case Types.INT    : return idata[x1+x2*scanSize[0]];
		case Types.FLOAT  : return fdata[x1+x2*scanSize[0]];
		case Types.DOUBLE : return ddata[x1+x2*scanSize[0]];
		default : throw new IllegalArgumentException("MArray::get:Unknown datatype = " + datatype);
		}		
	}
	public double get(int x1, int x2, int x3)
	{
		switch(datatype)
		{
		case Types.BYTE   : return bdata[x1+x2*scanSize[0]+x3*scanSize[1]];
		case Types.SHORT  : return sdata[x1+x2*scanSize[0]+x3*scanSize[1]];
		case Types.INT    : return idata[x1+x2*scanSize[0]+x3*scanSize[1]];
		case Types.FLOAT  : return fdata[x1+x2*scanSize[0]+x3*scanSize[1]];
		case Types.DOUBLE : return ddata[x1+x2*scanSize[0]+x3*scanSize[1]];
		default : throw new IllegalArgumentException("MArray::get:Unknown datatype = " + datatype);
		}		
	}
	public double get(int x1, int x2, int x3, int x4)
	{
		switch(datatype)
		{
		case Types.BYTE   : return bdata[x1+x2*scanSize[0]+x3*scanSize[1]+x4*scanSize[2]];
		case Types.SHORT  : return sdata[x1+x2*scanSize[0]+x3*scanSize[1]+x4*scanSize[2]];
		case Types.INT    : return idata[x1+x2*scanSize[0]+x3*scanSize[1]+x4*scanSize[2]];
		case Types.FLOAT  : return fdata[x1+x2*scanSize[0]+x3*scanSize[1]+x4*scanSize[2]];
		case Types.DOUBLE : return ddata[x1+x2*scanSize[0]+x3*scanSize[1]+x4*scanSize[2]];
		default : throw new IllegalArgumentException("MArray::get:Unknown datatype = " + datatype);
		}		
	}
	public double get(int x1, int x2, int x3, int x4, String borderType)
	{
		if (borderType.equals(Types.ZERO_PAD))
		{
			if (x1<0 || x1>=dimSize[0] 
			                        || x2<0 || x2>=dimSize[1]
			                                               || x3<0 || x3>=dimSize[2]
			                                                                      || x4<0 || x4>=dimSize[3]) return 0.;
		}
		else return get(indexBorder(x1, 0, borderType), 
				indexBorder(x2, 1, borderType), 
				indexBorder(x3, 2, borderType),
				indexBorder(x4, 3, borderType));

		return get(x1, x2, x3, x4);
	}

	public double get(int x1, int x2, int x3, String borderType)
	{
		if (borderType.equals(Types.ZERO_PAD))
		{
			if (x1<0 || x1>=dimSize[0] 
			                        || x2<0 || x2>=dimSize[1]
			                                               || x3<0 || x3>=dimSize[2]) return 0.;
		}
		else return get(indexBorder(x1, 0, borderType), 
				indexBorder(x2, 1, borderType), 
				indexBorder(x3, 2, borderType));

		return get(x1, x2, x3);
	}
	public double get(int x1, int x2, String borderType)
	{
		if (borderType.equals(Types.ZERO_PAD))
		{
			if (x1<0 || x1>=dimSize[0] 
			                        || x2<0 || x2>=dimSize[1]) return 0.;
		}
		else return get(indexBorder(x1, 0, borderType), indexBorder(x2, 1, borderType));

		return get(x1, x2);
	}
	public double get(int x1, String borderType)
	{
		if (borderType.equals(Types.ZERO_PAD))
		{
			if (x1<0 || x1>=dimSize[0]) return 0.;
		}
		else return get(indexBorder(x1, 0, borderType));

		return get(x1);
	}
	public double get(int[] x)
	{
		int p = x[0];
		for (int i=1; i<x.length; i++)
			p += x[i] * scanSize[i-1];
		switch(datatype)
		{
		case Types.BYTE   : return bdata[p];
		case Types.SHORT  : return sdata[p];
		case Types.INT    : return idata[p];
		case Types.FLOAT  : return fdata[p];
		case Types.DOUBLE : return ddata[p];
		default : throw new IllegalArgumentException("MArray::get:Unknown datatype = " + datatype);
		}		
	}
	public double get(int[] x, String borderType, boolean modifiable)
	{
		int[] xnew = null;
		if (!modifiable) xnew = new int[x.length];
		else xnew = x;

		for (int i=0; i<x.length; i++)
		{
			if (borderType.equals(Types.ZERO_PAD))
			{
				if (x[i]<0 || x[i]>=dimSize[i]) return 0.;
				else xnew[i] = x[i];
			}
			else xnew[i] = indexBorder(x[i], i, borderType);
		}

		return get(xnew);
	}

	public double getAbsSum()
	{
		double sum = 0;

		for (int i=0; i<nelem; i++) sum += Math.abs(get(i));

		return sum;
	}
	public int getDim()
	{
		return dim;
	}
	public int[] getDimSize()
	{
		int[] ds = new int[dim];
		for (int i=0; i<dim; i++)
			ds[i] = dimSize[i];

		return ds;
	}
	public int getDimSize(int i)
	{
		return dimSize[i];
	}
	public double[] getElemSize()
	{
		double[] es = new double[dim];
		for (int i=0; i<dim; i++)
			es[i] = elemSize[i];

		return es;
	}
	public double getElemSize(int i)
	{
		return elemSize[i];
	}
	public double getExtperc()
	{
		return extperc;
	}
	// p == -1 means p == +infty
	public double getlpNorm (double p)
	{
		double sum = 0;

		if (p == 1) return getAbsSum();
		if (p == -1) return Math.max(Math.abs(getMin()), Math.abs(getMax()));

		for (int i=0; i<nelem; i++) 
			sum += Math.pow(Math.abs(get(i)), p);

		return Math.pow(sum, 1./p);
	}	
	// p == -1 means p == +infty
	public double getLpNorm (double p)
	{
		double vol = 1.;
		for (int i=0; i<dim; i++) vol *= elemSize[i];

		if (p == 1) return (getAbsSum() * vol);
		if (p == -1) return Math.max(Math.abs(getMin()), Math.abs(getMax()));

		double sum = 0;
		for (int i=0; i<nelem; i++) 
			sum += Math.pow(Math.abs(get(i)), p);

		sum *= vol;

		return Math.pow(sum, 1./p);
	}	
	public double getMax()
	{
		double m = Double.NEGATIVE_INFINITY;
		double val;

		for (int i=0; i<nelem; i++) 
		{ 
			val = get(i);   
			if (val>m) m = val; 
		}
		return m;
	}	
	public int getMaxsize()
	{
		return maxsize;
	}
	public double getMean()
	{
		return getSum() / nelem;
	}
	public double getMedian()
	{
		// calcule the median by Torben's algorithm
		double medv;
		int less=0, greater=0, equal=0;
		double min, max;
		double guess=0., maxltguess=0., mingtguess=0.;
		boolean done = false;
		double vv;

		min = Double.POSITIVE_INFINITY; 
		max = Double.NEGATIVE_INFINITY;
		for (int x=0; x<nelem; x++)
		{
			vv = get(x);
			if (vv<min) min = vv;
			if (vv>max) max = vv;
		}

		while (!done)
		{
			guess = (min+max)/2.;
			less = 0;
			greater = 0;
			equal = 0;
			maxltguess = min;
			mingtguess = max;

			for (int x=0; x<nelem; x++)
			{
				vv = get(x);
				if (vv<guess)
				{
					less++;
					if (vv>maxltguess) maxltguess = vv;
				} 
				else if (vv>guess)
				{
					greater++;
					if (vv<mingtguess) mingtguess = vv;
				} 
				else equal++;
			}

			if (less<=(nelem+1)/2 && greater<=(nelem+1)/2) done = true;
			else if (less>greater) max = maxltguess;
			else min = mingtguess;
		}
		if (less >= (nelem+1)/2) medv = maxltguess;
		else if (less+equal >= (nelem+1)/2) medv = guess;
		else medv = mingtguess;

		return medv;
	}
	public double getMin()
	{
		double m = Double.POSITIVE_INFINITY;
		double val;

		for (int i=0; i<nelem; i++) 
		{ 
			val = get(i);   
			if (val<m) m = val; 
		}
		return m;
	}
	public int getNelem()
	{
		return nelem;
	}
	public Object getProperty()
	{
		return property;
	}

	public double getStd()
	{
		double sum = 0, sum2 = 0;
		double val;

		for (int i=0; i<nelem; i++) 
		{ 
			val = get(i); 
			sum2 += val*val; 
			sum += val; 
		}

		return nelem <= 1 ? 0 : Math.sqrt((sum2 - sum*sum/nelem) / (nelem - 1));
	}
	
	public double getVar()
	{
		if (nelem <=1)
			return 0;
		double sum = 0, sum2 = 0;
		double val;

		for (int i=0; i<nelem; i++) 
		{ 
			val = get(i); 
			sum2 += val*val; 
			sum += val; 
		}

		return (sum2 - sum*sum/nelem) / (nelem - 1);
	}
	
	/**
	 * 
	 * @return the mad ( mean average distance )
	 */	
	public double getMeanAverageDistance( )
	{
		double mean = getMean();
		double a = 0;
		double s ;
				
		for (int i=0; i<nelem; i++) 
		{ 
			s = get(i) - mean;
			a=a+ Math.abs( s );
		}
		
		return nelem <= 1 ? 0 : (( a/nelem ));		
	}
	
	public double[] getFirstFourMoments()
	{
		double[] moments = new double[4];
		
		double sum = 0, sum2 = 0;
		double val;

		for (int i=0; i<nelem; i++) 
		{ 
			val = get(i); 
			sum2 += val*val; 
			sum += val; 
		}
		moments[0] = sum/nelem;//mean
		double mean = moments[0];
		moments[1] = (sum2 - sum*sum/nelem) / (nelem - 1);//variance
		double sum3 = 0;
		double sum4 = 0;
		for (int i=0; i<nelem; i++) 
		{ 
			val = get(i); 
			sum3 += Math.pow(val -mean , 3);
			sum4 += Math.pow(val -mean , 4); 
		}
		moments[2] = sum3/nelem;
		moments[3] = sum4/nelem;
		return moments;
	}
	
	public double getStd(double mean)
	{
		double diff2 = 0;
		double val;

		for (int i=0; i<nelem; i++) 
		{ 
			val = get(i); 
			diff2 += (mean-val)*(mean-val); 

		}		
		return nelem <= 1 ? 0 : Math.sqrt(diff2 / (nelem - 1));
	}

	public double getSum()
	{
		double sum = 0;

		for (int i=0; i<nelem; i++) sum += get(i);

		return sum;
	}
	public double getSum2()
	{
		double sum = 0;

		for (int i=0; i<nelem; i++) sum += Math.pow(get(i),2);

		return sum;
	}
	public int getType()
	{
		return datatype;
	}
	/** index of data point while taking the border into account
	 *  @param x the coordinate value
	 *  @param dim the concerned dimension (0,1,2...)
	 *  @param borderType the type of border strategy (CONT_EXT or SYM_EXT)
	 */ 
	int indexBorder(int x, int dim, String borderType)
	{
		if (x>=0 && x<dimSize[dim]) return x;

		if (borderType.equals(Types.CONT_EXT))
		{
			return ((x<0) ? 0 : dimSize[dim]-1);
		}
		else if (borderType.equals(Types.SYM_EXT))
		{
			return ((x<0) ? -x : 2*(dimSize[dim]-1)-x);
		}

		return x;
	}
	public void mulConst(double c)
	{
		for (int i=0; i<nelem; i++)	set(i, c*get(i));
	}

	public void mulMA(MArray ma)
	{
		int len = (nelem <= ma.getNelem() ? nelem : ma.getNelem());

		for (int i=0; i<len; i++) set(i, get(i)*ma.get(i));
	}

	public void not()
	{
		for (int i=0; i<nelem; i++) set(i, (get(i) == 0) ? 1 : 0);
	}
	public void resize(int n1)
	{
		resize(new int[]{n1});
	}
	public void resize(int n1, int n2)
	{
		resize(new int[]{n1, n2});
	}


	public void resize(int n1, int n2, int n3)
	{
		resize(new int[]{n1, n2, n3});
	}
	public void resize(int n1, int n2, int n3, int n4)
	{
		resize(new int[]{n1, n2, n3, n4});
	}
	public void resize(int[] size)
	{		
		int newnelem = 0;

		dim = 0;
		for (int i=0; (i<size.length && size[i]>0); i++) 
		{
			if (i == 0) newnelem = 1;
			dim++;
			newnelem *= size[i];
		}

		int maxDim = Math.max(dim, 4);
		if (dimSize == null) dimSize = new int[maxDim];
		if (dimSize == null) dimSize = new int[maxDim];
		else if (dimSize.length < maxDim) dimSize = new int[dim];

		for (int i=0; i<dim; i++) dimSize[i] = size[i];	
		for (int i=dim; i<maxDim; i++) dimSize[i] = 1;

		if (newnelem > maxsize)
		{
			maxsize = Math.max((int) (newnelem * extperc), newnelem);
			switch(datatype)
			{
//			case Types.BYTE   : bdata = new byte[maxsize]; break;
			case Types.BYTE   : bdata = new short[maxsize]; break;
			case Types.SHORT  : sdata = new short[maxsize]; break;
			case Types.INT    : idata = new int[maxsize]; break;
			case Types.FLOAT  : fdata = new float[maxsize];break;
			case Types.DOUBLE : ddata = new double[maxsize]; break;
			default : throw new IllegalArgumentException("MArray::resize:Unknown datatype = " + datatype);
			}							
		}
		nelem = newnelem;
		if (dim >= 2)
		{
			if (scanSize == null) scanSize = new int[maxDim-1];
			else if (scanSize.length < maxDim-1) scanSize = new int[maxDim-1];
			scanSize[0] = dimSize[0];
			for (int i=3; i<=maxDim; i++)
				scanSize[i-2] = scanSize[i-3] * dimSize[i-2];
		}
		else scanSize = null;

		if (elemSize == null)
		{
			elemSize = new double[maxDim];
			for (int i=0; i<maxDim; i++) elemSize[i] = 1.;
		}
		else if (elemSize.length != maxDim)
		{
			double[] es = new double[maxDim];
			for (int i=0; i<Math.min(maxDim, elemSize.length); i++) es[i] = elemSize[i];
			for (int i=Math.min(maxDim, elemSize.length); i<maxDim; i++) es[i] = 1.;
			elemSize = es;
		}
	}
	public void set(int x1, double v)
	{
		switch(datatype)
		{
		case Types.BYTE   : bdata[x1] = (byte) v; break;
		case Types.SHORT  : sdata[x1] = (short) v; break;
		case Types.INT    : idata[x1] = (int) v; break;
		case Types.FLOAT  : fdata[x1] = (float) v; break;
		case Types.DOUBLE : ddata[x1] = v; break;
		default : throw new IllegalArgumentException("MArray::set:Unknown datatype = " + datatype);
		}		
	}
	public void set(int x1, int x2, double v)
	{
		switch(datatype)
		{
		case Types.BYTE   : bdata[x1+x2*scanSize[0]] = (byte) v; break;
		case Types.SHORT  : sdata[x1+x2*scanSize[0]] = (short) v; break;
		case Types.INT    : idata[x1+x2*scanSize[0]] = (int) v; break;
		case Types.FLOAT  : fdata[x1+x2*scanSize[0]] = (float) v; break;
		case Types.DOUBLE : ddata[x1+x2*scanSize[0]] =  v; break;
		default : throw new IllegalArgumentException("MArray::set:Unknown datatype = " + datatype);
		}		
	}

	public void set(int x1, int x2, int x3, double v)
	{
		switch(datatype)
		{
//		case Types.BYTE   : bdata[x1+x2*scanSize[0]+x3*scanSize[1]] = (byte) v; break;
		case Types.BYTE   : bdata[x1+x2*scanSize[0]+x3*scanSize[1]] = (short) v; break;		
		case Types.SHORT  : sdata[x1+x2*scanSize[0]+x3*scanSize[1]] = (short) v; break;
		case Types.INT    : idata[x1+x2*scanSize[0]+x3*scanSize[1]] = (int) v; break;
		case Types.FLOAT  : fdata[x1+x2*scanSize[0]+x3*scanSize[1]] = (float) v; break;
		case Types.DOUBLE : ddata[x1+x2*scanSize[0]+x3*scanSize[1]] =  v; break;
		default : throw new IllegalArgumentException("MArray::set:Unknown datatype = " + datatype);
		}		
	}
	public void set(int x1, int x2, int x3, int x4, double v)
	{
		switch(datatype)
		{
//		case Types.BYTE   : bdata[x1+x2*scanSize[0]+x3*scanSize[1]+x4*scanSize[2]] = (byte) v; break;
		case Types.BYTE   : bdata[x1+x2*scanSize[0]+x3*scanSize[1]+x4*scanSize[2]] = (short) v; break;		
		case Types.SHORT  : sdata[x1+x2*scanSize[0]+x3*scanSize[1]+x4*scanSize[2]] = (short) v; break;
		case Types.INT    : idata[x1+x2*scanSize[0]+x3*scanSize[1]+x4*scanSize[2]] = (int) v; break;
		case Types.FLOAT  : fdata[x1+x2*scanSize[0]+x3*scanSize[1]+x4*scanSize[2]] = (float) v; break;
		case Types.DOUBLE : ddata[x1+x2*scanSize[0]+x3*scanSize[1]+x4*scanSize[2]] = v; break;
		default : throw new IllegalArgumentException("MArray::set:Unknown datatype = " + datatype);
		}		
	}
	public void set(int[] x, double v)
	{
		int p = x[0];
		for (int i=1; i<x.length; i++)
			p += x[i] * scanSize[i-1];
		switch(datatype)
		{
//		case Types.BYTE   : bdata[p] = (byte) v; break;
		case Types.BYTE   : bdata[p] = (short) v; break;		
		case Types.SHORT  : sdata[p] = (short) v; break;
		case Types.INT    : idata[p] = (int) v; break;
		case Types.FLOAT  : fdata[p] = (float) v; break;
		case Types.DOUBLE : ddata[p] = v; break;
		default : throw new IllegalArgumentException("MArray::set:Unknown datatype = " + datatype);
		}		
	}
	public void setConst(double c)
	{
		for (int i=0; i<nelem; i++) set(i, c);
	}
	public void setElemSize(double[] es)
	{
		int len = Math.min(es.length, dim);
		for (int i=0; i<len; i++) elemSize[i] = es[i];
	}

	public void setMA(MArray ma)
	{
		if (ma.getType() != datatype)
		{
			datatype = ma.getType();
			maxsize = 0;
			nelem = 0;
			dim = 0;
			dimSize = null;
			scanSize = null;
			bdata = null; 
			sdata = null; 
			idata = null; 
			fdata = null; 
			ddata = null;
		}
		extperc = ma.getExtperc();
		int[] dims = new int[ma.getDim()];
		int[] maDimSize = ma.getDimSize();
		for (int i=0; i<dims.length; i++) dims[i] = maDimSize[i];
		resize(dims);

		for (int i=0; i<nelem; i++) set(i, ma.get(i));

		setElemSize(ma.getElemSize());
		setProperty(ma.getProperty());
	}

	public void setProperty(Object prop)
	{
		property = prop;
	}

	public void sqr()
	{
		double val;

		for (int i=0; i<nelem; i++) 
		{ 
			val = get(i); 
			set(i, val*val); 
		}
	}

	public void sqrt()
	{
		for (int i=0; i<nelem; i++) set(i, Math.sqrt(get(i)));
	}

	public void sqrt(double negvalue)
	{
		double val;

		for (int i=0; i<nelem; i++) 
		{ 
			val = get(i); 
			set(i, (val<0 ? negvalue : Math.sqrt(val)));
		}
	}

	/**
	 * subMA( inputMA )
	 * result is : this - inputMA;   ( substract current MA with argument )
	 * @param ma
	 */
	public void subMA(MArray ma)
	{
		int len = (nelem <= ma.getNelem() ? nelem : ma.getNelem());

		for (int i=0; i<len; i++) set(i, get(i)-ma.get(i));
	}

	public void toFile(String filename)
	{
		try {
			BufferedWriter out = new BufferedWriter(new FileWriter(filename));
			out.write(toString()+"\n");
			for (int cnt=0; cnt<nelem; cnt++)
			{
				out.write(get(cnt)+" ");
			}
			out.close();
		} catch (IOException e) {
		}
	}
	public String toString()
	{
		String s = "MArray : datatype=" + datatype + " nelem=" + nelem + " usage=" + (int)(100.*nelem/maxsize) + "%\n";
		for (int i=0; i<dimSize.length; i++)
			s += " dimSize" + (i+1) + "=" + dimSize[i];
		s += "\n";
		for (int i=0; i<elemSize.length; i++)
			s += " elemSize" + (i+1) + "=" + elemSize[i];
		s += "\n";
		s += "Min=" + getMin();
		s += " Max=" + getMax();
		s += " Mean=" + getMean();
		s += " Std=" + getStd();
		if (property != null)
			s += "\n Prop=" + property.toString();

		return s;
	}
}