"""
Functions to create Java proxy classes for Python classes.

Copyright 2013 Institut Pasteur.
 
This file is part of ICY.
 
ICY is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.
 
ICY is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
 
You should have received a copy of the GNU General Public License
along with ICY. If not, see <http://www.gnu.org/licenses/>.

Created on Tue Nov 14

@author: tlecomte
"""

import site
import os
import subprocess
import sys
import tempfile
import inspect
import importlib
import shutil

from icyclamp import SerializableProxies

from java.util.jar import JarOutputStream
from java.io import FileOutputStream
from java.util.zip import ZipEntry


def produce_proxy_class(module_name, class_name, importPath, outdir):
    """
    Produces a .class file containing a Java proxy of the input Python class.
    
    This file can be used by Java without further compilation. It will get
    its package name from the definition of __proxymaker__ in the class definition.
    """
    SerializableProxies.serialized_path = outdir
    if importPath not in sys.path:
        sys.path.insert(0, importPath)
        
    # if the module has already been imported, reload it so that the class is produced
    # otherwise, simply import it
    if module_name in sys.modules:
        mod = reload(sys.modules[module_name])
    else:
        mod = importlib.import_module(module_name)
    
    plugin = getattr(mod, class_name)
    
    return inspect.getsourcefile(mod)


def produce_jar_jdk(outdir, jarname):
    """
    Produce a jar from a directory, using the 'jar' utility.
    
    Since the 'jar' utility is only available with the JDK,
    this method would not work on the JRE. 
    """
    subprocess.check_call(["jar", "cf", jarname, "-C", outdir, "."])
    return jarname


def add_to_jar(jarOut, entry, prefix=None):
    # entry must be an absolute path on disk
    
    # handle root case
    if prefix == None and os.path.isdir(entry):
        for subentry in os.listdir(entry):
            add_to_jar(jarOut, os.path.join(entry, subentry), "")
        return    
    
    # non-root
    jarSep = '/'
    jarEntry = prefix + os.path.basename(entry)  # path relative to the jar root
    if os.path.isdir(entry):
        jarOut.putNextEntry(ZipEntry(jarEntry + jarSep))
        for subentry in os.listdir(entry):
            add_to_jar(jarOut, os.path.join(entry, subentry), jarEntry + jarSep)
    else:
        jarOut.putNextEntry(ZipEntry(jarEntry))
        bytes = open(entry, "rb").read()
        jarOut.write(bytes)
        jarOut.closeEntry()
	

def produce_jar(outdir, jarname):
    """
    Produce a jar from a directory
    
    This function does not use the 'jar' utility, so it does work on the JRE. 
    """
    fout = FileOutputStream(jarname)
    jarOut = JarOutputStream(fout)
    add_to_jar(jarOut, outdir)
    jarOut.close()
    fout.close()
    return jarname


def create_proxy_jar(moduleName, className, importPath, jarName):
	"""Create a jar containing what is needed for a plugin"""

	outdir = tempfile.mkdtemp()
	
	filename = produce_proxy_class(moduleName, className, importPath, outdir)

	pydir = os.path.join(outdir, "PythonLibs", "site-packages")
	os.makedirs(pydir)
	shutil.copy(filename, pydir)
	
	print "Done serializing to %s" %(outdir)
	print "Module source file is %s" %(filename)
	
	produce_jar(outdir, jarName)
	
	print "Jar created: %s" %(jarName)

	shutil.rmtree(outdir)


def create_proxyClasses_jar(pluginClasses, folder, jarName):
    """Create a jar containing what is needed for a plugin"""

    outdir = tempfile.mkdtemp()
    
    for klass in pluginClasses:
        produce_proxy_class(klass.moduleName, klass.klass.__name__, klass.dirName, outdir)

    print "Done serializing proxy classes to %s" %(outdir)

    pydir = os.path.join(outdir, "PythonLibs", "site-packages")
    os.makedirs(pydir)

    for f in os.listdir(folder):
        root, ext = os.path.splitext(f)
        if ext == ".py":
            filename = os.path.join(folder, f)
            print "Copying Python file %s" %(filename)
            shutil.copy(filename, pydir)    
    
    produce_jar(outdir, jarName)
    
    print "Jar created: %s" %(jarName)

    shutil.rmtree(outdir)


def main():
	thisdir = os.path.dirname(os.path.abspath(inspect.getsourcefile(lambda _: None)))
	jarName = os.path.join(thisdir, "proxies.jar")
	create_proxy_jar("pythonThresholderExample", "PythonThresholderExample", thisdir, jarName)

if __name__ == "__main__":
    main()

