package plugins.tlecomte.pythonPackager;

/*
 * Copyright 2013 Institut Pasteur.
 * 
 * This file is part of ICY.
 * 
 * ICY is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * ICY is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with ICY. If not, see <http://www.gnu.org/licenses/>.
 */

import java.io.File;

import icy.file.FileUtil;
import icy.gui.dialog.ConfirmDialog;
import icy.gui.dialog.MessageDialog;

import org.python.core.PyStringMap;
import org.python.core.PySystemState;
import org.python.util.PythonInterpreter;

import plugins.adufour.ezplug.EzLabel;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzVarFolder;
import plugins.adufour.ezplug.EzVarText;
import plugins.tlecomte.pythonExtractor.PythonLibraries;

@PythonLibraries
public class PythonPluginPackager extends EzPlug {

	EzLabel description = new EzLabel("Choose the folder where the Python plugins are located."
			+ "\n\nThe Plugin classes will be automatically looked for in that folder,"
			+ " and Java proxies will be produced for them."
			+ "\n\nIn addition to that, all the Python scripts found at the"
			+ " root of this folder will be included in the jar. This means that the"
			+ " main Python script can depend on other Python files in this folder.");
	
	EzVarFolder folderChooser = new EzVarFolder("Plugin folder:", null);
	EzLabel jarDescription = new EzLabel("\nChoose the name of the Jar file that will be produced"
			+ " (which will be created in the Plugin folder):");
	EzVarText jarFileChooser = new EzVarText("Jar output file:\n", "myplugin.jar");

	@Override
	public void clean() {
		
		
	}

	@Override
	protected void execute() {    
	    if (folderChooser.getValue() == null) {
	    	// user did not select a folder
	    	MessageDialog.showDialog("Please select the input folder where your Python plugins are located.", MessageDialog.ERROR_MESSAGE );
	    	return;
	    }

	    String jarName = jarFileChooser.getValue(); 
		if (jarName == null || jarName == "") {
			// user did not select a jar output file
			MessageDialog.showDialog("Please select a name for the Jar file that will be created.", MessageDialog.ERROR_MESSAGE );
			return;
		}

	    String folderPath = folderChooser.getValue().getAbsolutePath();
	    String jarPath = folderPath + FileUtil.separator + jarName;
	    
	    // force the .jar extension
	    if (!jarPath.endsWith(".jar")) {
	    	jarPath += ".jar";
	    }
	    
	    // test for jarFile existence and asks the user if it is ok to overwrite
	    File jarFile = new File(jarPath);
	    if (jarFile.exists()) {
	    	boolean overwriteConfirm = ConfirmDialog.confirm("Overwrite jar file ?", "The jar file " + jarPath + " already exists. Overwrite ?", ConfirmDialog.OK_CANCEL_OPTION);
	    	
	    	if (!overwriteConfirm) {
	    		return;
	    	}
	    }

	    PythonInterpreter py = new PythonInterpreter(new PyStringMap(), new PySystemState());
	     
		py.set("folder", folderPath);
	    py.set("jarName", jarPath);
	    
		py.exec("import icypluginanalyzer");
		py.exec("classes = icypluginanalyzer.analyse_folder(folder)");
		py.exec("validClasses = icypluginanalyzer.validateClasses(classes)");

		py.exec("print \"Valid classes:\"");
		py.exec("print validClasses");
		py.exec("validClassesCount = len(validClasses)");

		Integer validClassesCount = 0;
		validClassesCount = py.get("validClassesCount", validClassesCount.getClass());
		
		if (validClassesCount == 0) {
			MessageDialog.showDialog("No valid classes have been found in this folder", MessageDialog.ERROR_MESSAGE );
            return;
		}
		
		py.exec("annotationcount = icypluginanalyzer.countAnnotations(validClasses)");
		
		Integer annotationCount = 0;
		annotationCount = py.get("annotationcount", annotationCount.getClass());
		
		if (annotationCount == 0) {
			String message;
			if (validClassesCount > 1) {
				message = "Please apply the PythonLibraries annotation to one of the plugin classes\n"
						+ " in this folder, to allow the Python scripts\n"
						+ " to be properly extracted and seen by Icy at runtime";
			} else {
				message = "Please apply the PythonLibraries annotation to the plugin class\n"
						+ " in this folder, to allow the Python scripts\n"
						+ " to be properly extracted and seen by Icy at runtime";
			}
			MessageDialog.showDialog(message, MessageDialog.ERROR_MESSAGE );
            return;
		}
		
		py.exec("import icypackager");
		py.exec("icypackager.create_proxyClasses_jar(validClasses, folder, jarName)");
		
		String message = String.format("The jar file has been written to:\n"
				+ "    %s\n\n"
				+ "When sending this jar to Icy website, please use one of the following classnames:\n",
				jarPath);
		
		py.exec("classNames = \"\"");
		py.exec("for klass in validClasses:\n"
				+ "    classNames += \"   \" + klass.getQualifiedName() + \"\\n\"");
		message += py.get("classNames", message.getClass());
		
		MessageDialog.showDialog(message, MessageDialog.INFORMATION_MESSAGE);		
	}

	@Override
	protected void initialize() {
		description.setNumberOfColumns(20);
		//description.setNumberOfRows(0);
		
	    addEzComponent(description);
	    addEzComponent(folderChooser);
	    addEzComponent(jarDescription);
	    addEzComponent(jarFileChooser);
	}
}
