package plugins.fab.mapoverlay;

import icy.canvas.IcyCanvas;
import icy.gui.frame.progress.AnnounceFrame;
import icy.gui.viewer.Viewer;
import icy.image.IcyBufferedImage;
import icy.image.IcyBufferedImageUtil;
import icy.image.colormap.IcyColorMap.IcyColorMapType;
import icy.painter.Anchor2D.Anchor2DListener;
import icy.painter.Overlay;
import icy.painter.PainterEvent;
import icy.roi.ROI;
import icy.sequence.Sequence;
import icy.sequence.SequenceEvent;
import icy.sequence.SequenceListener;
import icy.type.collection.array.Array1DUtil;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.geom.Path2D;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.List;

import plugins.kernel.roi.roi2d.ROI2DPolygon;

public class MapOverlay extends Overlay implements Anchor2DListener, SequenceListener
{
    Anchor2D p0;
    Anchor2D p0top3;
    Anchor2D p0top1;

    Anchor2D p1;
    Anchor2D p1top0;
    Anchor2D p1top2;

    Anchor2D p2;
    Anchor2D p2top1;
    Anchor2D p2top3;

    Anchor2D p3;
    Anchor2D p3top0;
    Anchor2D p3top2;

    IcyBufferedImage sourceImage = null;
    IcyBufferedImage finalImage = null;

    Sequence sourceSequence = null;
    Sequence outSequence = null;
    
    public MapOverlay( Sequence sequenceToDraw, Sequence sourceSequence, 
    		IcyBufferedImage sourceImage )
    {
    	super("Map Overlay");
    	
    	if ( sequenceToDraw == sourceSequence )
    	{
    		throw new IllegalArgumentException("Sequences must be different");
    	}

    	sourceSequence.addListener(this );
    	
    	this.setPriority( OverlayPriority.IMAGE_TOP );
        
        this.sourceSequence = sourceSequence;
        this.outSequence = sequenceToDraw;
        
        p0 = new Anchor2D( 0, 0);
        p0top3 = new Anchor2D( 0, (int) (0.3f * sequenceToDraw.getHeight()));
        p0top1 = new Anchor2D( (int) (0.3f * sequenceToDraw.getWidth()), 0);

        p1 = new Anchor2D( sequenceToDraw.getWidth(), 0);
        p1top0 = new Anchor2D( sequenceToDraw.getWidth() - (int) (0.3f * sequenceToDraw.getWidth()), 0);
        p1top2 = new Anchor2D( sequenceToDraw.getWidth(), (int) (0.3f * sequenceToDraw.getHeight()));

        p2 = new Anchor2D( sequenceToDraw.getWidth(), sequenceToDraw.getHeight());
        p2top1 = new Anchor2D( sequenceToDraw.getWidth(), sequenceToDraw.getHeight()
                - (int) (0.3f * sequenceToDraw.getHeight()));
        p2top3 = new Anchor2D( sequenceToDraw.getWidth() - (int) (0.3f * sequenceToDraw.getWidth()),
                sequenceToDraw.getHeight());

        p3 = new Anchor2D( 0, sequenceToDraw.getHeight());
        p3top0 = new Anchor2D( 0, sequenceToDraw.getHeight() - (int) (0.3f * sequenceToDraw.getHeight()));
        p3top2 = new Anchor2D( (int) (0.3f * sequenceToDraw.getWidth()), sequenceToDraw.getHeight());

        p0.setPriority( OverlayPriority.SHAPE_TOP );
        p0top3.setPriority( OverlayPriority.SHAPE_TOP );
        p0top1.setPriority( OverlayPriority.SHAPE_TOP );
        p1.setPriority( OverlayPriority.SHAPE_TOP );
        p1top0.setPriority( OverlayPriority.SHAPE_TOP );
        p1top2.setPriority( OverlayPriority.SHAPE_TOP );
        p2.setPriority( OverlayPriority.SHAPE_TOP );
        p2top1.setPriority( OverlayPriority.SHAPE_TOP );
        p2top3.setPriority( OverlayPriority.SHAPE_TOP );
        p3.setPriority( OverlayPriority.SHAPE_TOP );
        p3top0.setPriority( OverlayPriority.SHAPE_TOP );
        p3top2.setPriority( OverlayPriority.SHAPE_TOP );
                
        sequenceToDraw.addOverlay(p0);
        sequenceToDraw.addOverlay(p0top3);
        sequenceToDraw.addOverlay(p0top1);
        sequenceToDraw.addOverlay(p1);
        sequenceToDraw.addOverlay(p1top0);
        sequenceToDraw.addOverlay(p1top2);
        sequenceToDraw.addOverlay(p2);
        sequenceToDraw.addOverlay(p2top1);
        sequenceToDraw.addOverlay(p2top3);
        sequenceToDraw.addOverlay(p3);
        sequenceToDraw.addOverlay(p3top0);
        sequenceToDraw.addOverlay(p3top2);

        p0.addAnchorListener(this);
        p0top3.addAnchorListener(this);
        p0top1.addAnchorListener(this);

        p1.addAnchorListener(this);
        p1top0.addAnchorListener(this);
        p1top2.addAnchorListener(this);

        p2.addAnchorListener(this);
        p2top1.addAnchorListener(this);
        p2top3.addAnchorListener(this);

        p3.addAnchorListener(this);
        p3top0.addAnchorListener(this);
        p3top2.addAnchorListener(this);

        p0.linkToAnchor2D(p0top3);
        p0.linkToAnchor2D(p0top1);

        p1.linkToAnchor2D(p1top0);
        p1.linkToAnchor2D(p1top2);

        p2.linkToAnchor2D(p2top1);
        p2.linkToAnchor2D(p2top3);

        p3.linkToAnchor2D(p3top0);
        p3.linkToAnchor2D(p3top2);

        updateDisplay();
    }

    /** Set the source image and creates the outputs */ 
    private void setSourceImage() {

    	//System.out.println("set source image...");
    	
    	try
    	{
    		sourceImage = sourceSequence.getFirstViewer().getCurrentImage();
    		Viewer viewer = sourceSequence.getFirstViewer();
            
    		boolean rebuildFinalImage = false;
    		    		
    		if ( finalImage == null ) rebuildFinalImage = true;
    		if ( finalImage != null && finalImage.getWidth() != outSequence.getWidth() && finalImage.getHeight() != outSequence.getHeight() ) rebuildFinalImage = true;
    		if ( finalImage != null && finalImage.getSizeC() != sourceImage.getSizeC() +1 ) rebuildFinalImage = true;
    		
    		if ( rebuildFinalImage )
    		{
    			System.out.println("Rebuild final image : w:" + outSequence.getWidth() + "  h:" + outSequence.getHeight() );
    			
    			// sizeC + 1 for alpha
    			finalImage = new IcyBufferedImage(
    					outSequence.getWidth(), outSequence.getHeight(), 
    					sourceImage.getSizeC() +1 , sourceImage.getDataType_() );

    			// build alpha color map
    			//finalImage.setColorMaps( sourceImage );

    			for ( int c = 0 ; c < sourceImage.getSizeC() ; c++ )
    			{
    				finalImage.setColorMap( c, viewer.getLut().getLutChannel( c ).getColorMap() , true );
    			}

    			finalImage.getColorMap( finalImage.getSizeC() -1 ).setType( IcyColorMapType.ALPHA );
    			finalImage.getColorMap( finalImage.getSizeC() -1 ).setAlphaControlPoint( 0 , 0 );
    			//finalImage.getColorMap( finalImage.getSizeC() -1 ).setAlphaControlPoint( 255 , 255 );
    			finalImage.getColorMap( finalImage.getSizeC() -1 ).setAlphaControlPoint( 255 , 255 );
    		}
			
    		// clean up previous final image ( data ) (and alpha is set to 0 = transparent)
			for ( int c = 0 ; c < finalImage.getSizeC() ; c++ )
			{
				Object data = finalImage.getDataXY( c );
				Array1DUtil.fill( data , 0 );        			        		
			}
    	}
    	catch( NullPointerException e)
    	{
    		// unsupported viewer, no viewer, unsupported canvas.
    	}    	

	}

	@Override
    public void keyPressed(KeyEvent e, Point2D imagePoint, IcyCanvas canvas)
    {
    	if ( e.getKeyChar()=='d' )
    	{
    		drawImage=!drawImage;
    		updateDisplay();
    		displayDrawImageStatus();
//    		canvas.getSequence().overlayChanged( (Overlay) null );
    	}

    	if ( e.getKeyChar()=='g' )
    	{
    		drawGrid=!drawGrid;
    		updateDisplay();
    		displayGridStatus();
//    		canvas.getSequence().overlayChanged( (Overlay) null );
    	}
    	if ( e.getKeyChar()=='r' )
    	{
    		copyROIs =!copyROIs;
    		displayCopyROIStatus();
    	}
    }

	String onOff( boolean value )
	{
		if ( value ) return "on";
		return "off";
	}
	
    private void displayCopyROIStatus() {
    	new AnnounceFrame("Map Overlay : ROI Copy (r): " + onOff( copyROIs ) , 5 );
	}

    private void displayGridStatus() {
    	new AnnounceFrame("Map Overlay : Grid display (g): " + onOff( drawGrid ) , 5 );
	}

    private void displayDrawImageStatus() {
    	new AnnounceFrame("Map Overlay : Draw image: (d): " + onOff( drawImage ) , 5 );
	}

	Point pointTest = null;

    @Override
    public void mouseClick(MouseEvent e, Point2D p, IcyCanvas canvas)
    {

    }

    @Override
    public void mouseDrag(MouseEvent e, Point2D p, IcyCanvas canvas)
    {

    }

    @Override
    public void mouseMove(MouseEvent e, Point2D p, IcyCanvas canvas)
    {

    }

    /**
     * @param t
     * @param p0
     *        point de depart
     * @param p1
     *        point de passage depart
     * @param p2
     *        point de passage fin
     * @param p3
     *        point de fin
     * @return
     */
    Point2D getPointOnCurve(double t, Anchor2D p0, Anchor2D p1, Anchor2D p2, Anchor2D p3)
    {
        final double unmoinsTauCube = Math.pow((1d - t), 3);
        final double unmoinsTauCarre = Math.pow((1d - t), 2);
        final double unmoinsT = 1d - t;

        final double x = unmoinsTauCube * p0.getX() + 3 * unmoinsTauCarre * t * p1.getX() + 3 * unmoinsT * t * t * p2.getX() + t * t
                * t * p3.getX();
        final double y = unmoinsTauCube * p0.getY() + 3 * unmoinsTauCarre * t * p1.getY() + 3 * unmoinsT * t * t * p2.getY() + t * t
                * t * p3.getY();

        return new Point2D.Double(x, y);
    }

    /**
     * @param t
     * @param p0
     *        point de depart
     * @param p1
     *        point de passage depart
     * @param p2
     *        point de passage fin
     * @param p3
     *        point de fin
     * @return
     */
    Point2D getPointOnCurve(double t, Point2D p0, Point2D p1, Point2D p2, Point2D p3)
    {
        final double unmoinsTauCube = Math.pow((1d - t), 3);
        final double unmoinsTauCarre = Math.pow((1d - t), 2);
        final double unmoinsT = 1d - t;

        final double x = unmoinsTauCube * p0.getX() + 3 * unmoinsTauCarre * t * p1.getX() + 3 * unmoinsT * t * t
                * p2.getX() + t * t * t * p3.getX();
        final double y = unmoinsTauCube * p0.getY() + 3 * unmoinsTauCarre * t * p1.getY() + 3 * unmoinsT * t * t
                * p2.getY() + t * t * t * p3.getY();

        return new Point2D.Double(x, y);
    }

    /**
     * Si t = 0 le point utilise est p1, et glisse ensuite vers p2
     * 
     * @param t
     * @param p1
     * @param p2
     * @return
     */
    Point2D ponderate2Points(double t, Anchor2D p1, Anchor2D p2)
    {
        final double x = t * p2.getX() + (1 - t) * p1.getX();
        final double y = t * p2.getY() + (1 - t) * p1.getY();
        return new Point2D.Double(x, y);
    }
    
    void drawControlPoints( Graphics2D g )
    {

        g.drawLine((int) Math.round(p0.getX()), (int) Math.round(p0.getY()), (int) Math.round(p0top3.getX()), (int) Math.round(p0top3.getY()));
        g.drawLine((int) Math.round(p0.getX()), (int) Math.round(p0.getY()), (int) Math.round(p0top1.getX()), (int) Math.round(p0top1.getY()));

        g.drawLine((int) Math.round(p1.getX()), (int) Math.round(p1.getY()), (int) Math.round(p1top0.getX()), (int) Math.round(p1top0.getY()));
        g.drawLine((int) Math.round(p1.getX()), (int) Math.round(p1.getY()), (int) Math.round(p1top2.getX()), (int) Math.round(p1top2.getY()));

        g.drawLine((int) Math.round(p2.getX()), (int) Math.round(p2.getY()), (int) Math.round(p2top1.getX()), (int) Math.round(p2top1.getY()));
        g.drawLine((int) Math.round(p2.getX()), (int) Math.round(p2.getY()), (int) Math.round(p2top3.getX()), (int) Math.round(p2top3.getY()));

        g.drawLine((int) Math.round(p3.getX()), (int) Math.round(p3.getY()), (int) Math.round(p3top0.getX()), (int) Math.round(p3top0.getY()));
        g.drawLine((int) Math.round(p3.getX()), (int) Math.round(p3.getY()), (int) Math.round(p3top2.getX()), (int) Math.round(p3top2.getY()));
    }
	

    @Override
    public void paint(Graphics2D g, Sequence sequence, IcyCanvas canvas)
    {
    	
        if (drawImage)
        {
        	setSourceImage();
        	renderMap();
            if (finalImage != null)
            {
                g.drawImage(IcyBufferedImageUtil.getARGBImage( finalImage ), null, 0, 0);
            }
        }
                
        Graphics2D g1 = (Graphics2D) g.create(); // big stroke (first pass, black)
        Graphics2D g2 = (Graphics2D) g.create(); // thin stroke, (second pass, white)
        g1.setStroke( new BasicStroke((float) canvas.canvasToImageLogDeltaX( 3 ) ) );
        g1.setColor( Color.black );
        g2.setStroke( new BasicStroke((float) canvas.canvasToImageLogDeltaX( 2 ) ) );
        g2.setColor( Color.white );
        		
   		drawControlPoints( g1 );		
   		drawControlPoints( g2 );		

        // surface bezier
        if ( drawGrid )
        {
        	drawGrid(g1);
        	drawGrid(g2);
        }
       
    }

    private void copyROI() {
    	
    	//List<ROI> roiList = sourceSequence.getROIs( plugins.kernel.roi.roi2d.ROI2DPolygon.class );
    	List<ROI> roiList = sourceSequence.getROIs( plugins.kernel.roi.roi2d.ROI2DPolygon.class );
    	
    	outSequence.beginUpdate();
    	
    	int currentZ = 0;
    	int currentT = 0;

    	try
    	{
    		currentZ = sourceSequence.getViewers().get( 0 ).getPositionZ();
    		currentT = sourceSequence.getViewers().get( 0 ).getPositionT();
    	}
    	catch( NullPointerException e )
    	{
    		// no viewer, compatible canvas...
    	}
    	
    	try
    	{
    		outSequence.removeAllROI();
			
    		for ( ROI roi : roiList )
    		{    			
    			ROI2DPolygon roiCopy = (ROI2DPolygon) roi.getCopy();
    			
    			if ( roiCopy.getZ() != currentZ )
    			{
    				if ( roiCopy.getZ() != -1 ) continue;
    			}
    			
    			if ( roiCopy.getT() != currentT )
    			{
    				if ( roiCopy.getT() != -1 ) continue;
    			}
    			
    			ArrayList<Point2D> pointList = roiCopy.getPoints();
    			for ( Point2D point : pointList )
    			{    			    			
    				point.setLocation( getGridLocation( point ) );

    			}
    			roiCopy.setPoints( (List<Point2D>) pointList );

    			// put to outSequence
    			outSequence.addROI( roiCopy );
    		}
    	}
    	finally
    	{
    		outSequence.endUpdate();
    	}
    	
	}

    /** Find location of a point in the grid */
	private Point2D getGridLocation(Point2D source) {

		// compute parametric value of the original point.
		double tx = source.getX() / sourceSequence.getWidth();
		double ty = source.getY() / sourceSequence.getHeight();
		
		// find location in grid.
		
        final Point2D start = getPointOnCurve(tx, p0, p0top1, p1top0, p1);
        final Point2D end = getPointOnCurve(tx, p3, p3top2, p2top3, p2);

        final Point2D startCtrl = ponderate2Points(tx, p0top3, p1top2);
        final Point2D endCtrl = ponderate2Points(tx, p3top0, p2top1);

        final Point2D pOut = getPointOnCurve(ty, start, startCtrl, endCtrl, end);
		
		return pOut;

	}

	private void drawGrid(Graphics2D g) {

        final double gridStep = 0.1d;
        
    	for (double tx = 0; tx <= 1; tx += gridStep)
    	{

    		// calcul du point de depart

    		final Point2D start = getPointOnCurve(tx, p0, p0top3, p3top0, p3);
    		final Point2D end = getPointOnCurve(tx, p1, p1top2, p2top1, p2);

    		final Point2D startCtrl = ponderate2Points(tx, p0top1, p3top2);
    		final Point2D endCtrl = ponderate2Points(tx, p1top0, p2top3);

    		final Path2D spline = new Path2D.Double();

    		for (double ty = 0; ty <= 1; ty += gridStep)
    		{

    			final Point2D p = getPointOnCurve(ty, start, startCtrl, endCtrl, end);

    			if (ty == 0)
    			{
    				spline.moveTo(p.getX(), p.getY());
    			}
    			else
    			{
    				spline.lineTo(p.getX(), p.getY());
    			}

    		}

    		g.draw(spline);
    	}

    	for (double ty = 0; ty <= 1; ty += gridStep)
    	{

    		// calcul du point de depart

    		final Point2D start = getPointOnCurve(ty, p0, p0top1, p1top0, p1);
    		final Point2D end = getPointOnCurve(ty, p3, p3top2, p2top3, p2);

    		final Point2D startCtrl = ponderate2Points(ty, p0top3, p1top2);
    		final Point2D endCtrl = ponderate2Points(ty, p3top0, p2top1);

    		final Path2D spline = new Path2D.Double();

    		for (double tx = 0; tx <= 1; tx += gridStep)
    		{

    			final Point2D p = getPointOnCurve(tx, start, startCtrl, endCtrl, end);

    			if (tx == 0)
    			{
    				spline.moveTo(p.getX(), p.getY());
    			}
    			else
    			{
    				spline.lineTo(p.getX(), p.getY());
    			}

    		}
    		
    		g.draw(spline);
    	}
    	
	}

	ArrayList<ActionListener> actionListenerList = new ArrayList<ActionListener>();

    public void addActionListener(ActionListener actionListener)
    {
        actionListenerList.add(actionListener);
    }

    public void removeActionListener(ActionListener actionListener)
    {
        actionListenerList.remove(actionListener);
    }

    class RenderImage implements Runnable
    {

        boolean stop = false;
        float percentDone = 0;

        
        
        @Override
        public void run()
        {
        	setSourceImage( );
            // zero fill alpha
            // byte[] alphaFinal = finalImage.getDataXYAsByte(3);
//            final byte[] finalImageData = finalImage.getDataXYAsByte(0);
//            final byte[] sourceImageData = sourceImage.getDataXYAsByte(0);

			        		

        	
//            final int widthFinal = finalImage.getWidth();
//            final int heightFinal = finalImage.getHeight();
            final int widthSource = sourceImage.getWidth();
            final int heightSource = sourceImage.getHeight();

            // compute final image:

            // grid computation

            final double step = 0.1d;

            // ArrayList<CorrespondingPoint> correspondingList = new
            // ArrayList<CorrespondingPoint>();

            // remplissage avec des triangles.

            final CorrespondingPoint[][] correspondingPointTab = new CorrespondingPoint[2 * (int) (1d / step)][2 * (int) (1d / step)];
            {
                int tableauX = 0;
                for (double tx = 0; tx <= 1; tx += step, tableauX++)
                {

                    // calcul du point de depart

                    final Point2D start = getPointOnCurve(tx, p0, p0top1, p1top0, p1);
                    final Point2D end = getPointOnCurve(tx, p3, p3top2, p2top3, p2);

                    final Point2D startCtrl = ponderate2Points(tx, p0top3, p1top2);
                    final Point2D endCtrl = ponderate2Points(tx, p3top0, p2top1);

                    int tableauY = 0;
                    for (double ty = 0; ty <= 1; ty += step, tableauY++)
                    {

                        final Point2D p = getPointOnCurve(ty, start, startCtrl, endCtrl, end);
                        {

                            final int sourceX = (int) (widthSource * tx);
                            final int sourceY = (int) (heightSource * ty);

                            final CorrespondingPoint corresp = new CorrespondingPoint(sourceX, sourceY, p.getX(), p
                                    .getY());

                            correspondingPointTab[tableauX][tableauY] = corresp;

                        }

                    }

                }

            }

            int tableauX = 0;
            for (double tx = 0; tx <= 1 - step; tx += step, tableauX++)
            {
                int tableauY = 0;
                for (double ty = 0; ty <= 1 - step; ty += step, tableauY++)

                {
                    {
                     
                        // graphics , p1 , p2 , p3 , uv1 , uv2, uv3
                        drawTriangle(finalImage, sourceImage, correspondingPointTab[tableauX][tableauY].grid,
                                correspondingPointTab[tableauX + 1][tableauY + 1].grid,
                                correspondingPointTab[tableauX][tableauY + 1].grid,
                                correspondingPointTab[tableauX][tableauY].image,
                                correspondingPointTab[tableauX + 1][tableauY + 1].image,
                                correspondingPointTab[tableauX][tableauY + 1].image);
                        drawTriangle(finalImage, sourceImage, correspondingPointTab[tableauX][tableauY].grid,
                                correspondingPointTab[tableauX + 1][tableauY + 1].grid,
                                correspondingPointTab[tableauX + 1][tableauY].grid,
                                correspondingPointTab[tableauX][tableauY].image,
                                correspondingPointTab[tableauX + 1][tableauY + 1].image,
                                correspondingPointTab[tableauX + 1][tableauY].image);

                        // System.out.println(correspondingPointTab[tableauX+1][tableauY+1]);
                    }
                }
            }
            finalImage.dataChanged();
            

        }

        private void drawTriangle(IcyBufferedImage finalImage, IcyBufferedImage sourceImage, Point2D p1, Point2D p2,
                Point2D p3, Point2D uv1, Point2D uv2, Point2D uv3)
        {
            final Point2D min, med, max;
            final Point2D minUV, medUV, maxUV;
            final Point2D endLeftUV, endRightUV;

            // find min, med & max y
            if (p1.getY() < p2.getY())
            {
                if (p1.getY() < p3.getY())
                {
                    min = p1;
                    minUV = uv1;

                    if (p2.getY() < p3.getY())
                    {
                        med = p2;
                        medUV = uv2;
                        max = p3;
                        maxUV = uv3;
                    }
                    else
                    {
                        med = p3;
                        medUV = uv3;
                        max = p2;
                        maxUV = uv2;
                    }
                }
                else
                {
                    min = p3;
                    minUV = uv3;
                    med = p1;
                    medUV = uv1;
                    max = p2;
                    maxUV = uv2;
                }
            }
            else
            {
                if (p2.getY() < p3.getY())
                {
                    min = p2;
                    minUV = uv2;

                    if (p1.getY() < p3.getY())
                    {
                        med = p1;
                        medUV = uv1;
                        max = p3;
                        maxUV = uv3;
                    }
                    else
                    {
                        med = p3;
                        medUV = uv3;
                        max = p1;
                        maxUV = uv1;
                    }
                }
                else
                {
                    min = p3;
                    minUV = uv3;
                    med = p2;
                    medUV = uv2;
                    max = p1;
                    maxUV = uv1;
                }
            }

            final double minX = min.getX();
            final double minY = min.getY();
            final double medDeltaX = med.getX() - minX;
            final double maxDeltaX = max.getX() - minX;
            final double medDeltaY = med.getY() - minY;
            final double maxDeltaY = max.getY() - minY;
            final double medStepX;
            final double maxStepX;

            if (medDeltaY != 0)
                medStepX = medDeltaX / medDeltaY;
            else
                medStepX = medDeltaX;
            if (maxDeltaY != 0)
                maxStepX = maxDeltaX / maxDeltaY;
            else
                maxStepX = maxDeltaX;

            // calculate coordinate step
            final double leftDeltaY;
            final double rightDeltaY;
            double leftStep;
            double rightStep;

            // find end left & end right
            if (medStepX < maxStepX)
            {
                endLeftUV = medUV;
                endRightUV = maxUV;

                leftDeltaY = medDeltaY;
                rightDeltaY = maxDeltaY;
                leftStep = medStepX;
                rightStep = maxStepX;
            }
            else
            {
                endLeftUV = maxUV;
                endRightUV = medUV;

                leftDeltaY = maxDeltaY;
                rightDeltaY = medDeltaY;
                leftStep = maxStepX;
                rightStep = medStepX;
            }

            double leftU = minUV.getX();
            double leftV = minUV.getY();
            double rightU = minUV.getX();
            double rightV = minUV.getY();
            double leftUStep;
            double leftVStep;
            double rightUStep;
            double rightVStep;

            // calculate UV step
            if (leftDeltaY != 0)
            {
                leftUStep = (endLeftUV.getX() - leftU) / leftDeltaY;
                leftVStep = (endLeftUV.getY() - leftV) / leftDeltaY;
            }
            else
            {
                leftUStep = 0;
                leftVStep = 0;
            }

            if (rightDeltaY != 0)
            {
                rightUStep = (endRightUV.getX() - rightU) / rightDeltaY;
                rightVStep = (endRightUV.getY() - rightV) / rightDeltaY;
            }
            else
            {
                rightUStep = 0;
                rightVStep = 0;
            }

            final Point2D leftUV = new Point2D.Double(leftU, leftV);
            final Point2D rightUV = new Point2D.Double(rightU, rightV);

            // first part
            int y;
            int bottom;
            double left = minX;
            double right = minX;

            y = (int) min.getY();
            bottom = (int) med.getY();

            while (y < bottom)
            {
                drawTexturedLine(y, left, right, leftUV, rightUV, sourceImage, finalImage);

                // next scanline
                left += leftStep;
                leftU += leftUStep;
                leftV += leftVStep;
                leftUV.setLocation(leftU, leftV);

                right += rightStep;
                rightU += rightUStep;
                rightV += rightVStep;
                rightUV.setLocation(rightU, rightV);

                y++;
            }

            final double endDeltaY = max.getY() - med.getY();

            if (medUV == endLeftUV)
            {
                // recalculate left step
                left = med.getX();
                leftU = medUV.getX();
                leftV = medUV.getY();
                leftUV.setLocation(leftU, leftV);

                if (endDeltaY != 0)
                    leftStep = (max.getX() - left) / endDeltaY;
                else
                    leftStep = 0;

                if (endDeltaY != 0)
                {
                    leftUStep = (maxUV.getX() - leftU) / endDeltaY;
                    leftVStep = (maxUV.getY() - leftV) / endDeltaY;
                }
                else
                {
                    leftUStep = 0;
                    leftVStep = 0;
                }
            }
            else
            {
                // recalculate right step
                right = med.getX();
                rightU = medUV.getX();
                rightV = medUV.getY();
                rightUV.setLocation(rightU, rightV);

                if (endDeltaY != 0)
                    rightStep = (max.getX() - right) / endDeltaY;
                else
                    rightStep = 0;

                if (endDeltaY != 0)
                {
                    rightUStep = (maxUV.getX() - rightU) / endDeltaY;
                    rightVStep = (maxUV.getY() - rightV) / endDeltaY;
                }
                else
                {
                    rightUStep = 0;
                    rightVStep = 0;
                }
            }

            // second part
            bottom = (int) max.getY();

            while (y < bottom)
            {
                drawTexturedLine(y, left, right, leftUV, rightUV, sourceImage, finalImage);

                // next scanline
                left += leftStep;
                leftU += leftUStep;
                leftV += leftVStep;
                leftUV.setLocation(leftU, leftV);

                right += rightStep;
                rightU += rightUStep;
                rightV += rightVStep;
                rightUV.setLocation(rightU, rightV);

                y++;
            }
        }

        private void drawTexturedLine(int y, double left, double right, Point2D leftUV, Point2D rightUV,
                IcyBufferedImage sourceImage, IcyBufferedImage finalImage)
        {
            final double delta = right - left;
            double u = leftUV.getX();
            double v = leftUV.getY();
            final double uStep;
            final double vStep;

            final int wSrc = sourceImage.getWidth();
            final int hSrc = sourceImage.getHeight();
            final int wDst = finalImage.getWidth();
            final int hDst = finalImage.getHeight();

            if ((left >= wDst) || (right < 0) || (y >= hDst) || (y < 0))
                return;

            // calculate UV step
            if (delta != 0)
            {
                uStep = (rightUV.getX() - u) / delta;
                vStep = (rightUV.getY() - v) / delta;
            }
            else
            {
                uStep = 0;
                vStep = 0;
            }

            final int x;
            final int r;

            // do clipping
            if (left < 0)
            {
                u += uStep * -left;
                v += vStep * -left;
                x = 0;
            }
            else
                x = (int) left;

            if (right >= wDst)
                r = wDst - 1;
            else
                r = (int) right;

//            final byte[] src = sourceImage.getDataXYAsByte(0);
//            final byte[] dst = finalImage.getDataXYAsByte(0);

                        
            final int src_pitch = sourceImage.getSizeX();
            int offset = x + (y * finalImage.getWidth());
            final int limit = offset + (r - x);

            while (offset < limit)
            {
            	final int u_i = (int) u;
            	final int v_i = (int) v;

            	if ((u_i >= 0) && (v_i >= 0) && (u_i < wSrc) && (v_i < hSrc))
            	{
            		for ( int c = 0 ; c < sourceImage.getSizeC() ; c++ )
                    {
            			Object srcArray = sourceImage.getDataXY( c );
            			Object dstArray = finalImage.getDataXY( c );
            			
            			Array1DUtil.setValue(dstArray, offset, 
            					Array1DUtil.getValue(srcArray, (v_i * src_pitch) + u_i, sourceImage.isSignedDataType() ));
            			
            			//dst[offset] = src[(v_i * src_pitch) + u_i];
            			
            			// alphaDst[offset] = (byte) 255;
                    }
            		
            		// set alpha to 255            		
        			Object dstArray = finalImage.getDataXY( finalImage.getSizeC() -1 );        			
        			Array1DUtil.setValue(dstArray, offset, 255 );
            		
            	}
//            	else
//            	{
//            		// put 0 value to all channels (so is transparent )
//            		// for ( int c = 0 ; c < sourceImage.getSizeC() ; c++ )
//            		for ( int c = 0 ; c < finalImage.getSizeC() ; c++ )
//                    {
//            			System.out.println(c);
//            			Object dstArray = finalImage.getDataXY( c );
//            			Array1DUtil.setValue(dstArray, offset, 0 );
//                    }
//            		
//            	}

            	u += uStep;
            	v += vStep;
            	offset++;
            }
            
        }

        class DistanceCorresp
        {
            public DistanceCorresp(CorrespondingPoint cp, double distance)
            {
                this.cp = cp;
                this.distance = distance;
            }

            CorrespondingPoint cp;
            double distance;
        }

        class CorrespondingPoint
        {
            public CorrespondingPoint(double xImage, double yImage, double xGrid, double yGrid)
            {
                grid = new Point2D.Double(xGrid, yGrid);
                image = new Point2D.Double(xImage, yImage);
            }

            double distanceSqToGrid(double x, double y)
            {
                return grid.distanceSq(x, y);
            }

            Point2D grid;
            Point2D image;

            @Override
            public String toString()
            {
                return "GridX: " + grid.getX() + " GridY: " + grid.getY() + " ImageX: " + image.getX() + " ImageY: "
                        + image.getY();
            }
        }

    }

    final RenderImage ri = new RenderImage();
    
    public void renderMap()
    {
    	if ( drawImage == true )
    	{
//    		final RenderImage ri = new RenderImage();
    		ri.run();
    	}
    }

    boolean drawImage = false;
    boolean drawGrid = true;
	boolean copyROIs = true;

    public void setDrawImage(boolean enabled)
    {
        drawImage = enabled;
        updateDisplay();	
    }

    @Override
    public void keyReleased(KeyEvent e, Point2D imagePoint, IcyCanvas canvas)
    {

    }

    @Override
    public void mousePressed(MouseEvent e, Point2D imagePoint, IcyCanvas canvas)
    {

    }

    @Override
    public void mouseReleased(MouseEvent e, Point2D imagePoint, IcyCanvas canvas)
    {

    }


	@Override
	public void positionChanged(icy.painter.Anchor2D source) {
					
		updateDisplay();
		
	}
	
	void updateDisplay()
	{
//		renderMap();
		// update ROI copy
        if ( copyROIs )
        {
        	copyROI();
        }
        outSequence.overlayChanged( this );
		
	}

	@Override
	public void painterChanged(PainterEvent event) {
		
	}

	@Override
	public void sequenceChanged(SequenceEvent sequenceEvent) {
		
		if ( sequenceEvent.getSequence() == sourceSequence )
		{
			updateDisplay();			
		}
		
	}

	@Override
	public void sequenceClosed(Sequence sequence) {

		//if ( sequence == sourceSequence )
		{
			outSequence.removeOverlay( this );			

			outSequence.removeOverlay(p0);
			outSequence.removeOverlay(p0top3);
			outSequence.removeOverlay(p0top1);
			outSequence.removeOverlay(p1);
			outSequence.removeOverlay(p1top0);
			outSequence.removeOverlay(p1top2);
			outSequence.removeOverlay(p2);
			outSequence.removeOverlay(p2top1);
			outSequence.removeOverlay(p2top3);
			outSequence.removeOverlay(p3);
			outSequence.removeOverlay(p3top0);
			outSequence.removeOverlay(p3top2);

			outSequence = null;
			sourceSequence = null;
		}

	}

	public void setDrawGrid(Boolean value) {
		drawGrid = value;
		updateDisplay();
		
	}

	public void setCopyROIs(Boolean value) {
		copyROIs = value;
		updateDisplay();		
	}


	

}
