package plugins.Yrvann.zhoubaoshi.SphericalParameterization;

import icy.sequence.Sequence;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintStream;

import javax.swing.SwingUtilities;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

import plugins.adufour.activemeshes.mesh.Mesh;
import plugins.adufour.activemeshes.painters.VTKMeshPainter;
import plugins.adufour.activemeshes.producers.VTKMeshReader;
import plugins.adufour.ezplug.EzGroup;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzVarBoolean;
import plugins.adufour.ezplug.EzVarFile;
import plugins.adufour.ezplug.EzVarFolder;


/** 
 *  This plug-in implements the algorithm published in the following article: 
 * 		3D surface filtering using spherical harmonics
 *  	Kun Zhou, Hujun Bao and Jiaoying Shi.
 *  	In: Computer-Aided Design 36(4):363-375 (2004)
 *  
 *  [Part I  - Loading the initial mesh]
 *  
 *  [Part II - Spherical parameterization]
 *  - 1) Building the progressive mesh (adopting Lindstrom's volume-preserving method)
 *  - 2) Building the spherical mesh
 *      a) Locating the centroid of the base mesh
 *      b) Recovering the spherical mesh
 *  - 3) Building the spherical parameterization
 *      
 *  @author Yrvann EMZIVAT (March 2014)
 *   
 **/

public class ZBSParameterization extends EzPlug {

	EzVarFile   varFileIn;
	EzVarFolder varFolderOut;

	EzVarBoolean exportCondition;
	EzVarBoolean paintCondition;
	
	@Override
	public void initialize()
	{		
		
		varFileIn = new EzVarFile("Input (Mesh): ", null);
		varFolderOut = new EzVarFolder("Output (Mesh): ", null);
				
		EzGroup groupFiles1 = new EzGroup("Spherical parameterization ", varFileIn, varFolderOut);
		super.addEzComponent(groupFiles1);

		exportCondition = new EzVarBoolean("Export (VTK)", false);
		paintCondition = new EzVarBoolean("View (VTK)", false);
		
		EzGroup groupSequence2 = new EzGroup("Options", exportCondition, paintCondition);
		super.addEzComponent(groupSequence2);
	}
		
	@SuppressWarnings({ "deprecation" })
	@Override
	public void execute()
	{		
		
		System.out.println("*** Execute (start) *******\n");
		boolean exportCondition = this.exportCondition.getValue();
		boolean paintCondition = this.paintCondition.getValue();
	
		
	/** Part I : Loading the initial mesh */
		
		VTKMeshReader VTKreaderMp = new VTKMeshReader(varFileIn.getValue(), true);
		VTKreaderMp.run();
		Mesh Mp = VTKreaderMp.getMesh();
		
// FIXME : Les faces et les sommets apparaissent en double avec la mthode getMesh();	
		int Vsize = Mp.vertices.size();
		int Fsize = Mp.faces.size();
		for (int i=Vsize-1; i>Vsize/2-1; i--) 
			Mp.vertices.remove(i);
		for (int i=Fsize-1; i>Fsize/2-1; i--)
			Mp.faces.remove(i);

		final VTKMeshPainter meshPaint = new VTKMeshPainter();
		meshPaint.addMesh(Mp);
		final Sequence seq = getActiveSequence();
		seq.addPainter(meshPaint); 	

		
	/** Part II : Spherical parameterization */
		
	// Building the Weighted Edges		
		WeightedEdge Ep = new WeightedEdge();
		System.out.println("Processing Weighted Edges [...]");	
		Ep.init(Mp);
		System.out.println("Processing Weighted Edges [.X.]\n...");			
		
	/** 1) Building the progressive mesh (adopting Lindstrom's volume-preserving method) */
		
		System.out.println("Progressive Mesh [...]");		
		SplitOperations Op = new SplitOperations();
		int p = 0;	

		boolean convexMesh = false;
		while (!convexMesh) {	
			boolean isConvex = true;			
			for (int i=0; i<Mp.faces.size(); i++) {
				if (isConvex && !Op.deadFaces.contains(i)) {
					Vector3d AB = new Vector3d();
					Vector3d AC = new Vector3d();
					Vector3d n = new Vector3d();	

					AB.x = Mp.vertices.get(Mp.faces.get(i).v2).position.x - Mp.vertices.get(Mp.faces.get(i).v1).position.x;
					AB.y = Mp.vertices.get(Mp.faces.get(i).v2).position.y - Mp.vertices.get(Mp.faces.get(i).v1).position.y;
					AB.z = Mp.vertices.get(Mp.faces.get(i).v2).position.z - Mp.vertices.get(Mp.faces.get(i).v1).position.z;
					AC.x = Mp.vertices.get(Mp.faces.get(i).v3).position.x - Mp.vertices.get(Mp.faces.get(i).v1).position.x;
					AC.y = Mp.vertices.get(Mp.faces.get(i).v3).position.y - Mp.vertices.get(Mp.faces.get(i).v1).position.y;
					AC.z = Mp.vertices.get(Mp.faces.get(i).v3).position.z - Mp.vertices.get(Mp.faces.get(i).v1).position.z;
					n.cross(AB,AC);
					
					double x = Mp.vertices.get(Mp.faces.get(i).v1).position.x;
					double y = Mp.vertices.get(Mp.faces.get(i).v1).position.y;
					double z = Mp.vertices.get(Mp.faces.get(i).v1).position.z;
					
					double a = n.x;
					double b = n.y;
					double c = n.z;
					double d = -a*x - b*y - c*z;
		
					double sign = 0;
					boolean first = true;
					for (int j=0; j<Mp.vertices.size(); j++) {
						if ((Mp.vertices.get(j) != null)&&((Mp.faces.get(i).v1 != j)&&(Mp.faces.get(i).v2 != j)&&(Mp.faces.get(i).v3 != j))) {
							x = Mp.vertices.get(j).position.x;
							y = Mp.vertices.get(j).position.y;
							z = Mp.vertices.get(j).position.z;
							if (first) {
								sign = a*x + b*y + c*z + d;
								first = false;	
							}
							else if (sign*(a*x + b*y + c*z + d) < -Double.MIN_VALUE)
								isConvex = false;
						}
					}
				}
			}	
			
			System.out.println("   Mesh n" + p + " (convex: " + isConvex + ")");
			
			if (!isConvex) {
				Ep.update(Mp,Op);					
				if (exportCondition) {				
// FIXME gestion des fichiers et dossiers  vrifier + MAC/Windows
					File VTKexport = new File(varFolderOut.getValue().toString() + "/ProgMesh" + (p+1) + ".vtk.txt");
					PrintStream ps = null;
					try {
						ps = new PrintStream(VTKexport);
					} catch (FileNotFoundException e) {
						e.printStackTrace();
					}
					Mp.exportToVTK(ps);
				}
			}
			
			if (paintCondition) {
				SwingUtilities.invokeLater(new Runnable() {
					public void run() {
						seq.painterChanged(meshPaint);
					}
				});
			}			
			
		// Abort condition 
			// TODO Better write [nbInitialVertices - p] to get the number of remaining vertices
			int nbVerticesPos = 0;
			for (int i=0;i<Mp.vertices.size();i++)
				if (Mp.vertices.get(i) != null)
					nbVerticesPos++;
			if (nbVerticesPos <= 12) {
				convexMesh = true;
				System.out.println("Ending edge decimation prematurely\n");
			}

			p++;
		}
		
		System.out.println("Progressive Mesh [.X.]\n...");

		
	/** 2) Building the spherical mesh */

		/** a)	Locating the centroid of the base mesh */

		System.out.println("Spherical Mesh [...]");
		
		Point3d center = new Point3d();
		double nbVertices = 0.;
		for (int i=0; i<Mp.vertices.size(); i++) {
			if (Mp.vertices.get(i) != null) {
				center.x += Mp.vertices.get(i).position.x;
				center.y += Mp.vertices.get(i).position.y;
				center.z += Mp.vertices.get(i).position.z;
				nbVertices++;
			}
		}
		center.x /= nbVertices;
		center.y /= nbVertices;
		center.z /= nbVertices;

		for (int i=0; i<Mp.vertices.size(); i++) {
			if (Mp.vertices.get(i) != null) {
				Mp.vertices.get(i).position.x -= center.x;
				Mp.vertices.get(i).position.y -= center.y;
				Mp.vertices.get(i).position.z -= center.z;
			}
		}
		
	    /** b) Recovering the spherical mesh */
		
		for (int i=0; i<Mp.vertices.size(); i++) {
			if (Mp.vertices.get(i) != null) {
				Vector3d V = new Vector3d();
				V.x = Mp.vertices.get(i).position.x;
				V.y = Mp.vertices.get(i).position.y;
				V.z = Mp.vertices.get(i).position.z;
				V.normalize();
				Mp.vertices.get(i).position.x = V.x;
				Mp.vertices.get(i).position.y = V.y;
				Mp.vertices.get(i).position.z = V.z;
			}			
		}

		if (exportCondition) {				
			File VTKexport = new File(varFolderOut.getValue().toString() + "/SphMesh" + (p) + ".vtk.txt");
			PrintStream ps = null;
			try {
				ps = new PrintStream(VTKexport);
			} catch (FileNotFoundException e) {
				e.printStackTrace();
			}
			Mp.exportToVTK(ps);
		}
		
		Op.rebuild(Mp,exportCondition,paintCondition,varFolderOut,seq,meshPaint);
		 
		System.out.println("Spherical Mesh [.X.]\n");
		System.out.println("*** Execute (end) *******");
		
	}
			
	@Override
	public void clean()
	{
			  
	}

}
