package plugins.adufour.trackprocessors;

import icy.gui.dialog.ConfirmDialog;
import icy.sequence.Sequence;
import icy.swimmingPool.SwimmingObject;
import icy.system.IcyHandledException;
import icy.util.XMLUtil;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

import plugins.adufour.vars.lang.Var;
import plugins.adufour.vars.lang.VarInteger;
import plugins.adufour.vars.util.VarListener;
import plugins.fab.trackmanager.Link;
import plugins.fab.trackmanager.PluginTrackManagerProcessor;
import plugins.fab.trackmanager.TrackGroup;
import plugins.fab.trackmanager.TrackSegment;
import plugins.nchenouard.spot.Detection;

/**
 * Track Manager plug-in that imports and exports tracks from the sequence metadata
 * 
 * @author Alexandre Dufour
 */
public class TracksIO extends PluginTrackManagerProcessor
{
    public static final String        XML_TRACKS_NODE    = "Tracks";
    
    private Sequence                  currentSequence    = null;
    
    private final Set<SwimmingObject> currentTrackGroups = new HashSet<SwimmingObject>();
    
    private final VarInteger          nbTracks           = new VarInteger("nbTrakcs", -1);
    
    private boolean                   reload             = false;
    
    public TracksIO()
    {
        setName("Import tracks from sequence");
        
        panel.setLayout(new BorderLayout());
        
        JPanel myPanel = new JPanel();
        
        // assemble the main interface
        myPanel.setLayout(new BoxLayout(myPanel, BoxLayout.X_AXIS));
        
        final JLabel labelNbTracks = new JLabel("No sequence selected");
        nbTracks.addListener(new VarListener<Integer>()
        {
            @Override
            public void valueChanged(Var<Integer> source, Integer oldValue, Integer newValue)
            {
                String text = "  This sequence has ";
                text += (newValue == 0 ? "no" : newValue);
                text += " track group";
                if (newValue > 1) text += "s";
                labelNbTracks.setText(text);
            }
            
            @Override
            public void referenceChanged(Var<Integer> source, Var<? extends Integer> oldReference, Var<? extends Integer> newReference)
            {
            }
        });
        
        myPanel.add(labelNbTracks);
        
        myPanel.add(Box.createHorizontalGlue());
        
        final JButton saveTracks = new JButton("Save selected tracks to sequence metadata");
        saveTracks.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent arg0)
            {
                if (currentSequence == null) throw new IcyHandledException("No sequence to save tracks to");
                
                boolean selectionExists = false;
                
                for (TrackSegment segment : trackPool.getTrackSegmentList())
                    if (selectionExists = segment.isAllDetectionSelected()) break;
                
                if (!selectionExists && !ConfirmDialog.confirm("Warning", "No tracks selected. This will erase all tracks from the sequence metadata. Proceed?")) return;
                
                Node tracksNode = currentSequence.getNode(XML_TRACKS_NODE);
                
                // clear old tracks
                while (tracksNode.getChildNodes().getLength() > 0)
                    tracksNode.removeChild(tracksNode.getChildNodes().item(0));
                
                // save current tracks
                saveSelectedTracks(tracksNode);
                
                currentSequence.saveXMLData();
                
                reload = true;
                
                trackPool.fireTrackEditorProcessorChange();
            }
        });
        
        myPanel.add(saveTracks);
        
        panel.setPreferredSize(new Dimension(0, 30));
        
        panel.add(myPanel, BorderLayout.CENTER);
    }
    
    @Override
    public void Close()
    {
        currentSequence = null;
        
        // clear locally-loaded tracks
        for (SwimmingObject trackGroupContainer : currentTrackGroups)
            trackPool.removeResult(trackGroupContainer);
        
        currentTrackGroups.clear();
    }
    
    @Override
    public void Compute()
    {
        if (currentSequence == trackPool.getDisplaySequence() && !reload) return;
        reload = false;
        
        currentSequence = trackPool.getDisplaySequence();
        
        // clear previously loaded tracks
        for (SwimmingObject trackGroupContainer : currentTrackGroups)
            trackPool.removeResult(trackGroupContainer);
        
        currentTrackGroups.clear();
        nbTracks.setValue(0);
        
        if (currentSequence == null) return;
        
        Node trackNode = currentSequence.isNodeExisting(XML_TRACKS_NODE);
        if (trackNode == null) return;
        
        currentTrackGroups.addAll(loadTracks(trackNode));
    }
    
    @Override
    public void displaySequenceChanged()
    {
        Compute();
    }
    
    private Set<SwimmingObject> loadTracks(Node node)
    {
        ArrayList<Element> trackGroupElementList = XMLUtil.getElements(node, "trackgroup");
        HashSet<SwimmingObject> loadedTrackGroups = new HashSet<SwimmingObject>();
        
        for (Element trackGroupElement : trackGroupElementList)
        {
            ArrayList<Element> trackSegmentElementList = XMLUtil.getElements(trackGroupElement);
            
            TrackGroup trackGroup = new TrackGroup(currentSequence);
            
            String groupDescription = XMLUtil.getAttributeValue(trackGroupElement, "description", "no description");
            trackGroup.setDescription(groupDescription);
            
            for (Element trackSegmentElement : trackSegmentElementList)
            {
                
                TrackSegment trackSegment = new TrackSegment();
                int id = XMLUtil.getAttributeIntValue(trackSegmentElement, "id", -1);
                if (id == -1)
                {
                    trackSegment.generateId();
                }
                else
                {
                    trackSegment.setId(id);
                }
                
                ArrayList<Element> detectionElementList = XMLUtil.getElements(trackSegmentElement);
                
                for (Element detectionElement : detectionElementList)
                {
                    String className = XMLUtil.getAttributeValue(detectionElement, "classname", Detection.class.getName());
                    
                    Detection detection = Detection.createDetection(className);
                    
                    detection.loadFromXML(detectionElement);
                    
                    trackSegment.addDetection(detection);
                }
                
                trackSegment.setAllDetectionSelected(false);
                trackGroup.addTrackSegment(trackSegment);
            }
            
            // SwimmingObject so = new SwimmingObject(trackGroup);
            // Icy.getMainInterface().getSwimmingPool().add(so);
            SwimmingObject trackGroupContainer = new SwimmingObject(trackGroup);
            trackPool.addResult(trackGroupContainer);
            loadedTrackGroups.add(trackGroupContainer);
        }
        
        // load links.
        
        ArrayList<Element> linkList = XMLUtil.getElements(node, "linklist");
        {
            for (Element linkListElement : linkList)
            {
                ArrayList<Element> trackSegmentElementList = XMLUtil.getElements(linkListElement);
                for (Element link : trackSegmentElementList)
                {
                    int idFrom = XMLUtil.getAttributeIntValue(link, "from", -1);
                    int idTo = XMLUtil.getAttributeIntValue(link, "to", -1);
                    
                    TrackSegment from = TrackSegment.getTrackSegmentById(idFrom);
                    TrackSegment to = TrackSegment.getTrackSegmentById(idTo);
                    
                    if (from != null && to != null)
                    {
                        trackPool.createLink(from, to);
                    }
                }
            }
        }
        
        nbTracks.setValue(loadedTrackGroups.size());
        
        return loadedTrackGroups;
    }
    
    private void saveSelectedTracks(Node node)
    {
        Element versionElement = XMLUtil.addElement(node, "trackfile");
        versionElement.setAttribute("version", "1");
        
        int nbSelectedTracks = 0;
        
        for (TrackGroup trackGroup : trackPool.getTrackGroupList())
        {
            boolean saveTrackGroup = false;
            
            // check first that the group has selected tracks
            for (TrackSegment trackSegment : trackGroup.getTrackSegmentList())
                if (saveTrackGroup = trackSegment.isAllDetectionSelected()) break;
            
            if (!saveTrackGroup) continue;
            
            nbSelectedTracks++;
            
            Element trackGroupElement = XMLUtil.addElement(node, "trackgroup");
            trackGroupElement.setAttribute("description", trackGroup.getDescription());
            
            for (TrackSegment trackSegment : trackGroup.getTrackSegmentList())
            {
                if (!trackSegment.isAllDetectionSelected()) continue;
                
                Element trackElement = XMLUtil.addElement(trackGroupElement, "track");
                trackElement.setAttribute("id", "" + trackSegment.getId());
                
                for (Detection detection : trackSegment.getDetectionList())
                {
                    Element detectionElement = XMLUtil.addElement(trackElement, "detection");
                    detectionElement.setAttribute("classname", detection.getClass().getName());
                    detection.saveToXML(detectionElement);
                }
            }
        }
        
        nbTracks.setValue(nbSelectedTracks);
        
        // save all links between selected tracks
        
        Element linklistElement = XMLUtil.addElement(node, "linklist");
        for (Link link : trackPool.getLinks())
        {
            if (link.getStartSegment().isAllDetectionSelected() && link.getEndSegment().isAllDetectionSelected())
            {
                Element trackElement = XMLUtil.addElement(linklistElement, "link");
                
                trackElement.setAttribute("from", "" + link.getStartSegment().getId());
                trackElement.setAttribute("to", "" + link.getEndSegment().getId());
            }
        }
    }
}
