package plugins.angelopo.pottslab;
 
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Arrays;
import java.util.ArrayList;
import java.util.Timer;
import java.util.TimerTask;
import java.awt.Rectangle;

import javax.swing.JProgressBar;

import plugins.adufour.ezplug.*;
import icy.gui.component.IcySlider;
import icy.gui.dialog.MessageDialog;
import icy.gui.frame.progress.AnnounceFrame;
import icy.sequence.Sequence;
import icy.sequence.SequenceUtil;
import icy.image.IcyBufferedImage;
import icy.image.IcyBufferedImageUtil;
import icy.network.NetworkUtil;
import icy.roi.BooleanMask2D;
import icy.roi.ROI2D;
import icy.roi.ROI2DArea;

/**
 * Graphical User Interface for the Interactive Potts Segmentation Icy plugin 
 * 
 * @author Vasileios Angelopoulos
 * 
 */

@SuppressWarnings("deprecation")
public class PottsSegmentation extends EzPlug
{
	
	// General parameters for the interface
	EzGUI gui;
	EzVarDouble gamma_var;
	EzVarBoolean advanced_mode;
	EzVarDouble muInit_var;
	EzVarDouble muStep_var;
	EzVarBoolean isotropic;
	IcySlider prev_scale;
	EzButton segm_button;
	EzButton merge_button;
	EzVarDouble gamma_modification;
	EzButton refine_button;
	EzButton extract_a_roi;
	EzVarBoolean new_window;
	EzButton help_button;
	JProgressBar prog_bar;
	
	
	// Parameters related to the Potts segmentation algorithm
	double[][] weights;
	double[] omega;
	double gamma;
	double muInit;
	double muStep;
	double stopTol;
	boolean verbose;
	boolean multiThreaded;
	boolean useADMM;
	
	
	/**
     * Initialize the Graphical User Interface.
     * @return
     */
	@Override
	protected void initialize () {
		gui = super.getUI();
		
		gui.setTitle("Potts Segmentation");
		gui.setActionPanelVisible(false);
		
		gamma_var = new EzVarDouble("Scale parameter (\u03B3)", 0.9, 0, Double.POSITIVE_INFINITY, 1e-2);
		
		advanced_mode = new EzVarBoolean("Advanced mode", false);
		muInit_var = new EzVarDouble("\u03BC (initial value)", 1e-3, 1e-4, 1e-2, 1e-4);
		muStep_var = new EzVarDouble("\u03BC (step value)", 2, 1.01, 2, 1e-2);
		isotropic = new EzVarBoolean("Isotropism", true);
		
		prev_scale = new IcySlider(1, 100, 25);
		prev_scale.setPaintLabels(true);
		prev_scale.setPaintTicks(true);
		
		segm_button = new EzButton("Segment", segment);
		segm_button.setToolTipText("Choose with the slider the size of the segmented image as a percentage of the original one");
		
		merge_button = new EzButton("Merge selected regions", merge);
		merge_button.setToolTipText("Mark with the ROI tool at least 2 segments for merging. Segments that belong to the same ROI will be merged");
		
		gamma_modification = new EzVarDouble("\u03B3 percentage for refinement", 50, 0, 100, 1e-1);
		refine_button = new EzButton("Refine selected regions", refine);
		refine_button.setToolTipText("Refinement is done using the new \u03B3 chosen above as a percentage of the one on the top");
		
		extract_a_roi = new EzButton("Extract marked segment as a ROI", roi);
		extract_a_roi.setToolTipText("Mark inside a segment with a ROI");
				
		new_window = new EzVarBoolean("Open result in new window", false);
		
		help_button = new EzButton("Online Help", help);
		help_button.setToolTipText("Access the online documentation for the plug-in");
		
		prog_bar = new JProgressBar();
		prog_bar.setBorderPainted(true);
		prog_bar.setStringPainted(true);
		prog_bar.setString("Status: Ready");
		
		EzGroup groupParameters = new EzGroup("Model parameters", gamma_var, advanced_mode, muInit_var, muStep_var, isotropic);
		EzGroup groupFunctions = new EzGroup("Edit segmentation", merge_button, gamma_modification, refine_button, extract_a_roi, new_window);
				
		super.addEzComponent(groupParameters);
		super.addComponent(prev_scale);
		super.addEzComponent(segm_button);
		super.addEzComponent(groupFunctions);
		super.addEzComponent(help_button);
		super.addComponent(prog_bar);
		
		advanced_mode.addVisibilityTriggerTo(muInit_var, true);
		advanced_mode.addVisibilityTriggerTo(muStep_var, true);
		advanced_mode.addVisibilityTriggerTo(isotropic, true);
		
		ImageTools.updateColorsLUT();
	}
	
	
	/**
     * Response to the button 'Segment' of the interface.
     * @return
     */
	ActionListener segment = new ActionListener () {
		@Override
		public void actionPerformed(ActionEvent segment_event)
		{
			prog_bar.setIndeterminate(true);
			prog_bar.setString("Status: Image is being processed.....");
			final AnnounceFrame frm  = new AnnounceFrame("This process may take several seconds to conclude, depending on the image type and size.", 15);
			Timer timer = new Timer();
            timer.schedule(new TimerTask()
            {
                @Override
                public void run()
                {
                	execute();
                    Timer timer_2 = new Timer();
                    timer_2.schedule(new TimerTask()
                    {
                        @Override
                        public void run()
                        {
                        	finilize_processing(frm);
                        }
                    }, 1);
                }
            }, 1);
		}
	};
	
	
	/**
     * Execute the main function of the program to perform 'Segment'.
     * @return
     */
	@Override
	protected void execute () {
		Sequence s_in = getActiveSequence();
		
		if(s_in == null) {
			MessageDialog.showDialog("This function needs an opened sequence.", MessageDialog.ERROR_MESSAGE);
			return;
		}
				
		IcyBufferedImage im_in = s_in.getImage(0, 0);
		IcyBufferedImage im_out;
		
		// Compute the scaling factor according to the use choice
		double factor = prev_scale.getValue() / 100.0;
		
		// Downsampling of the initial image
		IcyBufferedImage im_in_down = ImageTools.resample(im_in, factor);
		
		// Find the dynamic range of the image
		double[] original_range = ImageTools.findDynamicRange(im_in_down);
		
		// Apply the segmentation algorithm on the downsampled image
		gamma = gamma_var.getValue() * factor;
		if (advanced_mode.getValue() == false) {
			muInit = gamma * 1e-2;
			muStep = 2;
			isotropic.setValue(true);
		}
		else {
			muInit = muInit_var.getValue() * factor;
			muStep = muStep_var.getValue();
		}
		stopTol = 1e-10;
		verbose = false;
		multiThreaded = true;
		useADMM = true;
		weights = new double[im_in_down.getSizeX()][im_in_down.getSizeY()];
		for (double[] cur_row: weights)
		    Arrays.fill(cur_row, 1.0);
		omega  = new double[2];
		
		if(!(isotropic.getValue())) {
			PLImage im_in_PL = ImageTools.IcytoPLImage(im_in_down, original_range);
			PLImage im_out_PL = JavaTools.minL2PottsADMM4(im_in_PL, gamma, weights, muInit, muStep, stopTol, verbose, multiThreaded, useADMM);
			im_out = ImageTools.PLtoIcyImage(im_out_PL, original_range, im_in_down.getDataType_());
		}
		else {
			omega[0] = Math.sqrt(2.0) - 1.0;
			omega[1] = 1.0 - Math.sqrt(2.0) / 2.0;
			PLImage im_in_PL = ImageTools.IcytoPLImage(im_in_down, original_range);
			PLImage im_out_PL = JavaTools.minL2PottsADMM8(im_in_PL, gamma, weights, muInit, muStep, stopTol, verbose, multiThreaded, useADMM, omega);
			im_out = ImageTools.PLtoIcyImage(im_out_PL, original_range, im_in_down.getDataType_());
		}
		
		// Find the boundaries of the segmentation image
		IcyBufferedImage im_out_boundaries = ImageTools.findBoundaries(im_out);
		
		// Randomize the gray levels of the segmentation result
		IcyBufferedImage im_out_rand;
		if(im_in.getSizeC() == 1) {
			im_out_rand = ImageTools.randomColors(im_out);
		}
		else {
			IcyBufferedImage im_out_avg = ImageTools.averageChannels(im_out);
			im_out_rand = ImageTools.randomColors(im_out_avg);
		}
		
		// Create the stack of the images and fill with the first three images
		Sequence s_temp = new Sequence();
		s_temp.setImage(0, 0, im_in_down);
		s_temp.setImage(0, 1, im_out);
		s_temp.setImage(0, 2, im_out_boundaries);
				
		// Compute the saturation image and create the image list that will be transformed to RGB
		IcyBufferedImage S = ImageTools.createSaturationImage(im_in_down, im_out_boundaries);
		ArrayList<IcyBufferedImage> im_to_transform = new ArrayList<IcyBufferedImage>();
		im_to_transform.add(im_out_rand);
		im_to_transform.add(S);
		if(im_in.getSizeC() == 1)
			im_to_transform.add(im_in_down);
		else {
			IcyBufferedImage im_in_down_avg = ImageTools.averageChannels(im_in_down);
			im_to_transform.add(im_in_down_avg);
		}
		
		// Transform from the HSV color representation to the RGB one
		IcyBufferedImage rgb_out = ImageTools.convertFromHSV(im_to_transform);
		
		// Fill the image stack with the last image and show the result on the screen
		Sequence s_out = new Sequence();
		if(im_in.getSizeC() == 1) {
			Sequence[] seq = new Sequence[]{s_temp, s_temp, s_temp};
			s_out = SequenceUtil.concatC(seq);
		}
		else
			s_out = s_temp;
		s_out.setImage(0, 3, rgb_out);
		s_out.setName(s_in.getName() + " - Segmented & HSV");
		addSequence(s_out);
	}
	
	
	/**
     * Response to the button 'Merge selected regions' of the interface.
     * @return
     */
	ActionListener merge = new ActionListener () {
		@Override
		public void actionPerformed(ActionEvent merge_event)
		{
			prog_bar.setIndeterminate(true);
			prog_bar.setString("Status: Image is being processed.....");
			final AnnounceFrame frm = new AnnounceFrame("This process may take several seconds to conclude, depending on the image type and size.", 15);
			Timer timer = new Timer();
            timer.schedule(new TimerTask()
            {
                @Override
                public void run()
                {
                	merge_process();
                	Timer timer_2 = new Timer();
                    timer_2.schedule(new TimerTask()
                    {
                        @Override
                        public void run()
                        {
                        	finilize_processing(frm);
                        }
                    }, 1);
                }
            }, 1);
		}
	};
	
	
	/**
     * Execute the function of the program to perform 'Merge'.
     * @return
     */
	protected void merge_process () {
		Sequence s_in = getActiveSequence();
		Sequence s_temp = SequenceUtil.getCopy(s_in);
		
		if(s_in == null) {
			MessageDialog.showDialog("This function needs an opened sequence.", MessageDialog.ERROR_MESSAGE);
			return;
		}
		
		// Use for the processing the segmentation image from the image stack
		IcyBufferedImage im_in = s_temp.getImage(0, 1);
		if(ImageTools.equalChannels(im_in)) {
			s_temp = SequenceUtil.extractChannel(s_in, 0);
			im_in = s_temp.getImage(0, 1);
		}
		
		// Find the dynamic range of the image
		ImageTools.findDynamicRange(s_temp.getImage(0, 0));
					
		ArrayList<ROI2D> rois = s_in.getROI2Ds();
		
		if(rois.isEmpty()) {
			MessageDialog.showDialog("There was no ROI set. Please set a separate ROI with at least 2 segments for each new segment that you want to create.");
			return;
		}
		else {
			IcyBufferedImage im_out =  IcyBufferedImageUtil.getCopy(im_in);
			int rois_num = rois.size();
			int nx = im_in.getSizeX();
			int ny = im_in.getSizeY();
			int nchannels = im_in.getSizeC();
			
			// Repeat the process of merging for every ROI set by the user
			for (int cur_roi = 0; cur_roi < rois_num; cur_roi++) {
				BooleanMask2D mask = rois.get(cur_roi).getBooleanMask(true);
				ArrayList<double[]> roi_values = new ArrayList<double[]>();
				ArrayList<int[]> roi_coord = new ArrayList<int[]>();
				double[] avg_pixel_value = new double[3];
				
				// Get the values and the coordinates of the pixels that belong to the current ROI 
				for (int i = 0; i < nx; i++)
					for (int j = 0; j < ny; j++)
						if(mask.contains(i, j)) {
							double[] cur_pixel_value = new double[nchannels];
							int[] cur_pixel_coord = new int[]{i, j};
							for (int c = 0; c < nchannels; c++)
								cur_pixel_value[c] = im_out.getData(cur_pixel_coord[0], cur_pixel_coord[1], c);
							roi_values.add(cur_pixel_value);
							roi_coord.add(cur_pixel_coord);
						}
				
				ArrayList<ArrayList<int[]>> pixel_per_segment = new ArrayList<ArrayList<int[]>>();
				int total_processed_pixels = 0;
				
				// Flood around each pixel of the ROI in order to locate of the pixels that belong to the segments for merging
				im_out.beginUpdate();
				try {
					for (int i = 0; i < roi_coord.size(); i++)
						pixel_per_segment.add(ImageTools.floodValue(im_out, roi_coord.get(i), roi_values.get(i)));
				}
				finally {
					im_out.endUpdate();
				}
				
				// Calculate the new value for the pixels after the merging as the weighted average of their values before the processing
				for (int i = 0; i < roi_coord.size(); i++) {
					for (int c = 0; c < nchannels; c++)
						avg_pixel_value[c] += pixel_per_segment.get(i).size() * roi_values.get(i)[c];
					total_processed_pixels += pixel_per_segment.get(i).size();
				}
				for (int c = 0; c < nchannels; c++)
					avg_pixel_value[c] = Math.round(avg_pixel_value[c] / total_processed_pixels);
				
				// Set the new value for all the pixels that belong to the segments for merging
				im_out.beginUpdate();
				try {
					for (int k = 0; k < pixel_per_segment.size(); k++)
						for (int l = 0; l < pixel_per_segment.get(k).size(); l++)
							for (int c = 0; c < nchannels; c++)
								im_out.setData(pixel_per_segment.get(k).get(l)[0], pixel_per_segment.get(k).get(l)[1], c, avg_pixel_value[c]);
				}
				finally {
					im_out.endUpdate();
				}
			}
			
			// Find the boundaries of the segmentation image after merging
			IcyBufferedImage im_out_boundaries = ImageTools.findBoundaries(im_out);
			
			// Randomize the gray levels of the new segmentation image
			IcyBufferedImage im_out_rand;
			if(im_in.getSizeC() == 1) {
				im_out_rand = ImageTools.randomColors(im_out);
			}
			else {
				IcyBufferedImage im_out_avg = ImageTools.averageChannels(im_out);
				im_out_rand = ImageTools.randomColors(im_out_avg);
			}
			
			s_temp.setImage(0, 1, im_out);
			s_temp.setImage(0, 2, im_out_boundaries);
			if(s_temp.getSizeZ() == 4)
				s_temp.removeImage(0, 3);
			
			// In case that the user wants the result wants the result on the same window
			if(!(new_window.getValue())) {
				if(im_in.getSizeC() == 1) {
					Sequence[] seq = new Sequence[]{s_temp, s_temp, s_temp};
					s_temp = SequenceUtil.concatC(seq);
				}

				// Remove from the stack some of the data before the processing and update it with the first images
				s_in.beginUpdate();
				s_in.removeAllImages();
				s_in.removeAllROI();
				s_in.setImage(0, 0, s_temp.getImage(0, 0));
				s_in.setImage(0, 1, s_temp.getImage(0, 1));
				s_in.setImage(0, 2, s_temp.getImage(0, 2));
				
				// Compute the saturation image and create the image list that will be transformed to RGB
				IcyBufferedImage S = ImageTools.createSaturationImage(s_temp.getImage(0, 0), im_out_boundaries);
				ArrayList<IcyBufferedImage> im_to_transform = new ArrayList<IcyBufferedImage>();
				im_to_transform.add(im_out_rand);
				im_to_transform.add(S);
				if(im_in.getSizeC() == 1)
					im_to_transform.add(s_temp.getImage(0, 0));
				else {
					IcyBufferedImage im_in_avg = ImageTools.averageChannels(s_temp.getImage(0, 0));
					im_to_transform.add(im_in_avg);
				}
				
				// Transform from the HSV color representation to the RGB one
				IcyBufferedImage rgb_out = ImageTools.convertFromHSV(im_to_transform);
				
				// Put the last image in the stack and show the output on the screen
				s_in.setImage(0, 3, rgb_out);
				s_in.setName(s_in.getName() + " - Merged");
				s_in.endUpdate();
			}
			
			// In case that the user want the result in a new window
			else {
				// Create a new sequence
				Sequence s_out = new Sequence();
				if(im_in.getSizeC() == 1) {
					Sequence[] seq = new Sequence[]{s_temp, s_temp, s_temp};
					s_out = SequenceUtil.concatC(seq);
				}
				else {
					s_out = s_temp;
					s_out.removeAllROI();
				}
				
				// Compute the saturation image and create the image list that will be transformed to RGB
				IcyBufferedImage S = ImageTools.createSaturationImage(s_temp.getImage(0, 0), im_out_boundaries);
				ArrayList<IcyBufferedImage> im_to_transform = new ArrayList<IcyBufferedImage>();
				im_to_transform.add(im_out_rand);
				im_to_transform.add(S);
				if(im_in.getSizeC() == 1)
					im_to_transform.add(s_temp.getImage(0, 0));
				else {
					IcyBufferedImage im_in_avg = ImageTools.averageChannels(s_temp.getImage(0, 0));
					im_to_transform.add(im_in_avg);
				}
				
				// Transform from the HSV color representation to the RGB one
				IcyBufferedImage rgb_out = ImageTools.convertFromHSV(im_to_transform);
				
				// Put the last image in the stack and show the output on the screen
				s_out.setImage(0, 3, rgb_out);
				s_out.setName(s_in.getName() + " - Merged");
				addSequence(s_out);
			}
		}
	}
	
	
	/**
     * Response to the button 'Refine selected regions' of the interface.
     * @return
     */
	ActionListener refine = new ActionListener () {
		@Override
		public void actionPerformed(ActionEvent refine_event)
		{
			prog_bar.setIndeterminate(true);
			prog_bar.setString("Status: Image is being processed.....");
			final AnnounceFrame frm = new AnnounceFrame("This process may take several seconds to conclude, depending on the image type and size.", 15);
			Timer timer = new Timer();
            timer.schedule(new TimerTask()
            {
                @Override
                public void run()
                {
                	refine_process();
                	Timer timer_2 = new Timer();
                    timer_2.schedule(new TimerTask()
                    {
                        @Override
                        public void run()
                        {
                        	finilize_processing(frm);
                        }
                    }, 1);
                }
            }, 1);
		}
	};
	
	
	/**
     * Execute the function of the program to perform 'Refine'.
     * @return
     */
	protected void refine_process () {
		Sequence s_in = getActiveSequence();
		Sequence s_temp = SequenceUtil.getCopy(s_in);
		
		if(s_in == null) {
			MessageDialog.showDialog("This function needs an opened sequence.", MessageDialog.ERROR_MESSAGE);
			return;
		}
		
		// Use for the processing the segmentation image from the image stack
		IcyBufferedImage im_in = s_temp.getImage(0, 1);
		if(ImageTools.equalChannels(im_in)) {
			s_temp = SequenceUtil.extractChannel(s_in, 0);
			im_in = s_temp.getImage(0, 1);
		}
		
		// Find the dynamic range of the image
		double[] original_range = ImageTools.findDynamicRange(s_temp.getImage(0, 0));
					
		ArrayList<ROI2D> rois = s_in.getROI2Ds();
					
		if(rois.isEmpty()) {
			MessageDialog.showDialog("There was no ROI set. Please set a ROI that is totally inside the segment that you want to refine.");
			return;
		}
		else {
			IcyBufferedImage temp_im =  IcyBufferedImageUtil.getCopy(im_in);
			IcyBufferedImage temp_im_resized;
			int rois_num = rois.size();
			int nchannels = im_in.getSizeC();
			
			// Repeat the refinement process for every ROI set by the user 
			for (int cur_roi = 0; cur_roi < rois_num; cur_roi++) {
				ArrayList<int[]> pixels_in_resized_segment = new ArrayList<int[]>();
				double[] roi_values = new double[nchannels];
				int[] roi_coord = new int[2];
				int[] roi_coord_resized = new int[2];
				double image_size_ratio = (double) temp_im.getSizeX() / ((double) s_temp.getImage(0, 0).getSizeX());
				
				// Get the coordinates and the value of one pixel (upper left point of the bounds) from the ROI
				roi_coord[0] = (int) rois.get(cur_roi).getPosition2D().getX();
				roi_coord[1] = (int) rois.get(cur_roi).getPosition2D().getY();
				for (int c = 0; c < nchannels; c++)
					roi_values[c] = temp_im.getData(roi_coord[0], roi_coord[1], c);
				
				temp_im_resized = ImageTools.resample(temp_im, (1.0/image_size_ratio));
				roi_coord_resized[0] = (int)(roi_coord[0] / image_size_ratio);
				roi_coord_resized[1] = (int)(roi_coord[1] / image_size_ratio);
				
				// Flood around the pixel in order to locate of the pixels that belong to this segment
				temp_im_resized.beginUpdate();
				try {
					pixels_in_resized_segment = ImageTools.floodValue(temp_im_resized, roi_coord_resized, roi_values);
				}
				finally {
					temp_im_resized.endUpdate();
				}
				
				// Compute the coordinates of the points that define the minimum rectangle containing the certain segment
				int min_X = ImageTools.arrayMinInDim(pixels_in_resized_segment, 0);
				int min_Y = ImageTools.arrayMinInDim(pixels_in_resized_segment, 1);
				int max_X = ImageTools.arrayMaxInDim(pixels_in_resized_segment, 0);
				int max_Y = ImageTools.arrayMaxInDim(pixels_in_resized_segment, 1);
				
				// Extract the sub-image of the minimum rectangle
				IcyBufferedImage im_for_proc = IcyBufferedImageUtil.getSubImage(s_temp.getImage(0, 0), 
						new Rectangle(min_X, min_Y, max_X-min_X+1, max_Y-min_Y+1));
				IcyBufferedImage im_after_proc;
				
				// Apply the segmentation algorithm on the sub-image
				gamma = gamma_var.getValue() * gamma_modification.getValue() / 100.0;
				if (advanced_mode.getValue() == false) {
					muInit = gamma * 1e-2;
					muStep = 2;
					isotropic.setValue(true);
				}
				else {
					muInit = muInit_var.getValue();
					muStep = muStep_var.getValue();
				}
				stopTol = 1e-10;
				verbose = false;
				multiThreaded = true;
				useADMM = true;
				weights = new double[im_for_proc.getSizeX()][im_for_proc.getSizeY()];
				for (double[] cur_row: weights)
				    Arrays.fill(cur_row, 1.0);
				omega  = new double[2];
					
				if(!(isotropic.getValue())) {
					PLImage im_in_PL = ImageTools.IcytoPLImage(im_for_proc, original_range);
					PLImage im_out_PL = JavaTools.minL2PottsADMM4(im_in_PL, gamma, weights, muInit, muStep, stopTol, verbose, multiThreaded, useADMM);
					im_after_proc = ImageTools.PLtoIcyImage(im_out_PL, original_range, im_for_proc.getDataType_());
				}
				else {
					omega[0] = Math.sqrt(2.0) - 1.0;
					omega[1] = 1.0 - Math.sqrt(2.0) / 2.0;
					PLImage im_in_PL = ImageTools.IcytoPLImage(im_for_proc, original_range);
					PLImage im_out_PL = JavaTools.minL2PottsADMM8(im_in_PL, gamma, weights, muInit, muStep, stopTol, verbose, multiThreaded, useADMM, omega);
					im_after_proc = ImageTools.PLtoIcyImage(im_out_PL, original_range, im_for_proc.getDataType_());
				}
				
				// Paste the new pixel values from the segment(s) on the given image in order to form the result after the refinement
				temp_im_resized.beginUpdate();
				try {
					for (int i = 0; i < im_after_proc.getSizeX(); i++)
						for (int j = 0; j < im_after_proc.getSizeY(); j++) {
							int[] cur_pixel = {i+min_X, j+min_Y};
							if(ImageTools.indexInArray(pixels_in_resized_segment, cur_pixel) != -1)
								for (int c = 0; c < nchannels; c++)
									temp_im_resized.setData(i+min_X, j+min_Y, c, im_after_proc.getData(i, j, c));
						}
				}
				finally {
					temp_im_resized.endUpdate();
				}
				
				temp_im = ImageTools.resample(temp_im_resized, image_size_ratio);
			}
			
			// Find the boundaries of the refined image
			IcyBufferedImage im_out_boundaries = ImageTools.findBoundaries(temp_im);
			
			// Randomize the gray levels for the refined image
			IcyBufferedImage temp_im_rand;
			if(im_in.getSizeC() == 1) {
				temp_im_rand = ImageTools.randomColors(temp_im);
			}
			else {
				IcyBufferedImage temp_im_avg = ImageTools.averageChannels(temp_im);
				temp_im_rand = ImageTools.randomColors(temp_im_avg);
			}
			
			s_temp.setImage(0, 1, temp_im);
			s_temp.setImage(0, 2, im_out_boundaries);
			if(s_temp.getSizeZ() == 4)
				s_temp.removeImage(0, 3);
			
			// In case that the user wants the result in the same window
			if(!(new_window.getValue())) {
				if(im_in.getSizeC() == 1) {
					Sequence[] seq = new Sequence[]{s_temp, s_temp, s_temp};
					s_temp = SequenceUtil.concatC(seq);
				}
				
				// Remove from the stack some of the data before the processing and update it with the first images
				s_in.beginUpdate();
				s_in.removeAllImages();
				s_in.removeAllROI();
				s_in.setImage(0, 0, s_temp.getImage(0, 0));
				s_in.setImage(0, 1, s_temp.getImage(0, 1));
				s_in.setImage(0, 2, s_temp.getImage(0, 2));
				
				// Compute the saturation image and create the image list that will be transformed to RGB
				IcyBufferedImage S = ImageTools.createSaturationImage(s_temp.getImage(0, 0), im_out_boundaries);
				ArrayList<IcyBufferedImage> im_to_transform = new ArrayList<IcyBufferedImage>();
				im_to_transform.add(temp_im_rand);
				im_to_transform.add(S);
				if(im_in.getSizeC() == 1)
					im_to_transform.add(s_temp.getImage(0, 0));
				else {
					IcyBufferedImage im_in_avg = ImageTools.averageChannels(s_temp.getImage(0, 0));
					im_to_transform.add(im_in_avg);
				}
				
				// Transform from the HSV color representation to the RGB one
				IcyBufferedImage rgb_out = ImageTools.convertFromHSV(im_to_transform);
				
				// Put the last image in the stack and show the output on the screen
				s_in.setImage(0, 3, rgb_out);
				s_in.setName(s_in.getName() + " - Refined");
				s_in.endUpdate();
			}
			
			// In case that the user wants the result in anew window
			else {
				// Create a new sequence
				Sequence s_out = new Sequence();
				if(im_in.getSizeC() == 1) {
					Sequence[] seq = new Sequence[]{s_temp, s_temp, s_temp};
					s_out = SequenceUtil.concatC(seq);
				}
				else {
					s_out = s_temp;
					s_out.removeAllROI();
				}
				
				// Compute the saturation image and create the image list that will be transformed to RGB
				IcyBufferedImage S = ImageTools.createSaturationImage(s_temp.getImage(0, 0), im_out_boundaries);
				ArrayList<IcyBufferedImage> im_to_transform = new ArrayList<IcyBufferedImage>();
				im_to_transform.add(temp_im_rand);
				im_to_transform.add(S);
				if(im_in.getSizeC() == 1)
					im_to_transform.add(s_temp.getImage(0, 0));
				else {
					IcyBufferedImage im_in_avg = ImageTools.averageChannels(s_temp.getImage(0, 0));
					im_to_transform.add(im_in_avg);
				}
				
				// Transform from the HSV color representation to the RGB one
				IcyBufferedImage rgb_out = ImageTools.convertFromHSV(im_to_transform);
				
				// Put the last image in the stack and show the output on the screen
				s_out.setImage(0, 3, rgb_out);
				s_out.setName(s_in.getName() + " - Refined");
				addSequence(s_out);
			}
		}
	}
	
	
	/**
     * Response to the button 'Extract marked segment as a ROI' of the interface.
     * @return
     */
	ActionListener roi = new ActionListener () {
		@Override
		public void actionPerformed(ActionEvent roi_event)
		{
			prog_bar.setIndeterminate(true);
			prog_bar.setString("Status: Image is being processed.....");
			final AnnounceFrame frm = new AnnounceFrame("This process may take several seconds to conclude, depending on the image type and size.", 15);
			Timer timer = new Timer();
            timer.schedule(new TimerTask()
            {
                @Override
                public void run()
                {
                	extract_roi_process();
                	Timer timer_2 = new Timer();
                    timer_2.schedule(new TimerTask()
                    {
                        @Override
                        public void run()
                        {
                        	finilize_processing(frm);
                        }
                    }, 1);
                }
            }, 1);
		}
	};
	
	
	/**
     * Execute the function of the program to perform 'Extraction of marked segment as a ROI'.
     * @return
     */
	protected void extract_roi_process () {
		Sequence s_in = getActiveSequence();
		Sequence s_temp = SequenceUtil.getCopy(s_in);
		
		if(s_in == null) {
			MessageDialog.showDialog("This function needs an opened sequence.", MessageDialog.ERROR_MESSAGE);
			return;
		}
		
		// Use for the processing the segmentation image from the image stack
		IcyBufferedImage im_in = s_temp.getImage(0, 1);
		if(ImageTools.equalChannels(im_in)) {
			s_temp = SequenceUtil.extractChannel(s_in, 0);
			im_in = s_temp.getImage(0, 1);
		}
		
		// Find the dynamic range of the image
		ImageTools.findDynamicRange(s_temp.getImage(0, 0));
					
		ArrayList<ROI2D> rois = s_in.getROI2Ds();
		
		if(rois.isEmpty()) {
			MessageDialog.showDialog("There was no ROI set. Please set a separate ROI for each segment that you want to extract.");
			return;
		}
		else {
			IcyBufferedImage im_out =  IcyBufferedImageUtil.getCopy(im_in);
			int rois_num = rois.size();
			int nx = im_in.getSizeX();
			int ny = im_in.getSizeY();
			int nchannels = im_in.getSizeC();
			
			// Repeat the process of merging for every ROI set by the user
			for (int cur_roi = 0; cur_roi < rois_num; cur_roi++) {
				BooleanMask2D mask = rois.get(cur_roi).getBooleanMask(true);
				ROI2DArea final_roi = new ROI2DArea();
				ArrayList<double[]> roi_values = new ArrayList<double[]>();
				ArrayList<int[]> roi_coord = new ArrayList<int[]>();
				
				// Get the values and the coordinates of the pixels that belong to the current ROI
				if(mask.isEmpty()) {
					double[] cur_pixel_value = new double[nchannels];
					int[] cur_pixel_coord = new int[2];
					cur_pixel_coord[0] = (int) rois.get(cur_roi).getPosition2D().getX();
					cur_pixel_coord[1] = (int) rois.get(cur_roi).getPosition2D().getY();
					for (int c = 0; c < nchannels; c++)
						cur_pixel_value[c] = im_out.getData(cur_pixel_coord[0], cur_pixel_coord[1], c);
					roi_values.add(cur_pixel_value);
					roi_coord.add(cur_pixel_coord);
				}
				else {
					for (int i = 0; i < nx; i++)
						for (int j = 0; j < ny; j++)
							if(mask.contains(i, j)) {
								double[] cur_pixel_value = new double[nchannels];
								int[] cur_pixel_coord = new int[]{i, j};
								for (int c = 0; c < nchannels; c++)
									cur_pixel_value[c] = im_out.getData(cur_pixel_coord[0], cur_pixel_coord[1], c);
								roi_values.add(cur_pixel_value);
								roi_coord.add(cur_pixel_coord);
							}
				}
				
				ArrayList<ArrayList<int[]>> pixel_per_segment = new ArrayList<ArrayList<int[]>>();
				
				// Flood around each pixel of the ROI in order to locate of the pixels that belong to the segments for extraction
				im_out.beginUpdate();
				try {
					for (int i = 0; i < roi_coord.size(); i++)
						pixel_per_segment.add(ImageTools.floodValue(im_out, roi_coord.get(i), roi_values.get(i)));
				}
				finally {
					im_out.endUpdate();
				}
				
				for (int i = 0; i < pixel_per_segment.size(); i++)
					for (int j = 0; j < pixel_per_segment.get(i).size(); j++) {
							int[] temp = new int[2];
							temp = pixel_per_segment.get(i).get(j);
							final_roi.addPoint(temp[0], temp[1]);
					}
				
				s_in.addROI(final_roi);
			}			
		}
	}
	
	
	/**
     * Response to the button 'Online Help' of the interface.
     * @return
     */
	ActionListener help = new ActionListener () {
		@Override
		public void actionPerformed(ActionEvent help_event)
		{
			NetworkUtil.openBrowser("http://icy.bioimageanalysis.org/plugin/Potts_Segmentation");
		}
	};
	
	
	/**
     * Finalize the processing of the image after the execution of the processing functions.
     * @return
     */
	private void finilize_processing ( AnnounceFrame frm ) {
		frm.close();
		prog_bar.setString("Status: Ready");
		prog_bar.setIndeterminate(false);
	}
	
	
	/**
     * Clear any possible garbage created during the execution of the program.
     * @return
     */
	@Override
	public void clean () {
		
	}
	
}