package plugins.angelopo.pottslab;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.Queue;
import java.util.Random;

import icy.image.IcyBufferedImage;
import icy.image.IcyBufferedImageUtil;
import icy.type.DataType;

/**
 * Class implementing useful tools for image processing using the Potts model 
 * 
 * @author Vasileios Angelopoulos
 * 
 */

public class ImageTools
{
	
	// Look-up table that contains the gray levels with their corresponding random gray value
	private static ArrayList<Double> random_colors_LUT;
	private static double[] image_dynamic_range = new double[] {0.0, 25000.0};
	
	
	/**
     * Initialization of the look-up table with the random gray levels.
     * @return
     */
	public static void updateColorsLUT () {
		double max_value = image_dynamic_range[1];
		double min_value = image_dynamic_range[0];
		double new_value;
		Random r = new Random();
		
		random_colors_LUT = new ArrayList<Double>();
		
		for(int i = 0; i < (int)(max_value-min_value+1); i++) {
			do {
				new_value = min_value + (max_value - min_value) * r.nextDouble();
			} while(indexInArray(random_colors_LUT, new_value) != -1);
			random_colors_LUT.add(new_value);
		}
	}
	
	
	/**
     * Find the dynamic range of an image.
     * @param in
     * @return
     */
	public static double[] findDynamicRange ( IcyBufferedImage in ) {
		int nchannels = in.getSizeC();
		double cur_max_value, max_value = Double.NEGATIVE_INFINITY;
		double cur_min_value, min_value = Double.POSITIVE_INFINITY;
		
		for(int c = 0; c < nchannels; c++) {
			cur_max_value = in.getChannelMax(c);
			cur_min_value = in.getChannelMin(c);
			if(cur_max_value > max_value)
				max_value = cur_max_value;
			if(cur_min_value < min_value)
				min_value = cur_min_value;
		}
		
		image_dynamic_range[0] = min_value;
		image_dynamic_range[1] = max_value;
		
		updateColorsLUT();
		
		return new double[] {min_value, max_value};
	}
	
	
	/**
     * Scale an image according to a given factor.
     * @param in
     * @param factor
     * @return
     */
	public static IcyBufferedImage resample ( IcyBufferedImage in, double factor ) {
		int nx = in.getSizeX();
		int ny = in.getSizeY();
		int nchannels = in.getSizeC();
		DataType type = in.getDataType_();
		int new_nx = (int)Math.round(nx*factor);
		int new_ny = (int)Math.round(ny*factor);
		IcyBufferedImage out = new IcyBufferedImage(new_nx, new_ny, nchannels, type);
		
		out.beginUpdate();
		try {
			out = IcyBufferedImageUtil.scale(in, new_nx, new_ny);
		}
		finally {
			out.endUpdate();
		}
		
		return out;
	}
	
	
	/**
     * Transform an image from the IcyBufferedImage type to the PLImage one.
     * @param in
     * @param original_range
     * @return
     */
	public static PLImage IcytoPLImage ( IcyBufferedImage in, double[] original_range ) {
		int nx = in.getSizeX();
		int ny = in.getSizeY();
		int nchannels = in.getSizeC();
		double[][][] temp_array = new double[nx][ny][nchannels];;
		
		for (int i = 0; i < nx; i++)
			for (int j = 0; j < ny; j++)
				for (int k = 0; k < nchannels; k++)
					// All PLImage images have dynamic range 1.0
					temp_array[i][j][k] = in.getData(i, j, k) / original_range[1];
		
		return new PLImage(temp_array);
	}
	
	
	/**
     * Transform an image from the PLImage type to the IcyBufferedImage one.
     * @param in
     * @param original_range
     * @param type
     * @return
     */
	public static IcyBufferedImage PLtoIcyImage ( PLImage in, double[] original_range, DataType type ) {
		int nx = in.mRow;
		int ny = in.mCol;
		int nchannels = in.mLen;
		double[][][] temp_array;
		IcyBufferedImage out = new IcyBufferedImage(nx, ny, nchannels, type);
		
		temp_array = in.toDouble3D();
		
		out.beginUpdate();
		try {
			for (int i = 0; i < nx; i++)
				for (int j = 0; j < ny; j++)
					for (int k = 0; k < nchannels; k++) {
						// Set back the dynamic range from 1.0 to the one of the IcyBufferedImage image
						temp_array[i][j][k] = temp_array[i][j][k] * original_range[1];
						out.setData(i, j, k, temp_array[i][j][k]);
					}
		}
		finally {
			out.endUpdate();
		}
			
		return out;
	}
	
	
	/**
     * Find the index of an element in a given Array list containing doubles.
     * @param array
     * @param element
     * @return
     */
	public static int indexInArray ( ArrayList<Double> array, double element ) {
		int index = -1;
		
		for (int k = 0; k < array.size(); k++)
			if(array.get(k) == element) {
				index = k;
				break;
			}
		
		return index;
	}
	
	
	/**
     * Find the index of an element in a given Array list containing arrays of integers.
     * @param array
     * @param element
     * @return
     */
	public static int indexInArray ( ArrayList<int[]> array, int[] element ) {
		int index = -1;
		boolean found = false;
		
		for (int k = 0; k < array.size(); k++) {
			int[] cur_element = array.get(k);
			for (int c = 0; c < element.length; c++) {
				if(cur_element[c] != element[c])
					break;
				if(c == (element.length - 1))
					found = true;
			}
			if(found) {
				index = k;
				break;
			}
		}
		
		return index;
	}
	
	
	/**
     * Find the minimum value for a certain dimension in a given Array list containing arrays of integers.
     * @param array
     * @param dim
     * @return
     */
	public static int arrayMinInDim ( ArrayList<int[]> array, int dim ) {
		int min = Integer.MAX_VALUE;
		
		for (int k = 0; k < array.size(); k++) {
			int[] cur_element = array.get(k);
			if(cur_element[dim] < min)
				min = cur_element[dim];
		}
		
		return min;
	}
	
	
	/**
     * Find the maximum value for a certain dimension in a given Array list containing arrays of integers.
     * @param array
     * @param dim
     * @return
     */
	public static int arrayMaxInDim ( ArrayList<int[]> array, int dim ) {
		int max = Integer.MIN_VALUE;
		
		for (int k = 0; k < array.size(); k++) {
			int[] cur_element = array.get(k);
			if(cur_element[dim] > max)
				max = cur_element[dim];
		}
		
		return max;
	}
	
	
	/**
     * Check if two multi-channel pixel values are equal.
     * @param pixel_value_1
     * @param pixel_value_2
     * @return
     */
	public static boolean equalValues ( double[] pixel_value_1, double[] pixel_value_2 ) {
		boolean equal = true;
		
		if(pixel_value_1.length != pixel_value_2.length)
			return !equal;
		
		for (int k = 0; k < pixel_value_1.length; k++)
			if(pixel_value_1[k] != pixel_value_2[k]) {
				return !equal;
			}
		
		return equal;
	}
	
	
	/**
     * Check if all the channels of a multi-channel image are equal.
     * @param in
     * @return
     */
	public static boolean equalChannels ( IcyBufferedImage in ) {
		int nx = in.getSizeX();
		int ny = in.getSizeY();
		int nchannels = in.getSizeC();
		boolean equal = true;
		
		for(int i = 0; i < nx; i++) {
			for(int j = 0; j < ny; j++) {
				for (int c = 1; c < nchannels; c++)
					if(in.getData(i, j, 0) != in.getData(i, j, c)) {
						equal = false;
						break;
					}
				if(!equal)
					break;
			}
			if(!equal)
				break;
		}
		
		return equal;
	}
	
	
	/**
     * Get the data from a given pixel of an image.
     * @param in
     * @param coord
     * @return
     */
	public static double[] getPixelData ( IcyBufferedImage in, int[] coord ) {
		int nchannels = in.getSizeC();
		double[] pixel_value = new double[nchannels];
		
		for (int c = 0; c < nchannels; c++)
			pixel_value[c] = in.getData(coord[0], coord[1], c);
		
		return pixel_value;
	}
	
	
	/**
     * Flood around a given pixel for all directions, stopping when you find different value than the given one.
     * Store the coordinates of the flooded pixels.
     * @param in
     * @param coord
     * @param cur_pixel_value
     * @return
     */
	public static ArrayList<int[]> floodValue ( IcyBufferedImage in, int[] coord, double[] cur_pixel_value ) {
		int nchannels = in.getSizeC();
		double[] temp_value = new double[] {image_dynamic_range[1], image_dynamic_range[1], image_dynamic_range[1]};
		ArrayList<int[]> segment_pixels = new ArrayList<int[]>();
		
		// In case that the pixel has been already processed, return
		if(equalValues(cur_pixel_value, temp_value))
			return segment_pixels;
		
		else {
			// Create an queue and add the given pixel
			Queue<int[]> Q = new LinkedList<int[]>();
			int[] cur_coord = coord;
			Q.add(cur_coord);
			
			// Repeat while the list is not empty
			while(!Q.isEmpty()) {
				cur_coord = Q.remove();
				if(!equalValues(ImageTools.getPixelData(in, cur_coord), cur_pixel_value))
					continue;
				else {
					int[] w = cur_coord;
					int[] e = new int[] {cur_coord[0]+1, cur_coord[1]};
					
					// Repeat for all the pixel that are western than the current one until you will find a pixel with different value 
					while((w[0] >= 0) && (equalValues(ImageTools.getPixelData(in, w), cur_pixel_value))) {
						for (int c = 0; c < nchannels; c++)
							in.setData(w[0], w[1], c, temp_value[c]);
						int[] temp_coord = {w[0], w[1]};
						segment_pixels.add(temp_coord);
						if((w[1] > 0) && (equalValues(ImageTools.getPixelData(in, new int[] {w[0], w[1]-1}), cur_pixel_value))) {
							Q.add(new int[] {w[0], w[1]-1});
						}
						if((w[1] < in.getSizeY() - 1) && (equalValues(ImageTools.getPixelData(in, new int[] {w[0], w[1]+1}), cur_pixel_value))) {
							Q.add(new int[] {w[0], w[1]+1});
						}
						w[0] = w[0] - 1;
					}
					
					// Repeat for all the pixel that are eastern than the current one until you will find a pixel with different value 
					while((e[0] < in.getSizeX()) && (equalValues(ImageTools.getPixelData(in, e), cur_pixel_value))) {
						for (int c = 0; c < nchannels; c++)
							in.setData(e[0], e[1], c, temp_value[c]);
						int[] temp_coord = {e[0], e[1]};
						segment_pixels.add(temp_coord);
						if((e[1] > 0) && (equalValues(ImageTools.getPixelData(in, new int[] {e[0], e[1]-1}), cur_pixel_value))) {
							Q.add(new int[] {e[0], e[1]-1});
						}
						if((e[1] < in.getSizeY() - 1) && (equalValues(ImageTools.getPixelData(in, new int[] {e[0], e[1]+1}), cur_pixel_value))) {
							Q.add(new int[] {e[0], e[1]+1});
						}
						e[0] = e[0] + 1;
					}
				}
			}
			return segment_pixels;
		}
	}
	
	
	/**
     * Create a random graylevel version of a given image according to the look-up table random_colors_LUT.
     * @param in
     * @return
     */
	public static IcyBufferedImage randomColors ( IcyBufferedImage in ) {
		int nx = in.getSizeX();
		int ny = in.getSizeY();
		int nchannels = in.getSizeC();
		IcyBufferedImage out = IcyBufferedImageUtil.getCopy(in);
		
		out.beginUpdate();
		try {
			for(int c = 0; c < nchannels; c++)
				for(int i = 0; i < nx; i++)
					for(int j = 0; j < ny; j++) {
						double cur_value = random_colors_LUT.get((int)out.getData(i, j, c));
						out.setData(i, j, c, cur_value);
					}
		}
		finally {
			out.endUpdate();
		}
		
		return out;
	}
	
	
	/**
     * Create a saturation image where each pixel has at 40% value the total white and at 60% the 
     * corresponding value of the boundaries image.
     * @param im
     * @param bound
     * @return
     */
	public static IcyBufferedImage createSaturationImage ( IcyBufferedImage im, IcyBufferedImage bound ) {
		double value;
		IcyBufferedImage S = new IcyBufferedImage(im.getSizeX(), im.getSizeY(), 1, im.getDataType_());
		
		S.beginUpdate();
		try {
			for(int i = 0; i < S.getSizeX(); i++)
				for(int j = 0; j < S.getSizeY(); j++) {
					value = 0.4 * image_dynamic_range[1] + 0.6 * bound.getData(i, j, 0);
					S.setData(i, j, 0, value);
				}
		}
		finally {
			S.endUpdate();
		}
		
		return S;
	}
	
	
	/**
     * Transform a pixel from the RGB color representation to the HSV one.
     * @param rgb
     * @return
     */
	public static double[] toHSV ( double[] rgb ) {
		double r = rgb[0];
		double g = rgb[1];
		double b = rgb[2];
		double Cmin, Cmax, Delta;
		double h, s, v;
		
	    Cmin = Math.min(r, Math.min(g, b));
	    Cmax = Math.max(r, Math.max(g, b));
	    
	    // Black color
	    if(Cmax == 0f)
	    	return new double[] {0, 0, 0};
	    
	    v = Cmax;
	    Delta = Cmax - Cmin;
	    s = Delta / Cmax;
	    
	    if(Delta == 0f)
		    return new double[] {0, s, v};
	    
	    // Color between yellow & magenta
	    if(Cmax == r)
		    h = ((g - b) / Delta) % 6;
	    // Color between cyan & yellow
	    else if(Cmax == g)
		    h = (b - r) / Delta + 2.0;
	    // Color between magenta & cyan
	    else
		    h = (r - g) / Delta + 4.0;
	    
	    // Hue has to be positive
	    if(h < 0)
		    h += 6f;
	    
	    return new double[] {h / 6f, s, v};
    }

	
	/**
     * Transform a pixel from the HSV color representation to the RGB one.
     * @param hsv
     * @return
     */
	public static double[] fromHSV ( double[] hsv ) {
		double h = hsv[0];
		double s = hsv[1];
		double v = hsv[2];
		double f, p, q, t;
		double r, g, b;
		int i;
		
		// No color
		if(s == 0f)
			return new double[] {v, v, v};
		
		// Sectors 0 to 5
		h *= 6f;
		i = (int) Math.floor(h);
		
		// Factorial part of h
		f = h - i;
		p = v * (1f - s);
		q = v * (1f - (s * f));
		t = v * (1f - (s * (1 - f)));
		
		switch(i) {
			case 0:
				r = v;
				g = t;
				b = p;
				break;
			case 1:
				r = q;
				g = v;
				b = p;
				break;
			case 2:
				r = p;
				g = v;
				b = t;
				break;
			case 3:
				r = p;
				g = q;
				b = v;
				break;
			case 4:
				r = t;
				g = p;
				b = v;
				break;
			default:
				r = v;
				g = p;
				b = q;
				break;
		}
		
		return new double[] {r, g, b};
	}
	
	
	/**
     * Transform an image from the RGB color representation to the HSV one.
     * @param im_to_transform
     * @return
     */
	public static ArrayList<IcyBufferedImage> convertToHSV ( IcyBufferedImage im_to_transform ) {
		int nx = im_to_transform.getSizeX();
		int ny = im_to_transform.getSizeY();
		int nchannels = im_to_transform.getSizeC();
		DataType type = im_to_transform.getDataType_();
		double[] cur_element = new double[nchannels];
		double[] hsv_element = new double[nchannels];
		ArrayList<IcyBufferedImage> out = new ArrayList<IcyBufferedImage>();
		
		for(int c = 0; c < nchannels; c++)
			out.add(new IcyBufferedImage(nx, ny, 1, type));
		
		for(int c = 0; c < nchannels; c++) {
			out.get(c).beginUpdate();
			try {
				for(int i = 0; i < nx; i++)
					for(int j = 0; j < ny; j++) {
						for(int c1 = 0; c1 < nchannels; c1++)
							cur_element[c1] = (im_to_transform.getData(i, j, c1) - image_dynamic_range[0]) / image_dynamic_range[1];
						hsv_element = toHSV(cur_element);
						out.get(c).setData(i, j, 0, (image_dynamic_range[1] * hsv_element[c]) + image_dynamic_range[0]);
					}
			}
			finally {
				out.get(c).endUpdate();
			}
		}
		
		return out;
	}
	
	
	/**
     * Transform an image from the HSV color representation to the RGB one.
     * @param hsv_image
     * @return
     */
	public static IcyBufferedImage convertFromHSV ( ArrayList<IcyBufferedImage> hsv_image ) {
		int nx = hsv_image.get(0).getSizeX();
		int ny = hsv_image.get(0).getSizeY();
		int nchannels = hsv_image.size();
		DataType type = hsv_image.get(0).getDataType_();
		double[] cur_element = new double[nchannels];
		double[] hsv_element = new double[nchannels];
		IcyBufferedImage out = new IcyBufferedImage(nx, ny, nchannels, type);
		
		out.beginUpdate();
		try {
			for(int i = 0; i < nx; i++)
				for(int j = 0; j < ny; j++) {
					for(int c = 0; c < nchannels; c++)
						hsv_element[c] = ((hsv_image.get(c).getData(i, j, 0)) - image_dynamic_range[0]) / image_dynamic_range[1];
					cur_element = fromHSV(hsv_element);
					for(int c = 0; c < nchannels; c++) {
						out.setData(i, j, c, (image_dynamic_range[1] * cur_element[c]) + image_dynamic_range[0]);
					}
				}
		}
		finally {
			out.endUpdate();
		}
		
		return out;
	}
	
	
	/**
     * Find the boundaries of an image by applying the Laplacian filter and threshold its output.
     * @param in
     * @return
     */
	public static IcyBufferedImage findBoundaries ( IcyBufferedImage in ) {
		int nx = in.getSizeX();
		int ny = in.getSizeY();
		int nchannels = in.getSizeC();
		DataType type = in.getDataType_();
		double[] new_pixel_value = new double[nchannels];
		boolean is_boundary = false;
		double T = 1e-8;
		double white = image_dynamic_range[1];
		double black = image_dynamic_range[0];
		IcyBufferedImage out = new IcyBufferedImage(nx, ny, nchannels, type);
		
		for(int c = 0; c < nchannels; c++)
			new_pixel_value[c] = 0.0;
		
		out.beginUpdate();
		try {
			for(int i = 0; i < nx; i++)
				for(int j = 0; j < ny; j++) {
					
					// Apply the Laplacian filter to the current pixel using mirror boundary conditions
					if(i > 1)
						for(int c = 0; c < nchannels; c++)
							new_pixel_value[c] += in.getData(i-1, j, c);
					else
						for(int c = 0; c < nchannels; c++)
							new_pixel_value[c] += in.getData(i+1, j, c);
					
					if(j > 1)
						for(int c = 0; c < nchannels; c++)
							new_pixel_value[c] += in.getData(i, j-1, c);
					else
						for(int c = 0; c < nchannels; c++)
							new_pixel_value[c] += in.getData(i, j+1, c);
					
					if(i < (nx - 1))
						for(int c = 0; c < nchannels; c++)
							new_pixel_value[c] += in.getData(i+1, j, c);
					else
						for(int c = 0; c < nchannels; c++)
							new_pixel_value[c] += in.getData(i-1, j, c);
					
					if(j < (ny - 1))
						for(int c = 0; c < nchannels; c++)
							new_pixel_value[c] += in.getData(i, j+1, c);
					else
						for(int c = 0; c < nchannels; c++)
							new_pixel_value[c] += in.getData(i, j-1, c);
					
					for(int c = 0; c < nchannels; c++)
						new_pixel_value[c] += (-4) * in.getData(i, j, c);
					
					// Check if a given pixel is has larger value than the threshold and accordingly choose its color
					for (int c = 0; c < nchannels; c++)
						if(new_pixel_value[c] > T) {
							is_boundary = true;
							break;
						}
					if(is_boundary) {
						is_boundary = false;
						for (int c = 0; c < nchannels; c++)
							out.setData(i, j, c, white);
					}
					else
						for (int c = 0; c < nchannels; c++)
							out.setData(i, j, c, black);
					
					for(int c = 0; c < nchannels; c++)
						new_pixel_value[c] = 0.0;
				}
		}
		finally {
			out.endUpdate();
		}
		
		return out;
	}
	
	
	/**
     * Average the channel values of a given multi-channel image.
     * @param in
     * @return
     */
	public static IcyBufferedImage averageChannels ( IcyBufferedImage in ) {
		int nx = in.getSizeX();
		int ny = in.getSizeY();
		int nchannels = in.getSizeC();
		double new_value = 0.0;
		IcyBufferedImage out = new IcyBufferedImage(nx, ny, 1, in.getDataType_());
		
		out.beginUpdate();
		try {
			for(int i = 0; i < nx; i++)
				for(int j = 0; j < ny; j++) {
					for(int c = 0; c < nchannels; c++)
						new_value += in.getData(i, j, c);
					new_value = new_value / nchannels;
					out.setData(i, j, 0, new_value);
					new_value = 0.0;
				}
		}
		finally {
			out.endUpdate();
		}
		
		return out;
	}
	
}