// Copyright 2016 INSERM 
// Written by Volker Baecker at Montpellier RIO Imaging 08.07.2016
//
// volker.baecker@mri.cnrs.fr
//
// This program is an icy-plugin that allows to run scripts from
// the command line while defining input parameters in an xml-file.
// It needs java version 1.8 or higher to work.
//
//  This program is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.

package plugins.volker.commandlinescriptrunner;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.List;
import javax.script.ScriptEngine;
import javax.script.ScriptException;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;
import icy.plugin.abstract_.PluginActionable;
import jdk.nashorn.api.scripting.NashornScriptEngineFactory;

/**
 * The CommandLineScriptRunner will run a javascript file from the
 * command line. It expects a job description file with the name job.xml
 * in the home folder of icy. The xml file gives the path to the script 
 * and the parameter values for running the script. The script must contain
 * a section that begins with a line
   // PARAMETER START
 * and ends with the line
   // PARAMETER END
 * The parameter values will be defined as variables in this section (the 
 * section can contain the definition of the variables in case the script
 * is not run from this plugin.
 * 
 *  Here is an example of the job.xml file:
 *  
    <job title='Gaussian Blur Filter' type='script'>
		<path>./scripts/gaussian_blur.js</path>
		<parameter-list>
			<parameter name="sigmaX">1.6</parameter>
			<parameter name="sigmaY">1.6</parameter>
			<parameter name="sigmaZ">1.6</parameter>
			<parameter name="inputFolder">"/media/in/"</parameter>
			<parameter name="outputFolder">"/media/out/"</parameter>
		</parameter-list>
	</job>
	
 * And here is the corresponding example script:
 * 
	// PARAMETER START
	var sigmaX = 1.6;
	var sigmaY = 1.6;
	var sigmaZ = 1.6;
	var inputFolder = "/media/in/";
	var outputFolder = "/media//out/";
	// PARAMETER END
	
	var ext = ".tif";
	var inputDir = new java.io.File(inputFolder); 
	var it = java.util.Arrays.asList(inputDir.listFiles()).iterator();
	while(it.hasNext()) {
		var file = it.next();
		if (!file.isDirectory() && file.getName().endsWith(ext)) {
			var sequence = Packages.icy.file.Loader.loadSequence(new java.io.File(file));
			var result = Packages.plugins.adufour.filtering.GaussianFilter.filter(sequence, sigmaX, sigmaY, sigmaZ);
			Packages.icy.file.Saver.save(result, new java.io.File(outputFolder + file.getName()), false);
			sequence.close();
			result.close();
		}
	}	
 */
public class CommandLineScriptRunner extends PluginActionable {
	public static final String fileName = "job.xml";
	private String type;
	private String title;
	private String path;
	private String[] parameterNames;
	private String[] parameterValues;
	private String script;

	@Override
	public void run() {
		this.readJobDescription();
		try {
			this.readAndModifyScript();
			this.runScript();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Read the script and replace the parameter section with the values defined
	 * in the file job.xml. Add a part at the beginning of the script that loads 
	 * nashorn:mozilla_compat.js so that Nashorn understands the importClass directive.
	 */
	private void readAndModifyScript() throws IOException {
		List<String> lines = Files.readAllLines((Paths.get(this.path)));
		this.script = "";
		boolean isInParameterSection = false;
		for (String line : lines) {
			if (line.contains("PARAMETER START")) {
				isInParameterSection = true;
				continue;
			}
			if (line.contains("PARAMETER END")) { 
				isInParameterSection = false;
				continue;
			}
			if (isInParameterSection) continue;
			script = script + line + "\n";
		}
		String parameterSection = "// PARAMETER START\n";
		int index = 0;
		for (String parameterName : this.parameterNames) {
			parameterSection += "var " + parameterName + " = " + this.parameterValues[index] + ";\n";
			index++;
		}
		parameterSection += "// PARAMETER END\n";
		String nashornSection = "try {" + "\n" +
				    "load(\"nashorn:mozilla_compat.js\");" + "\n" + 
				"} catch (e) {" + "\n" +
				"   // ignore the exception - perhaps we are running on Rhino!" + "\n" +
				"}" + "\n";
		this.script = nashornSection + parameterSection + this.script;
	}

	/**
	 * Execute the script. It only works with the Nashorn js-engine. I don't know how
	 * to set the class-loader for rhino or in the general case, so that the icy-plugins will be found.
	 */
	private void runScript() {
		System.out.println("executing "+this.type+": " + this.title);
		NashornScriptEngineFactory fact = new NashornScriptEngineFactory();
		ScriptEngine jsEngine = fact.getScriptEngine(this.getClass().getClassLoader());
		if (jsEngine == null) {
			System.err.println("Unable to use ScriptEngine with JavaScript.");
			return;
		}
		try {
			jsEngine.eval(this.script);
			System.out.println("FINISHED "+this.type+": " + this.title);
		} catch (ScriptException ex) {
			System.err.println("Line : " + ex.getLineNumber() + "\n");
			System.err.println(ex.getMessage());
		}
	}

	/**
	 * Read the path, the type, the title and the parameters (names and values)
	 * from the file job.xml
	 */
	private void readJobDescription() {
		File jobFile = new File(fileName);
		DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance();
		DocumentBuilder dBuilder;
		try {
			dBuilder = dbFactory.newDocumentBuilder();
			Document doc = dBuilder.parse(jobFile);
			doc.getDocumentElement().normalize();
			this.type = doc.getDocumentElement().getAttribute("type");
			this.title = doc.getDocumentElement().getAttribute("title");
			this.path = doc.getDocumentElement().getElementsByTagName("path").item(0).getTextContent();
			NodeList elementsByTagName = doc.getDocumentElement().getElementsByTagName("parameter");
			int nrOfParameters = elementsByTagName.getLength();
			this.parameterNames = new String[nrOfParameters];
			this.parameterValues = new String[nrOfParameters];
			for (int i = 0; i < elementsByTagName.getLength(); i++) {
				Element item = (Element) elementsByTagName.item(i);
				parameterNames[i] = item.getAttribute("name");
				parameterValues[i] = item.getTextContent();
			}
		} catch (ParserConfigurationException e) {
			e.printStackTrace();
		} catch (SAXException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
}
