package plugins.adufour.roi;

import java.awt.Point;

import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

import org.apache.poi.ss.usermodel.Workbook;

import icy.math.MathUtil;
import icy.roi.ROI;
import icy.roi.ROI2D;
import icy.roi.ROI3D;
import icy.type.point.Point3D;
import icy.type.point.Point5D;
import plugins.adufour.blocks.tools.roi.ROIBlock;
import plugins.adufour.blocks.util.VarList;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzVarBoolean;
import plugins.adufour.ezplug.EzVarSequence;
import plugins.adufour.vars.lang.VarROIArray;
import plugins.adufour.vars.lang.VarWorkbook;
import plugins.adufour.workbooks.IcySpreadSheet;
import plugins.adufour.workbooks.Workbooks;
import plugins.kernel.roi.descriptor.measure.ROIMassCenterDescriptorsPlugin;

/**
 * This plug-in takes 2 sets of ROI, and calculates spatial inclusion and distances from one set
 * w.r.t. to the other
 * 
 * @author Alexandre Dufour
 */
public class ROIInclusionAnalysis extends EzPlug implements ROIBlock
{
    // EzGUI
    
    EzVarSequence seqOuterROI = new EzVarSequence("Take outer ROI from");
    EzVarSequence seqInnerROI = new EzVarSequence("Take inner ROI from");
    EzVarBoolean includeOverlap = new EzVarBoolean("Include overlapping ROI", true);
    
    // Headless
    
    VarROIArray outerROI = new VarROIArray("Outer (enclosing) ROI");
    VarROIArray innerROI = new VarROIArray("Inner (enclosed) ROI");
    
    // Output
    
    VarWorkbook workbook = new VarWorkbook("Analysis", (Workbook) null);
    
    @Override
    public void declareInput(VarList inputMap)
    {
        inputMap.add("Outer ROI", outerROI);
        inputMap.add("Inner ROI", innerROI);
        inputMap.add("Include overlap", includeOverlap.getVariable());
    }
    
    @Override
    public void declareOutput(VarList outputMap)
    {
        outputMap.add("workbook", workbook);
    }
    
    @Override
    public void clean()
    {
    }
    
    @Override
    protected void initialize()
    {
        addEzComponent(seqOuterROI);
        addEzComponent(seqInnerROI);
        addEzComponent(includeOverlap);
        getUI().setParametersIOVisible(false);
    }
    
    @Override
    public void execute()
    {
        if (!isHeadLess())
        {
            // EzGUI: take ROI from the specified sequences
            outerROI.setValue(seqOuterROI.getValue(true).getROIs().toArray(new ROI[0]));
            innerROI.setValue(seqInnerROI.getValue(true).getROIs().toArray(new ROI[0]));
        }
        
        Workbook wb = Workbooks.createEmptyWorkbook();
        int row = 0, summaryRow = 0;
        
        // Create the sheets
        IcySpreadSheet summarySheet = Workbooks.getSheet(wb, "Summary");
        IcySpreadSheet detailSheet = Workbooks.getSheet(wb, "Details");
        
        // Write headers
        detailSheet.setRow(0, "Enclosing ROI", "X", "Y", "Z", "Enclosed ROI", "X", "Y", "Z", "Oriented dist. to edge", "Oriented dist. to edge (%)", "Absolute dist. to edge");
        summarySheet.setRow(0, "Enclosing ROI", "X", "Y", "Z", "nb. enclosed ROI", "Mean oriented dist. to edge", "Mean oriented dist. to edge (%)", "Mean absolute dist. to edge");
        
        for (ROI enclosing : outerROI)
        {
            summaryRow++;
            getStatus().setCompletion((double) summaryRow / outerROI.size());
            
            Point5D cOut = ROIMassCenterDescriptorsPlugin.computeMassCenter(enclosing);
            double outX = MathUtil.round(cOut.getX(), 2);
            double outY = MathUtil.round(cOut.getY(), 2);
            double outZ = MathUtil.round(cOut.getZ(), 2);
            String outName = enclosing.getName();
            
            int nbInside = 0;
            double avgOD2E = 0, avgOD2E_pct = 0, avgAD2E = 0;
            for (ROI enclosed : innerROI)
            {
                double[] distances = inclusionAnalysis(enclosing, enclosed, true);
                
                if (!Double.isNaN(distances[0]))
                {
                    row++;
                    nbInside++;
                    
                    Point5D cIn = ROIMassCenterDescriptorsPlugin.computeMassCenter(enclosed);
                    double inX = MathUtil.round(cIn.getX(), 2);
                    double inY = MathUtil.round(cIn.getY(), 2);
                    double inZ = MathUtil.round(cIn.getZ(), 2);
                    String inName = enclosed.getName();
                    
                    double oD2E = distances[0];
                    double oD2E_pct = distances[1];
                    double absD2E = distances[2];
                    
                    avgOD2E += oD2E;
                    avgOD2E_pct += oD2E_pct;
                    avgAD2E += absD2E;
                    
                    oD2E = MathUtil.round(oD2E, 2);
                    oD2E_pct = MathUtil.round(oD2E_pct, 2);
                    absD2E = MathUtil.round(absD2E, 2);
                    
                    detailSheet.setRow(row, outName, outX, outY, outZ, inName, inX, inY, inZ, oD2E, oD2E_pct, absD2E);
                }
            }
            
            avgOD2E = MathUtil.round(avgOD2E / nbInside, 2);
            avgOD2E_pct = MathUtil.round(avgOD2E_pct / nbInside, 2);
            avgAD2E = MathUtil.round(avgAD2E / nbInside, 2);
            
            summarySheet.setRow(summaryRow, outName, outX, outY, outZ, nbInside, avgOD2E, avgOD2E_pct, avgAD2E);
        }
        
        workbook.setValue(wb);
        
        if (!isHeadLess())
        {
            // EzGUI: display the workbook
            Workbooks.show(wb, "Inclusion analysis");
        }
    }
    
    /**
     * This method performs the inclusion analysis of a so-called "inner" ROI with respect to a
     * so-called "outer" ROI. It is first determined whether the specified "inner" ROI is contained
     * in (or overlaps with) the specified "outer" ROI. If so, three distance values are measured:
     * <ul>
     * <li>An "oriented" distance (in pixels) from the center of the "inner" ROI to the edge of the
     * "outer" ROI, calculated along a ray going from the center of the "outer" ROI through the
     * center of the "inner" ROI</li>
     * <li>The equivalent, relative "oriented" distance from center (0) to edge (1).</li>
     * <li>The "non-oriented" (closest) distance from the center of the "inner" ROI.</li>
     * </ul>
     * 
     * @param outerROI
     *            the outer (i.e. enclosing) ROI
     * @param innerROI
     *            the inner (i.e. enclosed) ROI
     * @param allowPartialOverlap
     *            set to <code>true</code> if partially overlapping ROI should be considered as
     *            "inside", or <code>false</code> if they should be discarded
     * @return A 3-element array containing the measured distances. The first element is the
     *         oriented distance (in pixels) from the center of the "inner" ROI to the edge of the
     *         "outer" ROI (or <code>NaN</code> if the "inner" ROI is not related to the "outer"
     *         ROI). The second element is the relative distance from center to edge, i.e.:
     *         <ul>
     *         <li>0 if the enclosed ROI is at the center of the enclosing ROI</li>
     *         <li>1 if the enclosed ROI is at the edge of the enclosing ROI</li>
     *         <li>NaN if the enclosed ROI is not inside the enclosing ROI</li>
     *         </ul>
     */
    public static double[] inclusionAnalysis(ROI outerROI, ROI innerROI, boolean allowPartialOverlap)
    {
        boolean enclosure = allowPartialOverlap ? outerROI.intersects(innerROI) : outerROI.contains(innerROI);
        
        if (!enclosure) return new double[] { Double.NaN, Double.NaN, Double.NaN };
        
        Point5D p5;
        
        // get the center of the enclosing ROI
        p5 = ROIMassCenterDescriptorsPlugin.computeMassCenter(outerROI);
        Point3d outCenter = new Point3d(p5.getX(), p5.getY(), p5.getZ());
        
        // get the center of the enclosed ROI
        p5 = ROIMassCenterDescriptorsPlugin.computeMassCenter(innerROI);
        Point3d inCenter = new Point3d(p5.getX(), p5.getY(), p5.getZ());
        
        // compute the "oriented" inclusion distance (i.e. center to center)
        double distToCenter = outCenter.distance(inCenter);
        double absoluteDistToEdge = Double.MAX_VALUE;
        
        // compute (locally) the radius of the enclosing ROI
        // do this lazily by stepping away from the center
        double step = 0.1;
        Point3d p = new Point3d(inCenter);
        Vector3d ray = new Vector3d();
        ray.sub(inCenter, outCenter);
        ray.normalize();
        ray.scale(step);
        
        // step away until we exit the ROI
        // NB: we might already be outside if the enclosed ROI is only overlapping
        if (outerROI instanceof ROI2D)
        {
            // incrementally compute the oriented distance
            while (((ROI2D) outerROI).contains(p.x, p.y))
                p.add(ray);
            
            // Compute the raw distance to edge
            for (Point edgePt : ((ROI2D) outerROI).getBooleanMask(true).getContourPoints())
            {
                double rawDist = edgePt.distance(inCenter.x, inCenter.y);
                if (rawDist < absoluteDistToEdge) absoluteDistToEdge = rawDist;
            }
        }
        else if (outerROI instanceof ROI3D)
        {
            // incrementally compute the oriented distance
            while (((ROI3D) outerROI).contains(p.x, p.y, p.z))
                p.add(ray);
            
            // Compute the raw distance to edge
            for (Point3D.Integer edgePt : ((ROI3D) outerROI).getBooleanMask(true).getContourPoints())
            {
                double rawDist = inCenter.distance(new Point3d(edgePt.x, edgePt.y, edgePt.z));
                if (rawDist < absoluteDistToEdge) absoluteDistToEdge = rawDist;
            }
        }
        else
        {
            // incrementally compute the oriented distance
            while (outerROI.contains(p.x, p.y, p.z, -1, -1))
                p.add(ray);
        }
        
        // the distance from p to the center is the "local" radius
        double localRadius = p.distance(outCenter);
        double orientedDistToEdge = localRadius - distToCenter;
        
        return new double[] { orientedDistToEdge, distToCenter / localRadius, absoluteDistToEdge };
    }
    
    // /**
    // * @param roiA
    // * @param roiB
    // * @return center-to-edge distance
    // */
    // static double c2e(ROI roiA, ROI roiB)
    // {
    // Rectangle5D boundsA = roiA.getBounds5D();
    // Point3d centerA = new Point3d(boundsA.getCenterX(), boundsA.getCenterY(),
    // boundsA.getCenterZ());
    //
    // double minDist = Double.MAX_VALUE;
    //
    // if (roiB instanceof ROI2D)
    // {
    // for (Point pt : ((ROI2D) roiB).getBooleanMask(true).getContourPoints())
    // {
    // double dist = centerA.distance(new Point3d(pt.x, pt.y, ((ROI2D) roiB).getZ()));
    // if (dist < minDist) minDist = dist;
    // }
    // return minDist;
    // }
    // else if (roiB instanceof ROI3D)
    // {
    // for (Point3D.Integer pt : ((ROI3D) roiB).getBooleanMask(true).getContourPoints())
    // {
    // double dist = centerA.distance(new Point3d(pt.x, pt.y, pt.z));
    // if (dist < minDist) minDist = dist;
    // }
    // return minDist;
    // }
    // else throw new UnsupportedOperationException("Cannot process a ROI of type " +
    // roiB.getClassName());
    // }
    //
    // /**
    // * @param roiA
    // * @param roiB
    // * @return edge-to-edge distance
    // */
    // static double e2e(ROI roiA, ROI roiB)
    // {
    // double minDist = Double.MAX_VALUE;
    //
    // if (roiA instanceof ROI2D)
    // {
    // for (Point ptA : ((ROI2D) roiA).getBooleanMask(true).getContourPoints())
    // {
    // if (roiB instanceof ROI2D)
    // {
    // for (Point ptB : ((ROI2D) roiB).getBooleanMask(true).getContourPoints())
    // {
    // double dist = ptA.distance(ptB);
    // if (dist < minDist) minDist = dist;
    // }
    // }
    // else if (roiB instanceof ROI3D)
    // {
    // Point3d ptA_3d = new Point3d(ptA.x, ptA.y, ((ROI2D) roiA).getZ());
    // for (Point3D.Integer ptB : ((ROI3D) roiB).getBooleanMask(true).getContourPoints())
    // {
    // double dist = ptA_3d.distance(new Point3d(ptB.x, ptB.y, ptB.z));
    // if (dist < minDist) minDist = dist;
    // }
    // }
    // }
    // return minDist;
    // }
    // else if (roiA instanceof ROI3D)
    // {
    // for (Point3D.Integer ptA : ((ROI3D) roiA).getBooleanMask(true).getContourPoints())
    // {
    // Point3d ptA_3d = new Point3d(ptA.x, ptA.y, ((ROI2D) roiA).getZ());
    //
    // for (Point3D.Integer ptB : ((ROI3D) roiB).getBooleanMask(true).getContourPoints())
    // {
    // double dist = ptA_3d.distance(new Point3d(ptB.x, ptB.y, ptB.z));
    // if (dist < minDist) minDist = dist;
    // }
    // }
    // return minDist;
    // }
    // else throw new UnsupportedOperationException("Cannot process a ROI of type " +
    // roiB.getClassName());
    // }
}
