package plugins.dmandache.denoise;

import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzVarDouble;
import plugins.adufour.ezplug.EzVarSequence;
import plugins.adufour.ezplug.EzVarText;
import plugins.adufour.ezplug.EzVarBoolean;
import plugins.adufour.vars.lang.VarSequence;
import plugins.ylemontag.gaussiannoiseestimator.GaussianNoiseEstimator;
import icy.image.IcyBufferedImage;
import icy.sequence.Sequence;
import icy.type.collection.array.Array1DUtil;
import java.util.concurrent.atomic.AtomicInteger;

public class Denoise extends EzPlug {
	
	/**
	 * ICY plugin for Compressed Sensing based denoising using TV regularizations
	 *  
	 * It is inspired by the method presented in the paper:
	 * W. Meiniel, E. Angelini and J.-C. Olivo-Marin
	 * "Image denoising by adaptive compressed sensing reconstructions and fusions", in Proc. SPIE 9597,September 2015.
	 * URL: http://biblio.telecom-paristech.fr/cgi-bin/download.cgi?id=15927
	 * 
	 * @author Diana Mandache
	 * @version 1.0
	 * @date 2017-02-20
	 * @license gpl v3.0
	 */
		
	String[] samplingMethodStr = new String[] { "Cutoff Frequency", "Gaussian", "Random"};
	
	EzVarSequence inputVar = new EzVarSequence("Input image");
	EzVarDouble samplingRateVar = new EzVarDouble( "Sampling rate (%)", 30 , 10 , 100 , 5);
	EzVarText samplingMethodVar = new EzVarText("Sampling Method", samplingMethodStr, 0, false );
	EzVarDouble cutoffFreqVar = new EzVarDouble( "Cut-off frequency", 0.3 , 0.1 , 1.0 , 0.1);
	EzVarBoolean varBoolean = new EzVarBoolean("Auto detect noise level", true);
	EzVarDouble noiseLevel = new EzVarDouble( "Noise level", 5 , 1 , 100 , 1);
	
	VarSequence outputVar = new VarSequence("Denoised image", null);

	
	@Override
	protected void initialize() {
		super.addEzComponent(inputVar);
		
		samplingRateVar.setToolTipText("Percent of coefficients sampled in the Fourier space");
		super.addEzComponent(samplingRateVar);
		
		samplingMethodVar.setToolTipText("Sampling methods in Fourier space");
		super.addEzComponent(samplingMethodVar);
		samplingMethodVar.addVisibilityTriggerTo(cutoffFreqVar, samplingMethodStr[0]);
		
		cutoffFreqVar.setToolTipText("Limit of fully sampled low-frequency coefficients");
		super.addEzComponent(cutoffFreqVar);
		
		varBoolean.setToolTipText("Choose whether to input or generate the level of noise");
		super.addEzComponent(varBoolean);
		varBoolean.addVisibilityTriggerTo(noiseLevel, false);
		
		noiseLevel.setToolTipText("Standard deviation of the Gaussian noise present in the image");
		super.addEzComponent(noiseLevel);
		
		super.setTimeDisplay(true);
	}

	@Override
	protected void execute() {
		
		Sequence inputSeq = inputVar.getValue();
		final double samplingRate = samplingRateVar.getValue() / 100.0;
		final double cutOffFreq = cutoffFreqVar.getValue();
		String samplingMethod = samplingMethodVar.getValue();
		Sequence outputSeq = new Sequence();
		double lambdaTemp;
		
		// Estimate level of Gaussian noise
		if (varBoolean.getValue() == false)
			lambdaTemp = noiseLevel.getValue();
		else{
			GaussianNoiseEstimator.Result out = GaussianNoiseEstimator.computeStandardDeviation(inputSeq);
			 lambdaTemp = out.get(0,0,0);
			if (lambdaTemp < 1 && lambdaTemp > 0.01)
				lambdaTemp *= 100;
			lambdaTemp = Math.round(lambdaTemp);
			lambdaTemp = Math.max(lambdaTemp, 1);
		}
		final double lambda = lambdaTemp;
		System.out.print("Level of noise =  " + lambda + "\n");

		// Compute number of reconstructions R = 100 / samplingRate
		final int numRec = (int) (100 / samplingRateVar.getValue());
		System.out.print("Num recs =  " + numRec + "\n");
		
		final int w = inputSeq.getSizeX();
		final int h = inputSeq.getSizeY();
		final int numChanels = inputSeq.getSizeC();
		
		final double[][] xhat= new double[numChanels][w*h];		// Denoised prediction
		
		// Denoise every channel
		for(int c=0; c<numChanels; c++){
			
			final double[] y_ch = new double[w*h];				// Channel c of the noisy image 
		
			Array1DUtil.arrayToDoubleArray(inputSeq.getDataXY(0, 0, c), 0, y_ch, 0, w*h, inputSeq.isSignedDataType());
	
			// Fourier masks
			final double[][] masks = FFTwrapper.multipleFFTMasks( w, h, samplingRate, numRec, samplingMethod, cutOffFreq, false);
			
			// R acquisitions in Fourier domain (yi) and R reconstructions (xi)
			final double[][][] yi = new double[numRec][2][w*h];
			final double[][] xi = new double[numRec][w*h];
	
			// Multi threaded approach for Fourier Sampling and Reconstruction
		    Thread [] theThreads = new Thread[numRec];
		    final AtomicInteger g = new AtomicInteger();
		    
		    Util.multiplyScalar1D(y_ch, 255);
		    // Initialize threads
		    for (int m = 0; m < numRec; m++) {
		      theThreads[m] = new Thread(new Runnable() {
		          public void run() { 
		        	  int local = g.incrementAndGet() - 1;
		    		  yi[local] = FFTwrapper.FFT_2D_with_mask(y_ch, masks[local], w, h);
		    		  xi[local] = FISTAwrapper.optimization(yi[local], masks[local], w, h, 80, lambda);
		          }
		      });
		    }
		    
		    // Start threads
		    for (int m = 0; m < numRec; m++) {
		    	theThreads[m].start();
		    }
	
		    // Wait for all threads to finish
		    for (int m = 0; m < numRec; m++) {
		      try {
				theThreads[m].join();
			} catch (InterruptedException e) {
				e.printStackTrace();
				}
		    }
		    
		    // Fusion reconstructions
			xhat[c] = Fusion.fusion(xi, y_ch, numRec, w, h);
		}

		// Display result
		outputSeq.setName("Denoised Image");
		outputSeq.setImage(0, 0, new IcyBufferedImage(w, h, xhat));		
		if (!isHeadLess()) {
			addSequence(outputSeq);
		}		
		outputVar.setValue(outputSeq);
	}

	@Override
	public void clean() {
		// TODO Auto-generated by Icy4Eclipse
	}
}

