package plugins.nchenouard.trackprocessorchallengetracksimporter;


import icy.gui.frame.IcyFrame;
import icy.gui.frame.progress.FailedAnnounceFrame;
import icy.main.Icy;
import icy.swimmingPool.SwimmingObject;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.io.File;
import java.util.ArrayList;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.filechooser.FileFilter;

import plugins.fab.trackmanager.PluginTrackManagerProcessor;
import plugins.fab.trackmanager.TrackGroup;
import plugins.fab.trackmanager.TrackSegment;

/**
 * Main class for the importer/exporter of tracks.
 *  Input and output files are in the .xml format used for the ISBI'2012 Particle Tracking Challenge
 *  Tracks are of the type TrackSegment that is used by the TrackManager plugin
 *  
 * @version February 3, 2012
 * 
 * @author Nicolas Chenouard
 * @author Fabrice de Chaumont
 * @author Ihor Smal
 */

public class TrackProcessorChallengeTracksImporter extends PluginTrackManagerProcessor
{

	public TrackProcessorChallengeTracksImporter() {
		setName("Particle tracking challenge importer/exporter");
		this.panel.setLayout(new GridLayout(1, 2));
		JButton loadTracksButton = new JButton("Load Tracks");
		this.panel.add(loadTracksButton);
		JButton saveTracksButton = new JButton("Save Tracks");
		this.panel.add(saveTracksButton);

		loadTracksButton.addActionListener(new ActionListener() {			
			@Override
			public void actionPerformed(ActionEvent arg0) {
				loadTracksFromFile();
			}
		});

		saveTracksButton.addActionListener(new ActionListener() {			
			@Override
			public void actionPerformed(ActionEvent arg0) {
				saveTracksToFile();
			}
		});
	}

	private void loadTracksFromFile() {
		JFileChooser fileChooser = new JFileChooser();
		fileChooser.setFileFilter(new FileFilter() {			
			@Override
			public String getDescription() {
				return "XML file filter";
			}
			@Override
			public boolean accept(File file) {
				if (file.isDirectory())
					return true;
				return file.getName().endsWith(".xml");
			}
		});
		fileChooser.setMultiSelectionEnabled(false);
		fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		fileChooser.setName("Input xml file selection");
		int returnVal = fileChooser.showOpenDialog(panel);
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			File file = fileChooser.getSelectedFile();
			// now ask for a trackGroup name
			TrackLoaderFrame loaderFrame = new TrackLoaderFrame(file);
			addIcyFrame(loaderFrame);
			loaderFrame.setVisible(true);
		}
	}

	private void saveTracksToFile()
	{
		JFileChooser fileChooser = new JFileChooser();
		fileChooser.setFileFilter(new FileFilter() {			
			@Override
			public String getDescription() {
				return "XML file filter";
			}
			@Override
			public boolean accept(File file) {
				if (file.isDirectory())
					return true;
				return file.getName().endsWith(".xml");
			}
		});
		fileChooser.setMultiSelectionEnabled(false);
		fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		fileChooser.setName("Input xml file selection");
		int returnVal = fileChooser.showOpenDialog(panel);
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			File file = fileChooser.getSelectedFile();
			// now ask for a trackGroup name
			TrackSaverFrame saverFrame = new TrackSaverFrame(file);
			addIcyFrame(saverFrame);
			saverFrame.setVisible(true);
		}
	}

	@Override
	public void Close() {}

	@Override
	public void Compute() {}

	@Override
	public void displaySequenceChanged() {}

	private class TrackLoaderFrame extends IcyFrame
	{
		final File file;
		String defaultTrackGroupName = "importedTracks_";
		JTextField trackGroupNameTF = new JTextField();
		JLabel nameIsUsedLabel = new JLabel();
		JCheckBox addToExistingGroupBox = new JCheckBox("Add tracks to the existing group");

		public TrackLoaderFrame(File file) 
		{
			this.file = file;
			JPanel mainPane = new JPanel();
			mainPane.setLayout(new GridBagLayout());

			GridBagConstraints c = new GridBagConstraints();
			c.fill = GridBagConstraints.HORIZONTAL;
			c.weightx = 1.0;
			c.weighty = 0;
			c.gridwidth = 2;

			c.gridx = 0;
			c.gridy = 0;
			mainPane.add(new JLabel("Name of the group of tracks to import:"), c);

			String defaultName = getCompatibleName();
			trackGroupNameTF.setText(defaultName);
			trackGroupNameTF.addKeyListener(new KeyListener() {
				@Override
				public void keyTyped(KeyEvent arg0) {
				}
				@Override
				public void keyReleased(KeyEvent arg0) {
					refreshNameIsUsedLabel();
				}
				@Override
				public void keyPressed(KeyEvent arg0) {
				}
			});
			c.gridx = 0;
			c.gridy = 1;
			mainPane.add(trackGroupNameTF, c);

			c.gridx = 0;
			c.gridy = 2;
			nameIsUsedLabel.setText("Warning: this name is already used");
			mainPane.add(nameIsUsedLabel, c);

			c.gridx = 0;
			c.gridy = 3;
			mainPane.add(addToExistingGroupBox, c);

			c.gridx = 0;
			c.gridy = 4;
			c.gridwidth = 1;
			JButton cancelButton = new JButton("Cancel");
			cancelButton.addActionListener(new ActionListener() {				
				@Override
				public void actionPerformed(ActionEvent arg0) {
					TrackLoaderFrame.this.close();
				}
			});	
			mainPane.add(cancelButton, c);


			c.gridx = 1;
			c.gridy = 4;
			JButton loadButton = new JButton("Load");
			loadButton.addActionListener(new ActionListener() {
				@Override
				public void actionPerformed(ActionEvent arg0)
				{
					loadTracks();
				}
			});
			mainPane.add(loadButton, c);


			refreshNameIsUsedLabel();
			this.setContentPane(mainPane);
			this.pack();
		}

		private void refreshNameIsUsedLabel()
		{
			boolean compatible = true;
			String name = trackGroupNameTF.getText();
			ArrayList<TrackGroup> tgList  = trackPool.getTrackGroupList();
			for (TrackGroup tg:tgList)
			{
				if (tg.getDescription()!=null && tg.getDescription().equals(name))
				{
					compatible = false;
					break;
				}
			}
			if (compatible)
			{
				nameIsUsedLabel.setEnabled(false);
				addToExistingGroupBox.setEnabled(false);
			}
			else
			{
				nameIsUsedLabel.setEnabled(true);
				addToExistingGroupBox.setEnabled(true);
			}
		}
		/**
		 * @return a description of TrackGroup that is not already used
		 * */
		private String getCompatibleName()
		{
			ArrayList<TrackGroup> tgList  = trackPool.getTrackGroupList();
			int i = 1;
			String compatibleName = defaultTrackGroupName + Integer.toString(i);
			boolean compatible = false;
			while (!compatible)
			{
				compatibleName = defaultTrackGroupName + Integer.toString(i);
				compatible = true;
				for (TrackGroup tg:tgList)
					if (tg.getDescription()!=null && tg.getDescription().equals(compatibleName))
					{
						compatible = false;
						break;
					}
				i++;
			}
			return compatibleName;
		}

		private void loadTracks()
		{
			ArrayList<TrackSegment> tracks = new ArrayList<TrackSegment>();
			try
			{
				tracks.addAll(TrackExportAndImportUtilities.importTracksFile(file));
			}
			catch (Exception e) {
				e.printStackTrace();
				new FailedAnnounceFrame("Unable to load the track file");
				return;
			}
			boolean compatible = true;
			TrackGroup tgSelected = null;
			String name = trackGroupNameTF.getText();
			ArrayList<TrackGroup> tgList  = trackPool.getTrackGroupList();
			for (TrackGroup tg:tgList)
			{
				if (tg.getDescription()!=null && tg.getDescription().equals(name))
				{
					tgSelected = tg;
					compatible = false;
					break;
				}
			}
			if (compatible || !addToExistingGroupBox.isSelected())
			{
				final TrackGroup newGroup = new TrackGroup(null);
				newGroup.setDescription(name);
				for (TrackSegment ts:tracks)
					newGroup.addTrackSegment(ts);
				SwingUtilities.invokeLater(new Runnable()
				{
					public void run() {
						// Add the given trackGroup
						SwimmingObject result = new SwimmingObject(newGroup);// should
						Icy.getMainInterface().getSwimmingPool().add(result);
					}
				});
			}
			else
				for (TrackSegment ts:tracks)
					tgSelected.addTrackSegment(ts);
			this.close();
		}
	}


	private class TrackSaverFrame extends IcyFrame
	{
		File file;
		JComboBox trackGroupsBox;
		final ArrayList<TrackGroup> groupList = new ArrayList<TrackGroup>();

		public TrackSaverFrame(File file)
		{
			this.file = file;

			JPanel mainPane = new JPanel();
			mainPane.setLayout(new GridBagLayout());

			GridBagConstraints c = new GridBagConstraints();
			c.fill = GridBagConstraints.HORIZONTAL;
			c.weightx = 1.0;
			c.weighty = 0;
			c.gridwidth = 2;

			c.gridx = 0;
			c.gridy = 0;
			mainPane.add(new JLabel("Group of tracks to save:"), c);


			ArrayList<TrackGroup> tgList = new ArrayList<TrackGroup>();
			tgList.addAll(trackPool.getTrackGroupList());
			int cntHaveDescription = 0;
			for (TrackGroup tg:tgList)
				if (tg.getDescription()!=null)
					cntHaveDescription++;
			String[] groupNames = new String[cntHaveDescription];
			groupList.clear();
			int i = 0;
			for (TrackGroup tg:tgList)
			{
				if (tg.getDescription()!=null)
				{
					groupList.add(tg);
					groupNames[i] = tg.getDescription();
					i++;
				}
			}
			trackGroupsBox = new JComboBox(groupNames);
			c.gridx = 0;
			c.gridy = 1;
			mainPane.add(trackGroupsBox, c);


			c.gridx = 0;
			c.gridy = 2;
			c.gridwidth = 1;
			JButton cancelButton = new JButton("Cancel");
			cancelButton.addActionListener(new ActionListener() {				
				@Override
				public void actionPerformed(ActionEvent arg0) {
					TrackSaverFrame.this.close();
				}
			});	
			mainPane.add(cancelButton, c);


			c.gridx = 1;
			c.gridy = 2;
			JButton saveButton = new JButton("Save");
			saveButton.addActionListener(new ActionListener() {
				@Override
				public void actionPerformed(ActionEvent arg0)
				{
					saveTracks();
				}
			});
			mainPane.add(saveButton, c);

			this.setContentPane(mainPane);
			this.pack();
		}

		private void saveTracks()
		{
			if (trackGroupsBox.getSelectedIndex()>=0 && !groupList.isEmpty())
			{
				TrackGroup tg = groupList.get(trackGroupsBox.getSelectedIndex());
				try{
					TrackExportAndImportUtilities.exportTracks(file, tg.getTrackSegmentList());
				}
				catch (Exception e)
				{
					new FailedAnnounceFrame("Unable to save the tracks to the file");
					e.printStackTrace();
				}
			}
			this.close();
		}
	}
}