package plugins.tprovoost.scale;

import icy.canvas.IcyCanvas;
import icy.canvas.IcyCanvas2D;
import icy.gui.main.GlobalSequenceListener;
import icy.gui.viewer.Viewer;
import icy.main.Icy;
import icy.math.MathUtil;
import icy.math.UnitUtil;
import icy.math.UnitUtil.UnitPrefix;
import icy.painter.Overlay;
import icy.plugin.abstract_.Plugin;
import icy.plugin.interface_.PluginDaemon;
import icy.preferences.XMLPreferences;
import icy.sequence.Sequence;
import icy.system.IcyHandledException;
import icy.system.thread.ThreadUtil;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.RenderingHints;
import java.awt.geom.Line2D;

import javax.swing.JLabel;
import javax.swing.JPanel;

import plugins.adufour.vars.gui.model.IntegerRangeModel;
import plugins.adufour.vars.gui.swing.SwingVarEditor;
import plugins.adufour.vars.lang.Var;
import plugins.adufour.vars.lang.VarBoolean;
import plugins.adufour.vars.lang.VarColor;
import plugins.adufour.vars.lang.VarEnum;
import plugins.adufour.vars.lang.VarInteger;
import plugins.kernel.canvas.VtkCanvas;

/**
 * Scale bar plug-in for Icy. Displays a scale bar on 2D viewers, with various display adjustment
 * options, such as:
 * <ul>
 * <li>Color</li>
 * <li>Size and unit (manually or automatically)</li>
 * <li>Position (each image corner)</li>
 * <li>Hide/Show text</li>
 * </ul>
 * These options can be managed from the graphical interface (in the "Layers" section of the
 * inspector), or within scripts (see {@link #addScaleBarTo(Sequence)} and
 * {@link #removeScaleBarFrom(Sequence)}
 * 
 * @author Thomas Provoost, Stephane Dallongeville, Alexandre Dufour
 */
public class Scale extends Plugin implements PluginDaemon, GlobalSequenceListener
{
    public enum ScaleBarLocation
    {
        IMAGE_BOTTOM_LEFT("Bottom left of image"),
        VIEWER_BOTTOM_LEFT("Bottom left of viewer"),
        IMAGE_BOTTOM_RIGHT("Bottom right of image"),
        VIEWER_BOTTOM_RIGHT("Bottom right of viewer"),
        IMAGE_TOP_LEFT("Top left of image"),
        VIEWER_TOP_LEFT("Top left of viewer"),
        IMAGE_TOP_RIGHT("Top right of image"),
        VIEWER_TOP_RIGHT("Top right of viewer");
        
        public final String name;
        
        private ScaleBarLocation(String name)
        {
            this.name = name;
        }
        
        public String toString()
        {
            return name;
        }
        
        public boolean isRelativeToViewer()
        {
            return this == VIEWER_BOTTOM_LEFT || this == VIEWER_BOTTOM_RIGHT || this == VIEWER_TOP_LEFT || this == VIEWER_TOP_RIGHT;
        }
        
        public boolean isRelativeToImage()
        {
            return this == IMAGE_BOTTOM_LEFT || this == IMAGE_BOTTOM_RIGHT || this == IMAGE_TOP_LEFT || this == IMAGE_TOP_RIGHT;
        }

    }
    
    private static XMLPreferences preferences = null;
    
    public static class ScaleBarOverlay extends Overlay
    {
        private static final double[]           scaleRoundedFactors = {
                                                                            1.0D,
                                                                            2.0D,
                                                                            3.0D,
                                                                            4.0D,
                                                                            5.0D,
                                                                            6.0D,
                                                                            7.0D,
                                                                            8.0D,
                                                                            9.0D,
                                                                            10.0D,
                                                                            20.0D,
                                                                            30.0D,
                                                                            40.0D,
                                                                            50.0D,
                                                                            60.0D,
                                                                            70.0D,
                                                                            80.0D,
                                                                            90.0D,
                                                                            100.0D,
                                                                            200.0D,
                                                                            300.0D,
                                                                            400.0D,
                                                                            500.0D,
                                                                            600.0D,
                                                                            700.0D,
                                                                            800.0D,
                                                                            900.0D };
        
        /**
         * Scale bar color (default: white)
         */
        private final VarColor                  color               = new VarColor("Color", new Color(preferences.getInt("color", new Color(255, 255, 255, 255).getRGB())))
                                                                    {
                                                                        public void setValue(Color newValue)
                                                                        {
                                                                            if (getValue().equals(newValue)) return;
                                                                            
                                                                            super.setValue(newValue);
                                                                            
                                                                            preferences.putInt("color", newValue.getRGB());
                                                                            
                                                                            painterChanged();
                                                                        }
                                                                    };
        
        /**
         * Scale bar location (default: bottom left)
         */
        private final VarEnum<ScaleBarLocation> location            = new VarEnum<Scale.ScaleBarLocation>("Location", ScaleBarLocation.valueOf(preferences.get("location",
                                                                            ScaleBarLocation.IMAGE_BOTTOM_LEFT.name())))
                                                                    {
                                                                        public void setValue(ScaleBarLocation newValue)
                                                                        {
                                                                            if (getValue().equals(newValue)) return;
                                                                            
                                                                            super.setValue(newValue);
                                                                            
                                                                            preferences.put("location", newValue.name());
                                                                            
                                                                            painterChanged();
                                                                        }
                                                                    };
        
        /**
         * Displays the text over the scale bar
         */
        private final VarBoolean                showText            = new VarBoolean("Display size", preferences.getBoolean("showText", true))
                                                                    {
                                                                        public void setValue(Boolean newValue)
                                                                        {
                                                                            if (getValue().equals(newValue)) return;
                                                                            
                                                                            super.setValue(newValue);
                                                                            
                                                                            preferences.putBoolean("showText", newValue);
                                                                            
                                                                            painterChanged();
                                                                        }
                                                                    };
        
        /**
         * Scale bar size (default: 10 units)
         */
        private final VarInteger                size                = new VarInteger("Size", preferences.getInt("size", 10))
                                                                    {
                                                                        public void setValue(Integer newValue)
                                                                        {
                                                                            if (getValue().equals(newValue)) return;
                                                                            
                                                                            super.setValue(newValue);
                                                                            
                                                                            preferences.putInt("size", newValue);
                                                                            
                                                                            painterChanged();
                                                                        }
                                                                    };
        
        /**
         * Scale bar size (default: 10 units)
         */
        private final VarInteger                thickness           = new VarInteger("Thickness", preferences.getInt("thickness", 5))
                                                                    {
                                                                        public void setValue(Integer newValue)
                                                                        {
                                                                            if (getValue().equals(newValue)) return;
                                                                            
                                                                            super.setValue(newValue);
                                                                            
                                                                            preferences.putInt("thickness", newValue);
                                                                            
                                                                            painterChanged();
                                                                        }
                                                                    };
        
        /**
         * Scale bar size (default: microns)
         */
        private final VarEnum<UnitPrefix>       unit                = new VarEnum<UnitUtil.UnitPrefix>("Unit", UnitPrefix.valueOf(preferences.get("unit", UnitPrefix.MICRO.name())))
                                                                    {
                                                                        public void setValue(UnitPrefix newValue)
                                                                        {
                                                                            if (getValue().equals(newValue)) return;
                                                                            
                                                                            super.setValue(newValue);
                                                                            
                                                                            preferences.put("unit", newValue.name());
                                                                            
                                                                            painterChanged();
                                                                        }
                                                                    };
        
        /**
         * Auto-adjust the size and unit of the scale bar
         */
        private final VarBoolean                autoSize            = new VarBoolean("Auto-adjust size", preferences.getBoolean("autoSize", false))
                                                                    {
                                                                        public void setValue(Boolean newValue)
                                                                        {
                                                                            if (getValue().equals(newValue)) return;
                                                                            
                                                                            super.setValue(newValue);
                                                                            
                                                                            preferences.putBoolean("autoSize", newValue);
                                                                            
                                                                            painterChanged();
                                                                            
                                                                        }
                                                                    };
        
        private JPanel                          optionPanel;
        
        private final Line2D.Double             line                = new Line2D.Double();
        
        public ScaleBarOverlay()
        {
            super("Scale bar", OverlayPriority.TOPMOST);
            
            Integer currentSize = size.getValue();
            size.setDefaultEditorModel(new IntegerRangeModel(10, 1, 999, 1));
            size.setValue(currentSize);
            
            Integer currentThickness = thickness.getValue();
            thickness.setDefaultEditorModel(new IntegerRangeModel(5, 1, 20, 1));
            thickness.setValue(currentThickness);
            
            // do graphical stuff to the graphical thread
            ThreadUtil.invokeLater(new Runnable()
            {
                @Override
                public void run()
                {
                    
                    initOptionPanel();
                }
            });
        }
        
        private void initOptionPanel()
        {
            optionPanel = new JPanel(new GridBagLayout());
            
            for (Var<?> variable : new Var<?>[] { location, color, autoSize, size, thickness, showText })
            {
                GridBagConstraints gbc = new GridBagConstraints();
                
                gbc.insets = new Insets(2, 10, 2, 5);
                gbc.fill = GridBagConstraints.BOTH;
                optionPanel.add(new JLabel(variable.getName()), gbc);
                
                // special case: show unit next to size
                if (variable == size)
                {
                    gbc.weightx = 0.5;
                    
                    SwingVarEditor<?> editor = (SwingVarEditor<?>) variable.createVarEditor(true);
                    optionPanel.add(editor.getEditorComponent(), gbc);
                    
                    gbc.gridwidth = GridBagConstraints.REMAINDER;
                    
                    SwingVarEditor<?> unitEditor = (SwingVarEditor<?>) unit.createVarEditor(true);
                    optionPanel.add(unitEditor.getEditorComponent(), gbc);
                }
                else
                {
                    gbc.weightx = 1;
                    gbc.gridwidth = GridBagConstraints.REMAINDER;
                    SwingVarEditor<?> editor = (SwingVarEditor<?>) variable.createVarEditor(true);
                    optionPanel.add(editor.getEditorComponent(), gbc);
                }
            }
        }
        
        boolean init = false;
        
        @Override
        public void paint(Graphics2D g, Sequence sequence, IcyCanvas canvas)
        {
            if (canvas instanceof VtkCanvas && !init)
            {
                init = true;
                
                // VtkCanvas vtk = (VtkCanvas) canvas;
                
                // vtkScalarBarActor sb = new vtkScalarBarActor();
                
                // vtk.getRenderer().AddActor(sb);
                
                return;
            }
            else if (g == null || !(canvas instanceof IcyCanvas2D)) return;
            
            IcyCanvas2D c2 = (IcyCanvas2D) canvas;
            
            Graphics2D g2 = (Graphics2D) g.create();
            
            g2.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_SPEED);
            
            g2.setColor(color.getValue());
            
            if (autoSize.getValue())
            {
                int sizeW = sequence.getSizeX() / 7;
                
                // don't draw a scale bar larger than the image itself
                if (location.getValue().isRelativeToImage())
                    sizeW = (int) Math.min(sizeW, sequence.getSizeX() * canvas.getScaleX() * 0.9);
                
                double valueReal = (sizeW * sequence.getPixelSizeX() / canvas.getScaleX());
                UnitPrefix bestUnit = UnitUtil.getBestUnit(valueReal * 0.1, UnitUtil.UnitPrefix.MICRO, 1);
                double valueRealBestUnit = UnitUtil.getValueInUnit(valueReal, UnitPrefix.MICRO, bestUnit);
                
                double closestScale = MathUtil.closest(valueRealBestUnit, scaleRoundedFactors);
                size.setValue((int) closestScale);
                unit.setValue(bestUnit);
            }
            
            String text = "" + size.getValue() + unit.getValue() + "m";
            
            double length = UnitUtil.getValueInUnit(size.getValue(), unit.getValue(), UnitPrefix.MICRO) / sequence.getPixelSizeX();
            
            int textYfactor = 0;
            float thickness = this.thickness.getValue().floatValue();
            
            switch (location.getValue())
            {
            case VIEWER_BOTTOM_LEFT:
                g2.transform(c2.getInverseTransform());
                line.x1 = canvas.getCanvasSizeX() * 0.05;
                line.x2 = line.x1 + length * c2.getScaleX();
                line.y1 = line.y2 = canvas.getCanvasSizeY() * 0.95;
                textYfactor = -2;
                break;
            
            case IMAGE_BOTTOM_LEFT:
                line.x1 = sequence.getSizeX() * 0.05;
                line.x2 = line.x1 + length;
                line.y1 = line.y2 = sequence.getSizeY() * 0.95;
                textYfactor = -2;
                thickness = (float) c2.canvasToImageLogDeltaX((int) thickness);
                break;
            
            case VIEWER_BOTTOM_RIGHT:
                g2.transform(c2.getInverseTransform());
                line.x1 = canvas.getCanvasSizeX() * 0.95;
                line.x2 = line.x1 - length * c2.getScaleX();
                line.y1 = line.y2 = canvas.getCanvasSizeY() * 0.95;
                textYfactor = -2;
                break;
            
            case IMAGE_BOTTOM_RIGHT:
                line.x1 = sequence.getSizeX() * 0.95;
                line.x2 = line.x1 - length;
                line.y1 = line.y2 = sequence.getSizeY() * 0.95;
                textYfactor = -2;
                thickness = (float) c2.canvasToImageLogDeltaX((int) thickness);
                break;
            
            case VIEWER_TOP_LEFT:
                g2.transform(c2.getInverseTransform());
                line.x1 = canvas.getCanvasSizeX() * 0.05;
                line.x2 = line.x1 + length * c2.getScaleX();
                line.y1 = line.y2 = canvas.getCanvasSizeY() * 0.05;
                textYfactor = 4;
                break;
            
            case IMAGE_TOP_LEFT:
                line.x1 = sequence.getSizeX() * 0.05;
                line.x2 = line.x1 + length;
                line.y1 = line.y2 = sequence.getSizeY() * 0.05;
                textYfactor = 4;
                thickness = (float) c2.canvasToImageLogDeltaX((int) thickness);
                break;
            
            case VIEWER_TOP_RIGHT:
                g2.transform(c2.getInverseTransform());
                line.x1 = canvas.getCanvasSizeX() * 0.95;
                line.x2 = line.x1 - length * c2.getScaleX();
                line.y1 = line.y2 = canvas.getCanvasSizeY() * 0.05;
                textYfactor = 4;
                break;
            
            case IMAGE_TOP_RIGHT:
                line.x1 = sequence.getSizeX() * 0.95;
                line.x2 = line.x1 - length;
                line.y1 = line.y2 = sequence.getSizeY() * 0.05;
                textYfactor = 4;
                thickness = (float) c2.canvasToImageLogDeltaX((int) thickness);
                break;
            }
            
            g2.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
            g2.draw(line);
            
            if (showText.getValue())
            {
                g2.setFont(g2.getFont().deriveFont(Font.BOLD, thickness * 3));
                int width = g2.getFontMetrics().stringWidth(text);
                g2.drawString(text, (float) (line.getBounds().getCenterX() - (0.5 * width)), (float) (line.getBounds().getCenterY() + thickness * textYfactor));
            }
            
            g2.dispose();
        }
        
        @Override
        public JPanel getOptionsPanel()
        {
            return optionPanel;
        }
        
        /**
         * @return <code>true</code> if the scale bar automatically adjusts its size and unit
         */
        public boolean getAutoSize()
        {
            return autoSize.getValue();
        }
        
        /**
         * @return the color of the scale bar
         */
        public Color getColor()
        {
            return color.getValue();
        }
        
        /**
         * @return the location of the scale bar on the image
         */
        public ScaleBarLocation getLocation()
        {
            return location.getValue();
        }
        
        /**
         * @return the scale bar size (in metric units as defined by the pixel size)
         */
        public double getSize()
        {
            return size.getValue();
        }
        
        /**
         * @return whether the scale bar's size is currently displayed next to it
         */
        public boolean getTextDisplay()
        {
            return showText.getValue();
        }
        
        /**
         * Sets whether the scale should automatically guess its optimal size and unit
         * 
         * @param autoSize
         */
        public void setAutoSize(boolean autoSize)
        {
            this.autoSize.setValue(autoSize);
        }
        
        /**
         * Sets the color of the scale bar
         * 
         * @param color
         *            the new color of the scale bar
         */
        public void setColor(Color color)
        {
            this.color.setValue(color);
        }
        
        /**
         * Sets the location of the scale bar
         * 
         * @param location
         *            the new location of the scale bar
         */
        public void setLocation(ScaleBarLocation location)
        {
            this.location.setValue(location);
        }
        
        /**
         * Sets whether the scale bar's size should appear next to it
         * 
         * @param displayText
         */
        public void setTextDisplay(boolean displayText)
        {
            this.showText.setValue(displayText);
        }
        
        /**
         * Sets the size of the scale bar in metric units, as defined by the pixel size
         * 
         * @param size
         *            the new size of the scale bar
         */
        public void setSize(int size)
        {
            this.size.setValue(size);
        }
        
        /**
         * Sets the scale bar unit
         * 
         * @param unit
         *            the new unit
         */
        public void setUnit(UnitPrefix unit)
        {
            this.unit.setValue(unit);
        }
    }
    
    @Override
    public void init()
    {
        if (preferences == null) preferences = getPreferencesRoot();
        
        Icy.getMainInterface().addGlobalSequenceListener(this);
    }
    
    @Override
    public void run()
    {
        for (Sequence s : getSequences())
            addScaleBarTo(s);
    }
    
    @Override
    public void stop()
    {
        for (Sequence s : getSequences())
            removeScaleBarFrom(s);
        
        Icy.getMainInterface().removeGlobalSequenceListener(this);
    }
    
    @Override
    public void sequenceOpened(Sequence sequence)
    {
        addScaleBarTo(sequence);
    }
    
    @Override
    public void sequenceClosed(Sequence sequence)
    {
        removeScaleBarFrom(sequence);
    }
    
    public static ScaleBarOverlay addScaleBarTo(Sequence sequence)
    {
        if (sequence == null) throw new IcyHandledException("Cannot add the scale bar: no sequence specified");
        
        if (sequence.hasOverlay(ScaleBarOverlay.class)) return (ScaleBarOverlay) sequence.getOverlays(ScaleBarOverlay.class).get(0);
        
        ScaleBarOverlay overlay = new ScaleBarOverlay();
        
        sequence.addOverlay(overlay);
        
        // Try to detect if the sequence is a screenshot (via its name)
        // => hide the overlay by default (but don't remove it)
        String name = sequence.getName().toLowerCase();
        if (name.startsWith("rendering") || name.startsWith("screen shot"))
        {
            for (Viewer viewer : sequence.getViewers())
                viewer.getCanvas().getLayer(overlay).setVisible(false);
        }
        
        return overlay;
    }
    
    public static void removeScaleBarFrom(Sequence sequence)
    {
        if (sequence == null) throw new IcyHandledException("Cannot remove the scale bar: no sequence specified");
        
        for (Overlay o : sequence.getOverlays(ScaleBarOverlay.class))
            sequence.removeOverlay(o);
    }
}
