package plugins.nchenouard.roiintensityevolution;

import icy.image.IcyBufferedImage;
import icy.roi.BooleanMask2D;
import icy.roi.ROI;
import icy.roi.ROI2D;
import icy.sequence.Sequence;
import icy.sequence.SequenceUtil;
import icy.type.collection.array.ArrayUtil;
import icy.type.rectangle.Rectangle5D;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.concurrent.locks.ReentrantLock;

import org.jfree.data.xy.XYSeries;

/**
 *  Object to manage intensity values of a ROI through time
 * 
 * @version 2.0 (08/02/2014)
 * 
 * 
 * @author Nicolas Chenouard (nicolas.chenouard.dev@gmail.com). Distributed under the GPLv3 license.
 */

public class ROIAnalysis
{
	private ROI roi;
	private Sequence sequence;
	private String description;

	private final XYSeries[] meanIntensity;
	private final XYSeries[] minIntensity;
	private final XYSeries[] maxIntensity;
	private final XYSeries[] medianIntensity;
	private final XYSeries[] sumIntensity;
	private final XYSeries[] varIntensity;
	private final XYSeries[] roiSize;
	
	ReentrantLock initLock = new ReentrantLock();

	public ROIAnalysis(ROI roi, Sequence sequence, String description, double threshold, boolean overthreshold, double scaling)
	{
		try{
			initLock.lock();
			if (roi == null)
				throw new IllegalArgumentException("NULL roi object is not allowed for creating a TrackAnalysis instance");
			if (sequence == null)
				throw new IllegalArgumentException("NULL Sequence object is not allowed for creating a TrackAnalysis instance");
			if (description == null)
				throw new IllegalArgumentException("NULL description object is not allowed for creating a TrackAnalysis instance");
			this.roi = roi;
			this.sequence = sequence;
			this.description = description;

			meanIntensity = new XYSeries[sequence.getSizeC()];
			for (int c = 0; c < sequence.getSizeC(); c++)
				meanIntensity[c] = new XYSeries(description);
			minIntensity = new XYSeries[sequence.getSizeC()];
			for (int c = 0; c < sequence.getSizeC(); c++)
				minIntensity[c] = new XYSeries(description);
			maxIntensity = new XYSeries[sequence.getSizeC()];
			for (int c = 0; c < sequence.getSizeC(); c++)
				maxIntensity[c] = new XYSeries(description);
			medianIntensity = new XYSeries[sequence.getSizeC()];
			for (int c = 0; c < sequence.getSizeC(); c++)
				medianIntensity[c] = new XYSeries(description);
			sumIntensity = new XYSeries[sequence.getSizeC()];
			for (int c = 0; c < sequence.getSizeC(); c++)
				sumIntensity[c] = new XYSeries(description);
			varIntensity = new XYSeries[sequence.getSizeC()];
			for (int c = 0; c < sequence.getSizeC(); c++)
				varIntensity[c] = new XYSeries(description);
			roiSize = new XYSeries[sequence.getSizeC()];
			for (int c = 0; c < sequence.getSizeC(); c++)
				roiSize[c] = new XYSeries(description);
			fillSeriesNoLock(threshold, overthreshold, scaling);
		}
		finally{
			initLock.unlock();
		}
	};
	
	
	private void fillSeriesNoLock(double threshold, boolean overthreshold, double areaScale)
	{
		for (int c = 0; c < sequence.getSizeC(); c++)
		{
			this.meanIntensity[c].clear();
			this.maxIntensity[c].clear();
			this.minIntensity[c].clear();
			this.varIntensity[c].clear();
			this.sumIntensity[c].clear();
			this.medianIntensity[c].clear();
		}
		
//		if (roi instanceof ROI2DArea)
		{
//			ROI2DArea roi2D = (ROI2DArea) roi;
			ROI2D roi2D = (ROI2D)roi;
			BooleanMask2D mask = roi2D.getBooleanMask(true);
			Rectangle5D.Integer bounds = roi.getBounds5D().toInteger();
			Sequence subSeq = SequenceUtil.getSubSequence(sequence, bounds);
			ArrayList<int[]> pixels = new ArrayList<int[]>();
			for (int y = bounds.y; y < bounds.y + bounds.sizeY; y++)
				for (int x = bounds.x; x < bounds.x + bounds.sizeX ; x++)
				{
					if (mask.contains(x, y))
						pixels.add(new int[]{x - bounds.x, y - bounds.y});
				}
			if (pixels.size() > 0)
			{
				int cntValues = pixels.size();
				for (int t = 0; t < subSeq.getSizeT(); t++)
				{
					IcyBufferedImage image = subSeq.getImage(t, 0);
					int width = image.getWidth();
					for (int c = 0; c < subSeq.getSizeC(); c++)
					{
						double[] data = (double[])ArrayUtil.arrayToDoubleArray(image.getDataXY(c), image.isSignedDataType());

						int[] p  = pixels.get(0);
						double value = data[p[0] + p[1]*width];
						double sumIntensity = value;
						double minIntensity = value;
						double maxIntensity = value;
						double sumSqIntensity = value*value;
						ArrayList<Double> valueList = new ArrayList<Double>();
						valueList.add(value);
						int cntPixThreshold = 0;
						if (overthreshold)
						{
							if (value > threshold)
								cntPixThreshold++;
							for (int i = 1; i < pixels.size(); i++)
							{
								p  = pixels.get(i);
								value = data[p[0] + p[1]*width];
								sumIntensity += value;
								sumSqIntensity += value*value;
								valueList.add(new Double(value));
								if (value < minIntensity)
									minIntensity = value;
								if (value > maxIntensity)
									maxIntensity = value;
								if (value > threshold)
									cntPixThreshold++;
							}
						}
						else
						{
							if (value < threshold)
								cntPixThreshold++;
							for (int i = 1; i < pixels.size(); i++)
							{
								p  = pixels.get(i);
								value = data[p[0] + p[1]*width];
								sumIntensity += value;
								sumSqIntensity += value*value;
								valueList.add(new Double(value));
								if (value < minIntensity)
									minIntensity = value;
								if (value > maxIntensity)
									maxIntensity = value;
								if (value < threshold)
									cntPixThreshold++;
							}
						}
						double meanIntensity = 0;
						double varIntensity = 0;
						if (cntValues > 0 )
						{
							meanIntensity = sumIntensity/cntValues;
							varIntensity = sumSqIntensity/cntValues - meanIntensity*meanIntensity;
						}
						this.meanIntensity[c].add(t, meanIntensity);
						this.maxIntensity[c].add(t, maxIntensity);
						this.minIntensity[c].add(t, minIntensity);
						this.varIntensity[c].add(t, varIntensity);
						this.sumIntensity[c].add(t, sumIntensity);
						if (!valueList.isEmpty())
						{
							Arrays.sort(valueList.toArray());
							if (cntValues == 1)
								this.medianIntensity[c].add(t, valueList.get(0));
							else
								if (cntValues%2 == 0)
								{
									this.medianIntensity[c].add(t, (valueList.get((int) cntValues/2)) + (valueList.get(((int) cntValues/2))-1)/2);
								}
								else
									this.medianIntensity[c].add(t, valueList.get((int) cntValues/2));				
						}
						this.roiSize[c].add(t, cntPixThreshold*areaScale);
					}
				}
			}
			else
			{
				for (int t = 0; t < sequence.getSizeT(); t++)
				{
					for (int c = 0; c < sequence.getSizeC(); c++)
					{
						this.meanIntensity[c].add(t, 0);
						this.maxIntensity[c].add(t, 0);
						this.minIntensity[c].add(t, 0);
						this.varIntensity[c].add(t, 0);
						this.sumIntensity[c].add(t, 0);
						this.medianIntensity[c].add(t ,0);
						this.roiSize[c].add(t, 0);
					}
				}
			}
		}
	}
	
	public void fillSeries(double threshold,  boolean overthreshold, double scale)
	{
		try{
			initLock.lock();
			fillSeriesNoLock(threshold, overthreshold, scale);
		}
		finally
		{
			initLock.unlock();
		}
	}

	public int getNumChannels()
	{
		try{
			initLock.lock();
			return sequence.getSizeC();
		}
		finally
		{
			initLock.unlock();
		}
	}

	public ROI getROI()
	{
		try{
			initLock.lock();
			return roi;
		}
		finally
		{
			initLock.unlock();
		}
	}


	public XYSeries[] getMeanIntensity()
	{
		try{
			initLock.lock();
			return meanIntensity;
		}
		finally
		{
			initLock.unlock();
		}
	}

	public XYSeries[] getMedianIntensity()
	{
		try{
			initLock.lock();
			return medianIntensity;
		}
		finally
		{
			initLock.unlock();
		}
	}

	public XYSeries[] getSumIntensity()
	{
		try{
			initLock.lock();
			return sumIntensity;
		}
		finally
		{
			initLock.unlock();
		}
	}

	public XYSeries[] getVarIntensity()
	{
		try{
			initLock.lock();
			return varIntensity;
		}
		finally
		{
			initLock.unlock();
		}
	}

	public XYSeries[] getMinIntensity()
	{
		try{
			initLock.lock();
			return minIntensity;
		}
		finally
		{
			initLock.unlock();
		}
	}

	public XYSeries[] getMaxIntensity()
	{
		try{
			initLock.lock();
			return maxIntensity;
		}
		finally
		{
			initLock.unlock();
		}
	}
	
	public XYSeries[] getROISize()
	{
		try{
			initLock.lock();
			return roiSize;
		}
		finally
		{
			initLock.unlock();
		}
	}

	public String getDescription()
	{
		try{
			initLock.lock();
			return description;
		}
		finally
		{
			initLock.unlock();
		}
	}
}