package plugins.tprovoost.videoimporter;

import icy.gui.dialog.MessageDialog;
import icy.gui.frame.progress.ProgressFrame;
import icy.main.Icy;
import icy.plugin.abstract_.PluginActionable;
import icy.preferences.PluginsPreferences;
import icy.preferences.XMLPreferences;
import icy.sequence.Sequence;
import icy.system.thread.ThreadUtil;

import java.awt.image.BufferedImage;
import java.io.File;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.filechooser.FileFilter;
import javax.swing.filechooser.FileNameExtensionFilter;

import plugins.fab.MiceProfiler.XugglerUtil;

import com.xuggle.xuggler.Global;
import com.xuggle.xuggler.ICodec;
import com.xuggle.xuggler.IContainer;
import com.xuggle.xuggler.IPacket;
import com.xuggle.xuggler.IPixelFormat;
import com.xuggle.xuggler.IStream;
import com.xuggle.xuggler.IStreamCoder;
import com.xuggle.xuggler.IVideoPicture;
import com.xuggle.xuggler.IVideoResampler;
import com.xuggle.xuggler.video.ConverterFactory;
import com.xuggle.xuggler.video.IConverter;

public class VideoImporter extends PluginActionable
{
	private static final String LAST_DIR = "last_directory";
	XMLPreferences prefs = PluginsPreferences.root(this);

	@Override
	public void run()
	{
		// Set the loading of videos
		final JFileChooser fc = new JFileChooser(prefs.get(LAST_DIR, ""));
		fc.setMultiSelectionEnabled(true);

		FileNameExtensionFilter filter = new FileNameExtensionFilter("AVI format", "avi", "AVI");
		fc.setFileFilter(filter);
		addFileFilter(fc, "MP4 format", "mp4", "MP4");
		addFileFilter(fc, "MOV format", "mov", "MOV");
		addFileFilter(fc, "WMV format", "wmv", "WMV");
		addFileFilter(fc, "MPEG format", "mpg", "MPG");
		fc.addChoosableFileFilter(new AllVideoFileFilter());

		// SHOW THE DIALOG
		int res = fc.showOpenDialog(Icy.getMainInterface().getMainFrame());
		if (res == JFileChooser.APPROVE_OPTION)
		{
			File[] files = fc.getSelectedFiles();
			if (files == null)
			{
				files = new File[]
				{ fc.getSelectedFile() };
			}
			for (int i = 0; i < files.length; ++i)
			{

				// RUN THE OPENING ON EVERY
				final File f = files[i];
				openVideo(f);
			}
		}
	}

	private void addFileFilter(JFileChooser fc, String description, String... extensions)
	{
		FileNameExtensionFilter filter = new FileNameExtensionFilter(description, extensions);
		fc.addChoosableFileFilter(filter);
	}

	private void openVideo(final File f)
	{
		prefs.put(LAST_DIR, f.getParent());

		ThreadUtil.bgRun(new Runnable()
		{
			@Override
			public void run()
			{
				boolean neverSeen = true;
				String filename = f.getAbsolutePath();
				ProgressFrame frame = new ProgressFrame("Loading Video: " + f.getName());
				frame.setLength(XugglerUtil.getVideoDuration(filename));

				// Create the sequence
				final Sequence sequence = new Sequence(f.getName());

				// new
				// AnnounceFrame("Please set the correct framerate on your sequence: "
				// + XugglerUtil.getVideoFrameRate(filename).getDenominator() +
				// " FPS.");

				// Let's make sure that we can actually convert video pixel
				// formats.
				if (!IVideoResampler.isSupported(IVideoResampler.Feature.FEATURE_COLORSPACECONVERSION))
					throw new RuntimeException("you must install the GPL version" + " of Xuggler (with IVideoResampler support) for " + "this demo to work");

				// Create a Xuggler container object
				IContainer container = IContainer.make();

				// Open up the container
				if (container.open(filename, IContainer.Type.READ, null) < 0)
					throw new IllegalArgumentException("could not open file: " + filename);

				// query how many streams the call to open found
				int numStreams = container.getNumStreams();

				// and iterate through the streams to find the first video
				// stream
				int videoStreamId = -1;
				IStreamCoder videoCoder = null;
				for (int i = 0; i < numStreams; i++)
				{
					// Find the stream object
					IStream stream = container.getStream(i);
					// Get the pre-configured decoder that can decode this
					// stream;
					IStreamCoder coder = stream.getStreamCoder();

					if (coder.getCodecType() == ICodec.Type.CODEC_TYPE_VIDEO)
					{
						videoStreamId = i;
						videoCoder = coder;
						break;
					}
				}
				if (videoStreamId == -1)
					throw new RuntimeException("could not find video stream in container: " + filename);

				/*
				 * Now we have found the video stream in this file. Let's open
				 * up our decoder so it can do work.
				 */
				if (videoCoder.open(null, null) < 0)
					throw new RuntimeException("could not open video decoder for container: " + filename);

				IVideoResampler resampler = null;
				if (videoCoder.getPixelType() != IPixelFormat.Type.BGR24)
				{
					// if this stream is not in BGR24, we're going to need to
					// convert it. The VideoResampler does that for us.
					resampler = IVideoResampler.make(videoCoder.getWidth(), videoCoder.getHeight(), IPixelFormat.Type.BGR24, videoCoder.getWidth(),
							videoCoder.getHeight(), videoCoder.getPixelType());
					if (resampler == null)
						throw new RuntimeException("could not create color space " + "resampler for: " + filename);
				}
				addSequence(sequence);
				try
				{
					/*
					 * Now, we start walking through the container looking at
					 * each packet.
					 */
					IPacket packet = IPacket.make();
					long firstTimestampInStream = Global.NO_PTS;
					long systemClockStartTime = 0;
					L1: while (container.readNextPacket(packet) >= 0)
					{
						/*
						 * Now we have a packet, let's see if it belongs to our
						 * video stream
						 */
						if (packet.getStreamIndex() == videoStreamId)
						{
							/*
							 * We allocate a new picture to get the data out of
							 * Xuggler
							 */
							IVideoPicture picture = IVideoPicture.make(videoCoder.getPixelType(), videoCoder.getWidth(), videoCoder.getHeight());

							int offset = 0;
							while (offset < packet.getSize())
							{
								/*
								 * Now, we decode the video, checking for any
								 * errors.
								 */
								int bytesDecoded = videoCoder.decodeVideo(picture, packet, offset);
								if (bytesDecoded < 0)
								{
									throw new RuntimeException("got error decoding video in: " + filename + " at offset: " + offset);
								}
								offset += bytesDecoded;

								/*
								 * Some decoders will consume data in a packet,
								 * but will not be able to construct a full
								 * video picture yet. Therefore you should
								 * always check if you got a complete picture
								 * from the decoder
								 */
								if (picture.isComplete())
								{
									IVideoPicture newPic = picture;
									/*
									 * If the resampler is not null, that means
									 * we didn't get the video in BGR24 format
									 * and need to convert it into BGR24 format.
									 */
									if (resampler != null)
									{
										// we must resample
										newPic = IVideoPicture.make(resampler.getOutputPixelFormat(), picture.getWidth(), picture.getHeight());
										if (resampler.resample(newPic, picture) < 0)
											throw new RuntimeException("could not resample video from: " + filename);
									}
									if (newPic.getPixelType() != IPixelFormat.Type.BGR24)
										throw new RuntimeException("could not decode video" + " as BGR 24 bit data in: " + filename);

									/**
									 * We could just display the images as
									 * quickly as we decode them, but it turns
									 * out we can decode a lot faster than you
									 * think. So instead, the following code
									 * does a poor-man's version of trying to
									 * match up the frame-rate requested for
									 * each IVideoPicture with the system clock
									 * time on your computer. Remember that all
									 * Xuggler IAudioSamples and IVideoPicture
									 * objects always give timestamps in
									 * Microseconds, relative to the first
									 * decoded item. If instead you used the
									 * packet timestamps, they can be in
									 * different units depending on your
									 * IContainer, and IStream and things can
									 * get hairy quickly.
									 */
									if (firstTimestampInStream == Global.NO_PTS)
									{
										// This is our first time through
										firstTimestampInStream = picture.getTimeStamp();
										// get the starting clock time so we can
										// hold up frames
										// until the right time.
										systemClockStartTime = System.currentTimeMillis();
									} else
									{
										long systemClockCurrentTime = System.currentTimeMillis();
										long millisecondsClockTimeSinceStartofVideo = systemClockCurrentTime - systemClockStartTime;
										// compute how long for this frame since
										// the first frame
										// in
										// the
										// stream.
										// remember that IVideoPicture and
										// IAudioSamples
										// timestamps
										// are
										// always in MICROSECONDS,
										// so we divide by 1000 to get
										// milliseconds.
										long millisecondsStreamTimeSinceStartOfVideo = (picture.getTimeStamp() - firstTimestampInStream) / 1000;
										final long millisecondsTolerance = 50; // and
																				// we
																				// give
																				// ourselfs
																				// 50
																				// ms
																				// of
																				// tolerance
										final long millisecondsToSleep = (millisecondsStreamTimeSinceStartOfVideo - (millisecondsClockTimeSinceStartofVideo + millisecondsTolerance));
										if (millisecondsToSleep > 0)
										{
											try
											{
												Thread.sleep(millisecondsToSleep);
											} catch (InterruptedException e)
											{
												// we might get this when the
												// user closes the
												// dialog
												// box, so
												// just return from the method.
												return;
											}
										}
									}

									// And finally, convert the BGR24 to an Java
									// buffered image
									boolean visible = Icy.getMainInterface().isOpened(sequence);
									if (neverSeen && visible)
									{
										neverSeen = false;
									} else if (!neverSeen && !visible)
									{
										break L1;
									}
									IConverter converter = ConverterFactory.createConverter(ConverterFactory.XUGGLER_BGR_24, newPic);
									BufferedImage javaImage = converter.toImage(newPic);
									int t = sequence.getSizeT();
									sequence.addImage(t, javaImage);
									frame.setPosition(t + 1);
								}
							}
						}

					}
				} catch (OutOfMemoryError e)
				{
					MessageDialog.showDialog("This video could not be opened entirely, please increase your available memory.", JOptionPane.WARNING_MESSAGE);
				} finally
				{
					/*
					 * Technically since we're exiting anyway, these will be
					 * cleaned up by the garbage collector... but because we're
					 * nice people and want to be invited places for Christmas,
					 * we're going to show how to clean up.
					 */
					if (videoCoder != null)
					{
						videoCoder.close();
						videoCoder = null;
					}
					if (container != null)
					{
						container.close();
						container = null;
					}
					frame.close();
				}
			}
		});

	}

	public static class AllVideoFileFilter extends FileFilter
	{
		@Override
		public boolean accept(File file)
		{
			try
			{
				return isOpenable(file.getAbsolutePath());
			} catch (IllegalArgumentException e)
			{
				return false;
			}
		}

		@Override
		public String getDescription()
		{
			return "All video files";
		}
	}

	private static boolean isOpenable(String filename)
	{
		if (new File(filename).isDirectory())
			return true;

		final IContainer container = IContainer.make();
		boolean res = container.open(filename, IContainer.Type.READ, null) >= 0;
		if (res)
		{
			container.close();
		}
		return res;
	}
}