/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.blobplugin.operations;

import icy.type.point.Point4D;

import java.util.ArrayList;
import java.util.List;

import plugins.big.blobplugin.descriptors.ImageDescriptor;

/**
 * @class MathOperations
 * 
 *        Abstract class implementing some basic mathematics operations.
 * 
 */
public abstract class MathOperations {
	/**
	 * Add the image A and B and copy the result in C.
	 * 
	 * @throws Exception
	 *             A, B and C must have the same size.
	 */
	static public void add(final ImageDescriptor A, final ImageDescriptor B,
			ImageDescriptor C) throws Exception {
		if (A.width != B.width || A.height != B.height || A.depth != B.depth
				|| A.width != C.width || A.height != C.height
				|| A.depth != C.depth) {
			throw new Exception(
					"A, B and C must have the same size in MathOperations.add");
		}

		for (int z = 0; z < A.depth; z++) {
			for (int y = 0; y < A.height; y++) {
				for (int x = 0; x < A.width; x++) {
					C.setValue(x, y, z, A.value(x, y, z) + B.value(x, y, z));
				}
			}
		}
	}

	static public void multiply(final ImageDescriptor A, double B,
			ImageDescriptor C) throws Exception {
		if (A.width != C.width || A.height != C.height || A.depth != C.depth) {
			throw new Exception(
					"A and B must have the same size in MathOperations.sqrt");
		}

		for (int z = 0; z < A.depth; z++) {
			for (int y = 0; y < A.height; y++) {
				for (int x = 0; x < A.width; x++) {
					C.setValue(x, y, z, A.value(x, y, z) * B);
				}
			}
		}
	}

	static public void multiply(double A, final ImageDescriptor B,
			ImageDescriptor C) throws Exception {
		multiply(B, A, C);
	}

	static public void multiply(final ImageDescriptor A,
			final ImageDescriptor B, ImageDescriptor C) throws Exception {
		if (A.width != B.width || A.height != B.height || A.depth != B.depth
				|| A.width != C.width || A.height != C.height
				|| A.depth != C.depth) {
			throw new Exception(
					"A, B and C must have the same size in MathOperations.add");
		}

		for (int z = 0; z < A.depth; z++) {
			for (int y = 0; y < A.height; y++) {
				for (int x = 0; x < A.width; x++) {
					C.setValue(x, y, z, A.value(x, y, z) * B.value(x, y, z));
				}
			}
		}
	}

	static public void sqrt(final ImageDescriptor A, ImageDescriptor B)
			throws Exception {
		if (A.width != B.width || A.height != B.height || A.depth != B.depth) {
			throw new Exception(
					"A and B must have the same size in MathOperations.sqrt");
		}

		for (int z = 0; z < A.depth; z++) {
			for (int y = 0; y < A.height; y++) {
				for (int x = 0; x < A.width; x++) {
					B.setValue(x, y, z, Math.sqrt(A.value(x, y, z)));
				}
			}
		}
	}

	/**
	 * Find the local maxima of an image.
	 * 
	 * @param src
	 *            The input image
	 * @param threshold
	 *            The minimal value of a maximum
	 * 
	 * @return Return the list of the local maxima coordinates.
	 */
	static public List<Point4D> getLocalMaxima(final ImageDescriptor src,
			double threshold) {
		List<Point4D> indices = new ArrayList<Point4D>();

		int[][] D = null;
		if (src.depth == 1) {
			D = new int[][] { { -1, -1, 0 }, { -1, 0, 0 }, { -1, 1, 0 },
					{ 0, -1, 0 }, { 0, 1, 0 }, { 1, -1, 0 }, { 1, 0, 0 },
					{ 1, 1, 0 } };
		} else {
			D = new int[][] { { -1, -1, -1 }, { -1, -1, 0 }, { -1, -1, 1 },
					{ -1, 0, -1 }, { -1, 0, 0 }, { -1, 0, 1 }, { -1, 1, -1 },
					{ -1, 1, 0 }, { -1, 1, 1 }, { 0, -1, -1 }, { 0, -1, 0 },
					{ 0, -1, 1 }, { 0, 0, -1 }, { 0, 0, 1 }, { 0, 1, -1 },
					{ 0, 1, 0 }, { 0, 1, 1 }, { 1, -1, -1 }, { 1, -1, 0 },
					{ 1, -1, 1 }, { 1, 0, -1 }, { 1, 0, 0 }, { 1, 0, 1 },
					{ 1, 1, -1 }, { 1, 1, 0 }, { 1, 1, 1 } };
		}

		ImageDescriptor bluredSrc = new ImageDescriptor(src.width, src.height,
				src.depth, src.type);
		try {
			Filters.gaussianBlur(src, bluredSrc, 2);
		} catch (Exception e) {
			e.printStackTrace();
			bluredSrc = src;
		}

		for (int z = 0; z < src.depth; ++z) {
			for (int y = 0; y < src.height; ++y) {
				for (int x = 0; x < src.width; ++x) {
					double value = src.value(x, y, z);

					if (value >= threshold) {
						boolean isMax = true;

						for (int[] d : D) {
							Double v = src.value(x + d[0], y + d[1], z + d[2]);
							if (!v.isNaN()) {
								isMax &= (value >= v);
							}
							if (!isMax) {
								break;
							}
						}

						if (isMax) {
							indices.add(new Point4D.Double(x, y, z, value));
						}
					}
				}
			}
		}

		return indices;
	}

	/**
	 * Compute the distance map of an image.
	 * 
	 * @param src
	 *            The input image
	 * @param dst
	 *            The result image
	 * 
	 * @throws Exception
	 *             src and dst must have the same size.
	 */
	static public void getDistanceMap(final ImageDescriptor src,
			ImageDescriptor dst) throws Exception {
		if (src.width != dst.width || src.height != dst.height
				|| src.depth != dst.depth) {
			throw new Exception(
					"src and dst must have the same size in MathOperations.getDistanceMap");
		}

		double[] buffer1 = (src.depth == 1) ? new double[src.depth * src.width
				* src.height] : dst.getData();
		double[] buffer2 = (src.depth == 1) ? dst.getData()
				: new double[src.depth * src.width * src.height];

		ImageDescriptor scaledSrc = new ImageDescriptor(src.width, src.height,
				src.depth, src.type);
		multiply(src, src.width * src.height * src.depth, scaledSrc);

		for (int z = 0; z < src.depth; z++) {
			int offset = src.offset + z * src.zStep;
			for (int row = 0; row < src.height; row++) {
				distanceTransformLine(scaledSrc.getData(), buffer1, offset
						+ row * src.yStep, src.width, 1);
			}
			for (int column = 0; column < src.width; column++) {
				distanceTransformLine(buffer1, buffer2, offset + column,
						src.height, src.yStep);
			}
		}

		if (src.depth > 1) {
			int i = src.offset;
			for (int y = 0; y < src.height; y++) {
				for (int x = 0; x < src.width; x++) {
					distanceTransformLine(buffer2, dst.getData(), i, src.depth,
							src.zStep);
					i++;
				}
				i += src.yStep - src.width;
			}
		}

		sqrt(dst, dst);
	}

	static public void getImageIntegral(final ImageDescriptor src,
			ImageDescriptor dst) throws Exception {
		if (src.width != dst.width || src.height != dst.height
				|| src.depth != dst.depth) {
			throw new Exception(
					"src and dst must have the same size in MathOperations.getImageIntegral");
		}

		ImageDescriptor scaledSrc = new ImageDescriptor(src.width, src.height,
				src.depth, src.type);
		switch (src.type) {
		case BYTE:
		case INT:
		case LONG:
		case SHORT:
		case UBYTE:
		case UINT:
		case ULONG:
		case UNDEFINED:
		case USHORT:
			multiply(src, 1.0 / 255.0, scaledSrc);
			break;
		case DOUBLE:
		case FLOAT:
			multiply(src, 1.0, scaledSrc);
			break;
		}

		for (int z = 0; z < src.depth; z++) {
			int offset = src.offset + z * src.zStep;
			for (int row = 0; row < src.height; row++) {
				accumulateLine(scaledSrc.getData(), dst.getData(), offset + row
						* src.yStep, src.width, 1);
			}
		}
	}

	private static void distanceTransformLine(final double[] src, double[] dst,
			int offset, int size, int step) {
		int k = 0;
		int[] v = new int[size];
		double[] z = new double[size + 1];
		double s;

		v[0] = 0;
		z[0] = Double.NEGATIVE_INFINITY;
		z[1] = Double.POSITIVE_INFINITY;

		int index = offset + step;
		for (int i = 1; i < size; i++) {
			for (;;) {
				s = (((src[index] + i * i) - (src[offset + v[k] * step] + v[k]
						* v[k])) / (2 * i - 2 * v[k]));

				if (s <= z[k]) {
					k--;
				} else {
					break;
				}
			}
			k++;
			index += step;

			v[k] = i;
			z[k] = s;
			z[k + 1] = Double.POSITIVE_INFINITY;
		}

		k = 0;
		for (int i = 0; i < size; i++) {
			while (z[k + 1] < i) {
				k++;
			}

			int d = i - v[k];
			dst[offset + i * step] = d * d + src[offset + v[k] * step];
		}
	}

	private static void accumulateLine(final double[] src, double[] dst,
			int offset, int size, int step) {
		dst[offset] = src[offset];
		int i = offset, ii = offset + step;
		for (int k = 1; k < size; ++k, i = ii, ii += step) {
			dst[ii] = dst[i] + src[ii];
		}
	}
}
