/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.blobgenerator.filters;

import icy.image.IcyBufferedImage;
import icy.type.DataType;

/**
 * Abstract class to blur images.
 * 
 * @version May 3, 2014
 * 
 * @author Julien Jacquemot
 */
public abstract class BlurFilter {
	/**
	 * Apply a Gaussian blur to the given image with the given standard
	 * deviation.
	 */
	public static IcyBufferedImage[] apply(final IcyBufferedImage inputs[],
			double sigma) {
		IcyBufferedImage res[] = new IcyBufferedImage[inputs.length];

		// Create the kernel
		int kernelSize = (int) Math.max(1, 6 * sigma);
		if (kernelSize % 2 == 0) {
			++kernelSize;
		}

		double[] kernel = new double[kernelSize];
		if (sigma == 0) {
			kernel[0] = 1;
		} else {
			int c = (kernelSize - 1) / 2;
			double sum = 0;
			for (int i = 0; i < kernelSize; i++) {
				sum += kernel[i] = Math.exp(-(i - c) * (i - c)
						/ (2 * sigma * sigma));
			}
			for (int i = 0; i < kernelSize; i++) {
				kernel[i] /= sum;
			}
		}

		IcyBufferedImage zBuffer[] = new IcyBufferedImage[inputs.length];
		for (int z = 0; z < zBuffer.length; ++z) {
			zBuffer[z] = new IcyBufferedImage(inputs[z].getWidth(),
					inputs[z].getHeight(), 1, DataType.DOUBLE);
			convolveDepthLine(inputs, zBuffer[z], kernel, z);
		}

		for (int z = 0; z < zBuffer.length; ++z) {
			res[z] = new IcyBufferedImage(zBuffer[z].getWidth(),
					zBuffer[z].getHeight(), 1, DataType.DOUBLE);

			double src[] = zBuffer[z].getDataXYAsDouble(0);
			double dst[] = res[z].getDataXYAsDouble(0);
			double buffer[] = new double[src.length];

			for (int r = 0; r < zBuffer[z].getHeight(); ++r) {
				convolveLine(src, buffer, kernel, r * zBuffer[z].getWidth(),
						zBuffer[z].getWidth(), 1);
			}

			for (int c = 0; c < zBuffer[z].getWidth(); ++c) {
				convolveLine(buffer, dst, kernel, c, zBuffer[z].getHeight(),
						zBuffer[z].getWidth());
			}
		}

		return res;
	}

	/**
	 * Convolves a line from an image with a given filter.
	 * 
	 * @param src
	 *            Input image
	 * @param dst
	 *            Destination image
	 * @param kernel
	 *            Kernel to use
	 * @param offset
	 *            First index of the line
	 * @param size
	 *            Size of the line
	 * @param step
	 *            Step to circulate between elements of the line
	 */
	private static void convolveLine(final double[] src, double[] dst,
			final double[] kernel, int offset, int size, int step) {
		int k0 = kernel.length / 2;

		int index = offset;
		for (int i = 0; i < size; ++i) {
			dst[index] = 0;
			for (int k = 0; k < kernel.length; ++k) {
				int j = i + k - k0;
				if (j < 0) {
					j = -j;
				} else if (j >= size) {
					j = 2 * size - j - 1;
				}

				dst[index] += src[offset + j * step] * kernel[k];
			}
			index += step;
		}
	}

	/**
	 * Convolves a line of an in image in depth with the given kernel.
	 * 
	 * @param srcImg
	 *            Input image
	 * @param dstImg
	 *            Destination image
	 * @param kernel
	 *            Kernel to use
	 * @param z0
	 *            Current depth
	 */
	private static void convolveDepthLine(final IcyBufferedImage[] srcImg,
			IcyBufferedImage dstImg, final double[] kernel, int z0) {
		int k0 = kernel.length / 2;
		double dst[] = dstImg.getDataXYAsDouble(0);
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = 0;
		}

		for (int k = 0; k < kernel.length; ++k) {
			int z = z0 + k - k0;
			while (z < 0 || z >= srcImg.length) {
				if (z < 0) {
					z = -z;
				} else {
					z = 2 * srcImg.length - z - 1;
				}
			}

			double src[] = srcImg[z].getDataXYAsDouble(0);
			for (int i = 0; i < src.length; ++i) {
				dst[i] += src[i] * kernel[k];
			}
		}
	}
}
