/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.blobgenerator.parameters;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import plugins.adufour.ezplug.EzComponent;
import plugins.adufour.ezplug.EzVarDouble;
import plugins.adufour.ezplug.EzVarEnum;
import plugins.adufour.ezplug.EzVarInteger;
import plugins.adufour.ezplug.EzVarNumeric;
import plugins.adufour.vars.lang.Var;
import plugins.adufour.vars.util.VarListener;
import plugins.big.bigsnakeutils.icy.gui.curve.Curve;
import plugins.big.bigsnakeutils.icy.gui.curve.EzVarCurve;
import plugins.big.bigsnakeutils.icy.gui.pair.EzVarPair;
import plugins.big.bigsnakeutils.icy.gui.pair.Pair;
import plugins.big.bigsnakeutils.icy.gui.pair.RangeRangeModel;
import plugins.big.blobgenerator.BlobGenerator;
import plugins.big.blobgenerator.filters.BlendingMode;

/**
 * Class containing all the parameters tunable by the use. It can be used with
 * Protocols.
 * 
 * @version May 3, 2014
 * 
 * @author Julien Jacquemot
 */
public class Parameters {
	/**
	 * HeadLess means that the plugin is launch through its normal version,
	 * otherwise is through Protocols
	 */
	final private boolean _isHeadLess;

	/** Map of all the EzVar indexed by their category. */
	final public HashMap<ParametersCategory, List<EzComponent>> ezVariables = new HashMap<ParametersCategory, List<EzComponent>>();

	/**
	 * Map of all the Var indexed by their category. Each Var is associated to
	 * an EzVar.
	 */
	final public HashMap<ParametersCategory, List<Var<?>>> variables = new HashMap<ParametersCategory, List<Var<?>>>();

	private final EzVarInteger _ezVar_cellsCount;
	private final EzVarPair.Integer _ezVar_cellsRadius;
	private final EzVarPair.Integer _ezVar_cellsVelocity;
	private final EzVarPair.Double _ezVar_cellsEccentricity1;
	private final EzVarPair.Double _ezVar_cellsEccentricity2;
	private final EzVarDouble _ezVar_cellsBlurRadius;

	private final EzVarInteger _ezVar_noiseLevels;
	private final EzVarDouble _ezVar_noiseIntensity;
	private final EzVarEnum<BlendingMode> _ezVar_noiseBlendingMode;
	private final EzVarCurve _ezVar_noiseProfile;

	private final EzVarDouble _ezVar_backgroundValue;
	private final EzVarPair.Double _ezVar_foregroundValueRange;
	private final EzVarInteger _ezVar_imgWidth;
	private final EzVarInteger _ezVar_imgHeight;
	private final EzVarInteger _ezVar_imgDepth;
	private final EzVarInteger _ezVar_duration;

	/** Default constructor. */
	public Parameters(final BlobGenerator parent, boolean headless) {
		_isHeadLess = headless;
		for (ParametersCategory category : ParametersCategory.values()) {
			ezVariables.put(category, new ArrayList<EzComponent>());
			variables.put(category, new ArrayList<Var<?>>());
		}

		_ezVar_cellsCount = newEzVarInteger(ParametersCategory.CellsProperties,
				"Cells Count", 5, 0, 1000, 1, "Number of cells to generate");
		_ezVar_cellsRadius = newEzVarIntegerRange(
				ParametersCategory.CellsProperties, "Cells Radius Range (px)",
				new Pair.Integer(5, 10), 1, 1000, 1,
				"Minimum and maximum cell radius");
		_ezVar_cellsVelocity = newEzVarIntegerRange(
				ParametersCategory.CellsProperties, "Cells Velocity Range",
				new Pair.Integer(-5, 5), -100, 100, 1,
				"Minimum and maximum cell velocity");
		_ezVar_cellsEccentricity1 = newEzVarDoubleRange(
				ParametersCategory.CellsProperties,
				"Cells Eccentricity Range 1",
				new Pair.Double(0.4, 0.8),
				0.0,
				0.99,
				0.1,
				"Minimum and maximum first eccentricity of cells.\nA value of 0 corresponds to a circle, a value close to 1 corresponds to an ellongated shape.");
		_ezVar_cellsEccentricity2 = newEzVarDoubleRange(
				ParametersCategory.CellsProperties,
				"Cells Eccentricity Range 2",
				new Pair.Double(0.0, 0.0),
				0.0,
				0.99,
				0.1,
				"Minimum and maximum second eccentricity of cells.\nA value of 0 corresponds to a circle, a value close to 1 corresponds to an ellongated shape.\n\nWARNING: Only make sense for 3D cells.");
		_ezVar_cellsBlurRadius = newEzVarDouble(
				ParametersCategory.CellsProperties, "Cells Blur Radius", 2.0,
				0.0, 20.0, 1.0,
				"Defines how much the cells are blured. A value of 0 means no blur.");

		_ezVar_noiseLevels = newEzVarInteger(
				ParametersCategory.NoiseProperties,
				"Noise Levels",
				8,
				1,
				12,
				1,
				"Scale of the noise.\nA value of 1 gives independant values for each pixels. Higher values lead to a more structured noise.");
		_ezVar_noiseIntensity = newEzVarDouble(
				ParametersCategory.NoiseProperties, "Noise Intensity", 0.5,
				0.0, 1.0, 0.1,
				"Intensity of the noise.\nA value of 0 means no noise.");
		_ezVar_noiseBlendingMode = newEzVarEnum(
				ParametersCategory.NoiseProperties,
				"Blending Mode",
				BlendingMode.SoftLight,
				BlendingMode.values(),
				"Defines the blending mode used to mix the cells and the noise.\nUse the preview to have an idea of the different available modes.\n\nWARNING: Uncheck the 'Auto bounds' box to really see what happens.");
		_ezVar_noiseProfile = newEzVarCurve(
				ParametersCategory.NoiseProperties,
				"Noise Profile",
				0.5,
				0.5,
				"Noise density function.\nRepresents the probability to observe one pixel at the given value (for a level of 1).\n Left click to add a control point, right click to remove a point.");

		_ezVar_backgroundValue = newEzVarDouble(
				ParametersCategory.SequenceProperties, "Background Value",
				0.05, 0.0, 1.0, 0.1,
				"Value of the background.\n0 is black and 1 is white.");
		_ezVar_foregroundValueRange = newEzVarDoubleRange(
				ParametersCategory.SequenceProperties,
				"Foreground Value Range", new Pair.Double(0.5, 0.8), 0.0, 1.0,
				0.1,
				"Range of value used to display the cells.\n0 is black, 1 is white.");
		_ezVar_imgWidth = newEzVarInteger(
				ParametersCategory.SequenceProperties, "Image Width", 256, 1,
				4096, 1, "Width of the generated image.");
		_ezVar_imgHeight = newEzVarInteger(
				ParametersCategory.SequenceProperties, "Image Height", 256, 1,
				4096, 1, "Height of the generated image.");
		_ezVar_imgDepth = newEzVarInteger(
				ParametersCategory.SequenceProperties, "Image Depth", 1, 1,
				4096, 1, "Depth of the generated image.");
		_ezVar_duration = newEzVarInteger(
				ParametersCategory.SequenceProperties,
				"Sequence Duration (frames)", 1, 1, 9999, 1,
				"Duration of the generated sequence.");

		// Create VarListeners to update the noise sequence when one of the
		// noise parameter is changed
		if (headless) {
			_ezVar_noiseLevels.getVariable().addListener(
					new VarListener<Integer>() {
						@Override
						public void valueChanged(Var<Integer> source,
								Integer oldValue, Integer newValue) {
							parent.valueChanged(source);
						}

						@Override
						public void referenceChanged(Var<Integer> source,
								Var<? extends Integer> oldReference,
								Var<? extends Integer> newReference) {
						}
					});
			_ezVar_noiseIntensity.getVariable().addListener(
					new VarListener<Double>() {
						@Override
						public void valueChanged(Var<Double> source,
								Double oldValue, Double newValue) {
							parent.valueChanged(source);
						}

						@Override
						public void referenceChanged(Var<Double> source,
								Var<? extends Double> oldReference,
								Var<? extends Double> newReference) {
						}
					});
			_ezVar_noiseBlendingMode.getVariable().addListener(
					new VarListener<BlendingMode>() {
						@Override
						public void valueChanged(Var<BlendingMode> source,
								BlendingMode oldValue, BlendingMode newValue) {
							parent.valueChanged(source);
						}

						@Override
						public void referenceChanged(Var<BlendingMode> source,
								Var<? extends BlendingMode> oldReference,
								Var<? extends BlendingMode> newReference) {
						}
					});
			_ezVar_noiseProfile.getVariable().addListener(
					new VarListener<Curve>() {
						@Override
						public void valueChanged(Var<Curve> source,
								Curve oldValue, Curve newValue) {
							parent.valueChanged(source);
						}

						@Override
						public void referenceChanged(Var<Curve> source,
								Var<? extends Curve> oldReference,
								Var<? extends Curve> newReference) {
						}
					});
			_ezVar_backgroundValue.getVariable().addListener(
					new VarListener<Double>() {
						@Override
						public void valueChanged(Var<Double> source,
								Double oldValue, Double newValue) {
							parent.valueChanged(source);
						}

						@Override
						public void referenceChanged(Var<Double> source,
								Var<? extends Double> oldReference,
								Var<? extends Double> newReference) {
						}
					});
		}
	}

	/** Returns the desired number of cells. */
	public int cellsCount() {
		return _ezVar_cellsCount.getValue();
	}

	/** Returns the range for the cells radius. */
	public Pair cellsRadiusRange() {
		return _ezVar_cellsRadius.getValue();
	}

	/** Returns the range for the cells velocity. */
	public Pair cellsVelocityRange() {
		return _ezVar_cellsVelocity.getValue();
	}

	/** Returns the range for the first cells eccentricity. */
	public Pair cellsEccentricityRange1() {
		return _ezVar_cellsEccentricity1.getValue();
	}

	/** Returns the range for the second cells eccentricity. */
	public Pair cellsEccentricityRange2() {
		return _ezVar_cellsEccentricity2.getValue();
	}

	/** Returns the blur factor for cells. */
	public double cellsBlurRadius() {
		return _ezVar_cellsBlurRadius.getValue();
	}

	/**
	 * Returns the maximal noise scale. All the scale between 1 and noiseLevel()
	 * will be accumulated.
	 */
	public int noiseLevels() {
		return _ezVar_noiseLevels.getValue();
	}

	/** Returns the nosie intensity. A value of 0 means no noise. */
	public double noiseIntensity() {
		return _ezVar_noiseIntensity.getValue();
	}

	/** Returns the blending mode used to mix the noise and the cells. */
	public BlendingMode noiseBlendingMode() {
		return _ezVar_noiseBlendingMode.getValue();
	}

	/** Returns the value used for the background of the images. */
	public double backgroundValue() {
		return _ezVar_backgroundValue.getValue();
	}

	/** Returns the range of values used to display the cells. */
	public Pair foregroundValueRange() {
		return _ezVar_foregroundValueRange.getValue();
	}

	/** Returns the noise density function. */
	public Curve noiseProfile() {
		return _ezVar_noiseProfile.getValue();
	}

	/** Returns the desired image width. */
	public int imageWidth() {
		return _ezVar_imgWidth.getValue();
	}

	/** Returns the desired image height. */
	public int imageHeight() {
		return _ezVar_imgHeight.getValue();
	}

	/** Returns the desired image depth. */
	public int imageDepth() {
		return _ezVar_imgDepth.getValue();
	}

	/** Returns the duration of the generated sequence. */
	public int sequenceDuration() {
		return _ezVar_duration.getValue();
	}

	/** Create a new EzVarDouble. */
	private EzVarDouble newEzVarDouble(ParametersCategory category,
			final String name, final Double defaultValue,
			final Double minValue, final Double maxValue,
			final Double stepValue, final String tooltip) {
		EzVarDouble var = new EzVarDouble(name);
		initEzVarNumeric(var, category, defaultValue, minValue, maxValue,
				stepValue, tooltip);
		var.setStep(0.01);
		return var;
	}

	/** Create a new EzVarInteger. */
	private EzVarInteger newEzVarInteger(ParametersCategory category,
			final String name, final Integer defaultValue,
			final Integer minValue, final Integer maxValue,
			final Integer stepValue, final String tooltip) {
		EzVarInteger var = new EzVarInteger(name);
		initEzVarNumeric(var, category, defaultValue, minValue, maxValue,
				stepValue, tooltip);
		return var;
	}

	/** Create a new EzVarEnum. */
	private <E extends Enum<E>> EzVarEnum<E> newEzVarEnum(
			ParametersCategory category, final String name, E defaultValue,
			final E[] values, final String tooltip) {
		EzVarEnum<E> var = new EzVarEnum<E>(name, values);

		var.setToolTipText(tooltip);
		var.setValue(defaultValue);

		ezVariables.get(category).add(var);
		variables.get(category).add(var.getVariable());

		return var;
	}

	/** Create a new EzVarPair.Integer as range editor. */
	private EzVarPair.Integer newEzVarIntegerRange(ParametersCategory category,
			final String name, final Pair.Integer defaultValue, int minValue,
			int maxValue, int stepValue, final String tooltip) {
		EzVarPair.Integer var = new EzVarPair.Integer(
				name,
				new RangeRangeModel(defaultValue, minValue, maxValue, stepValue));
		initEzVarRange(var, category, defaultValue, minValue, maxValue,
				stepValue, tooltip);
		return var;
	}

	/** Create a new EzVarRange.Double as range editor. */
	private EzVarPair.Double newEzVarDoubleRange(ParametersCategory category,
			final String name, final Pair.Double defaultValue, double minValue,
			double maxValue, double stepValue, final String tooltip) {
		EzVarPair.Double var = new EzVarPair.Double(name, new RangeRangeModel(
				defaultValue, minValue, maxValue, stepValue));
		initEzVarRange(var, category, defaultValue, minValue, maxValue,
				stepValue, tooltip);
		return var;
	}

	/** Create a new EzVarCurve. */
	private EzVarCurve newEzVarCurve(ParametersCategory category,
			final String name, double y0, double y1, final String tooltip) {
		EzVarCurve var = new EzVarCurve.Polynomial(name, y0, y1);

		var.setToolTipText(tooltip);

		ezVariables.get(category).add(var);
		variables.get(category).add(var.getVariable());

		return var;
	}

	/** Initialize an EzVarNumeric (EzVarInteger or EzVarDouble) */
	private <N extends Number> void initEzVarNumeric(EzVarNumeric<N> var,
			final ParametersCategory category, final N defaultValue,
			final Comparable<N> minValue, final Comparable<N> maxValue,
			final N stepValue, final String tooltip) {
		if (_isHeadLess) { // temp (bug with the min)
			var.setValues(defaultValue, minValue, maxValue, stepValue);
		} else {
			var.setStep(stepValue);
			var.setMaxValue(maxValue);
		}
		var.setToolTipText(tooltip);
		var.setValue(defaultValue);

		ezVariables.get(category).add(var);
		variables.get(category).add(var.getVariable());
	}

	/** Initialize an EzVarPair to be a range editor. */
	private void initEzVarRange(EzVarPair var,
			final ParametersCategory category, final Pair defaultValue,
			double minValue, double maxValue, double stepValue,
			final String tooltip) {
		var.setToolTipText(tooltip);
		var.setValue(defaultValue);
		var.setLinkingWord("to");

		ezVariables.get(category).add(var);
		variables.get(category).add(var.getVariable());
	}
}
