package plugins.adufour.cmyk;

import icy.gui.viewer.Viewer;
import icy.image.IcyBufferedImage;
import icy.image.IcyBufferedImageUtil;
import icy.image.colormap.LinearColorMap;
import icy.image.lut.LUT;
import icy.sequence.Sequence;
import icy.system.IcyHandledException;
import icy.system.thread.ThreadUtil;

import java.awt.Color;
import java.awt.RenderingHints;
import java.awt.color.ColorSpace;
import java.awt.color.ICC_ColorSpace;
import java.awt.color.ICC_Profile;
import java.awt.image.BufferedImage;
import java.awt.image.ColorConvertOp;
import java.awt.image.ColorModel;
import java.awt.image.ComponentColorModel;
import java.awt.image.DataBuffer;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;

import javax.imageio.ImageIO;

import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzVarBoolean;
import plugins.adufour.ezplug.EzVarEnum;
import plugins.adufour.ezplug.EzVarFile;

public class ConvertToCMYK extends EzPlug
{
    public static enum CMYK_Profile
    {
        CoatedFOGRA27,
        CoatedFOGRA39,
        JapanColor2001Coated,
        JapanColor2001Uncoated,
        JapanColor2002Newspaper,
        JapanWebCoated,
        UncoatedFOGRA29,
        USSheetfedCoated,
        USSheetfedUncoated,
        USWebCoatedSWOP,
        USWebUncoated,
        WebCoatedFOGRA28
    }
    
    public static final HashMap<CMYK_Profile, ICC_Profile> CMYK_ColorSpaces = new HashMap<ConvertToCMYK.CMYK_Profile, ICC_Profile>(CMYK_Profile.values().length);
    
    static
    {
        for (CMYK_Profile profile : CMYK_Profile.values())
        {
            try
            {
                InputStream profileStream = ConvertToCMYK.class.getResourceAsStream(profile.name() + ".icc");
                CMYK_ColorSpaces.put(profile, ICC_Profile.getInstance(profileStream));
            }
            catch (Exception e)
            {
                System.err.println("Warning: unable to load CMYK profile: " + profile.name() + ". Reason: " + e.getMessage());
            }
        }
    }
    
    public static float[] rgbToCmyk(CMYK_Profile profile, float... rgb)
    {
        if (rgb.length != 3) throw new IcyHandledException("The image must be in 3-channel RGB format");
        
        return new ICC_ColorSpace(CMYK_ColorSpaces.get(profile)).fromRGB(rgb);
    }
    
    public static float[] cmykToRgb(CMYK_Profile profile, float... cmyk)
    {
        if (cmyk.length != 4) throw new IcyHandledException("The image must be in 3-channel RGB format");
        
        return new ICC_ColorSpace(CMYK_ColorSpaces.get(profile)).toRGB(cmyk);
    }
    
    public EzVarEnum<CMYK_Profile> profiles = new EzVarEnum<CMYK_Profile>("CMYK profile", CMYK_Profile.values(), CMYK_Profile.USWebCoatedSWOP);
    
    public EzVarFile output = new EzVarFile("Save as...", null);
    
    public EzVarBoolean view = new EzVarBoolean("Display CMYK channels", false);
    
    @Override
    protected void initialize()
    {
        addEzComponent(profiles);
        addEzComponent(output);
        addEzComponent(view);
    }
    
    @Override
    public void execute()
    {
        Viewer v = getActiveViewer();
        
        if (v == null) throw new IcyHandledException("[CMYK converter] Open an image first.");
        
        LUT lut = v.getLut();
        
        if (lut.getNumChannel() != 3) throw new IcyHandledException("The image must be in 3-channel RGB format");
        
        ICC_Profile profile = CMYK_ColorSpaces.get(profiles.getValue());
        
        IcyBufferedImage image = v.getCurrentImage();
        
        int width = image.getWidth();
        int height = image.getHeight();
        
        final IcyBufferedImage cmykImage = new IcyBufferedImage(width, height, 4, image.getDataType_());
        
        RenderingHints hints = new RenderingHints(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
        hints.put(RenderingHints.KEY_COLOR_RENDERING, RenderingHints.VALUE_COLOR_RENDER_QUALITY);
        hints.put(RenderingHints.KEY_DITHERING, RenderingHints.VALUE_DITHER_ENABLE);
        ColorConvertOp op = new ColorConvertOp(new ICC_Profile[] { ICC_Profile.getInstance(ColorSpace.CS_sRGB), profile }, hints);
        
        op.filter(image.getRaster(), cmykImage.getRaster());
        
        if (output.getValue() != null)
        {
            try
            {
                final ColorModel model = new ComponentColorModel(new ICC_ColorSpace(profile), new int[] { 8, 8, 8, 8 }, false, false, ColorModel.OPAQUE, DataBuffer.TYPE_BYTE);
                BufferedImage cmykImageReal = new BufferedImage(model, cmykImage.getRaster(), false, null);
                ImageIO.write(cmykImageReal, "TIFF", output.getValue());
            }
            catch (IOException e)
            {
                e.printStackTrace();
            }
        }
        
        if (view.getValue())
        {
            final IcyBufferedImage cyan = IcyBufferedImageUtil.extractChannel(cmykImage, 0);
            cyan.setColorMap(0, new LinearColorMap("Cyan", Color.white, Color.cyan), true);
            
            final IcyBufferedImage magenta = IcyBufferedImageUtil.extractChannel(cmykImage, 1);
            magenta.setColorMap(0, new LinearColorMap("Magenta", Color.white, Color.magenta), true);
            
            final IcyBufferedImage yellow = IcyBufferedImageUtil.extractChannel(cmykImage, 2);
            yellow.setColorMap(0, new LinearColorMap("Yellow", Color.white, Color.yellow), true);
            
            final IcyBufferedImage black = IcyBufferedImageUtil.extractChannel(cmykImage, 3);
            black.setColorMap(0, new LinearColorMap("Black", Color.white, Color.black), true);
            
            ThreadUtil.invokeLater(new Runnable()
            {
                public void run()
                {
                    Sequence s = null;
                    
                    s = new Sequence("Cyan", cyan);
                    s.setChannelName(0, "Cyan");
                    new Viewer(s);
                    
                    s = new Sequence("Magenta", magenta);
                    s.setChannelName(0, "Magenta");
                    new Viewer(s);
                    
                    s = new Sequence("Yellow", yellow);
                    s.setChannelName(0, "Yellow");
                    new Viewer(s);
                    
                    s = new Sequence("Black", black);
                    s.setChannelName(0, "Black");
                    new Viewer(s);
                }
            });
        }
    }
    
    @Override
    public void clean()
    {
    
    }
}
