package plugins.fab.manualtnt;

import icy.canvas.IcyCanvas;
import icy.file.xls.XlsManager;
import icy.sequence.Sequence;
import icy.util.XMLUtil;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;

import jxl.demo.XML;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

import com.sun.org.apache.xerces.internal.dom.ElementDefinitionImpl;
import com.sun.org.apache.xerces.internal.impl.xs.opti.DefaultNode;

import ome.xml.model.Ellipse;

public class TNTAnnotation {

	private ArrayList<Cell> cellList = new ArrayList<Cell>();
	private ArrayList<TNT> tntList = new ArrayList<TNT>();	

	/** Persistent node */
	Node node ;
	File xlsFile ;
	double scale;
	
	public TNTAnnotation(Node node , File xlsFile , double scale ) {
		this.node = node;
		this.xlsFile = xlsFile;
		this.scale = scale;
	}

	public ArrayList<Cell> getCellList()
	{
		return new ArrayList<Cell> ( cellList );
	}

	public ArrayList<TNT> getTNTList()
	{
		return new ArrayList<TNT> ( tntList );
	}
	
	public void addCell( Cell cell )
	{
		cellList.add( cell );
		savePersistentData( );
	}

	public void addTNT( TNT tnt )
	{
		tntList.add( tnt );
		savePersistentData( );
	}
	
	public void removeTNT( TNT tnt )
	{
		tntList.remove( tnt );
		savePersistentData( );
	}

	public void removeCell( Cell cell )
	{
		cellList.remove( cell );
		savePersistentData( );
	}

	public void paint(Graphics2D g, Sequence sequence, IcyCanvas canvas) {

		g.setFont( new Font ( "Arial", Font.BOLD , 14 ) );
		
		for ( TNT tnt : tntList )
		{
			Line2D line = new Line2D.Double( tnt.a , tnt.b );
			
			g.setColor( Color.black );
			g.setStroke( new BasicStroke( 5 ) );
			g.draw( line );
			
			g.setColor( Color.green );
			g.setStroke( new BasicStroke( 4 ) );
			g.draw( line );
			
			// draw cell connection
			
			g.setColor( Color.orange );
			final float dash1[] = {10.0f};
		    final BasicStroke dashed =
		        new BasicStroke(1.0f,
		                        BasicStroke.CAP_ROUND,
		                        BasicStroke.JOIN_BEVEL,
		                        10.0f, dash1, 0.0f);
		    g.setStroke( dashed );
			{
				Cell cell = getClosestCell( tnt.a );
				if ( cell !=null )
				{				
					Line2D line2 = new Line2D.Double( tnt.a , cell.massCenter );
					g.draw( line2 );
				}
			}
			
			{
				Cell cell = getClosestCell( tnt.b );
				if ( cell !=null )
				{				
					Line2D line2 = new Line2D.Double( tnt.b, cell.massCenter );
					g.draw( line2 );
				}
			}
		}
		
		g.setStroke( new BasicStroke( 1 ) );
		
		for ( Cell cell : cellList )
		{
			Ellipse2D ellipse = new Ellipse2D.Double( cell.massCenter.getX()-20, cell.massCenter.getY()-20, 41, 41 );
			Ellipse2D ellipseOut = new Ellipse2D.Double( cell.massCenter.getX()-22, cell.massCenter.getY()-22, 45, 45 );
			
			g.setColor( Color.black );
			g.fill( ellipseOut );
			
			Color color = Color.green;
			
			switch ( cell.cellClass ) {
			case 0:
				color = Color.green;
				break;
			case 1:
				color = Color.red;
				break;
			case 2:
				color = Color.blue;
				break;
			case 3:
				color = Color.yellow;
				break;

			}
			
			g.setColor( color );
			g.fill( ellipse );
			
			// draw number
			
			int index = cellList.indexOf( cell );
			g.setColor( Color.black );
			g.drawString("" + index , (int) cell.massCenter.getX(), (int)cell.massCenter.getY() );
		}

		
		
	}
	
	public Cell getClosestCell( Point2D point )
	{
		Cell bestCell = null;
		double bestDistance = Double.MAX_VALUE;
		
		for ( Cell cell: cellList )
		{
			double distance = cell.massCenter.distance( point );
			if ( distance < bestDistance )
			{
				bestDistance = distance;
				bestCell = cell;
			}
		}
		
		return bestCell;
		
	}

	public int getNumberOfCellsClass( int index ) {

		int nb = 0;
		for( Cell cell: cellList)
		{
			if ( cell.cellClass == index ) nb++;
		}
		return nb;
		
	}

	public void loadPersistentData( ) {

		// load cell node
		Element cellNode = XMLUtil.getElement( node , "cell" );
		if ( cellNode!= null )
		{
			// load cells
			ArrayList<Element> cellElementList = XMLUtil.getElements(cellNode);

			for ( Element node : cellElementList )
			{

				Cell cell = new Cell( new Point2D.Double(
						XMLUtil.getAttributeDoubleValue( node, "x", 0 ),
						XMLUtil.getAttributeDoubleValue( node, "y", 0 )
						) );
				cell.cellClass = XMLUtil.getAttributeIntValue( node, "class", 0 );
				cellList.add( cell );
			}
		}		
		// load tnt node
		Element tntNode = XMLUtil.getElement( node , "tnt" );
		if ( tntNode != null )
		{
			// load tnts
			ArrayList<Element> tntElementList = XMLUtil.getElements(tntNode);

			for ( Element node : tntElementList )
			{
				TNT tnt = new TNT( new Point2D.Double(
						XMLUtil.getAttributeDoubleValue( node, "x1", 0 ),
						XMLUtil.getAttributeDoubleValue( node, "y1", 0 )
						),
						new Point2D.Double(
								XMLUtil.getAttributeDoubleValue( node, "x2", 0 ),
								XMLUtil.getAttributeDoubleValue( node, "y2", 0 )
								) 
						);			
				tntList.add( tnt );
			}
		}

	}

	public void savePersistentData( ) {
		
		// destroy all
		XMLUtil.removeAllChildren( node );
		
		// create Cell node
		Element cellNode = XMLUtil.addElement( node , "cell" );
		
		// fill cell node
		for ( Cell cell: cellList )
		{
			Element cellElement = XMLUtil.addElement( cellNode , "cell");			
			XMLUtil.setAttributeDoubleValue( cellElement , "x", cell.massCenter.getX() );
			XMLUtil.setAttributeDoubleValue( cellElement , "y", cell.massCenter.getY() );
			XMLUtil.setAttributeIntValue( cellElement , "class", cell.cellClass );
		}
		
		// Create TNT node
		Element tntNode = XMLUtil.addElement( node , "tnt" );
//		cellNode.setTextContent("tnt");
		XMLUtil.addNode( node , tntNode );
		
		// fill tnt node
		for ( TNT tnt: tntList )
		{
			Element tntElement = XMLUtil.addElement( tntNode , "tnt");			
			XMLUtil.setAttributeDoubleValue( tntElement , "x1", tnt.a.getX() );
			XMLUtil.setAttributeDoubleValue( tntElement , "y1", tnt.a.getY() );
			XMLUtil.setAttributeDoubleValue( tntElement , "x2", tnt.b.getX() );
			XMLUtil.setAttributeDoubleValue( tntElement , "y2", tnt.b.getY() );			
		}
				
	}

	public int getNumberOfTNTConnectedToCell( Cell cell )
	{
		int nb = 0;
		
		for ( TNT tnt : tntList )
		{
			if ( getClosestCell( tnt.a ) == cell || getClosestCell( tnt.b ) == cell )
			{
				nb++;
			}
		}
		
		return nb;
	}
	
	public void saveXLS() {

		XlsManager xls;
		try {
			
			xls = new XlsManager( xlsFile );
			
			xls.createNewPage("results");
			
			int y = 0;
			xls.setLabel( 0 , y, "Number of cell:");
			xls.setNumber( 1 , y, cellList.size() );
			y++;
			y++;
			
			for ( int i = 0 ; i< 4 ; i++ )
			{
				
				xls.setLabel( 0 , y, "Number of cell class "+i+" :");
				xls.setNumber( 1 , y, getNumberOfCellsClass( i ) );
				y++;
			}
			
			y++;
			
			// number of TNT for each class
			
			for ( int i = 0 ; i< 4 ; i++ )
			{
				xls.setLabel( 0 , y, "Number of TNT forming cell of class "+i+" :");
				
				int nbTNT = 0;
				for ( Cell cell : getCellList() )
				{
					if ( cell.cellClass == i )
					{
						// look if a tnt is link to this cell
						for ( TNT tnt : getTNTList() )
						{
							if ( getClosestCell( tnt.a ) == cell || getClosestCell( tnt.b ) == cell )
							{
								nbTNT++;
								break;
							}							
						}
					}
				}
				
				xls.setNumber( 1 , y, nbTNT );
				y++;
			}
			
			y++;
			
			for ( int i = 0 ; i < 4 ; i++ )
			{
				xls.setLabel( 0 , y, "Average TNT connexion for cell class "+i+" :");
				double averageTNTConnexionForClass = getAverageTNTConnexionForClass( i );
				xls.setNumber( 1 , y, averageTNTConnexionForClass );
				y++;
			}			
			
			y++;
			
			// number of connexion between cell class.
			xls.setLabel( 0, y, "Number of connexion between cell class" );
			
			y++;
			
			{
				int result[][] = getNumberOfConnexionBetweenClass();
				for ( int xx = 0 ; xx < 4 ; xx++ )
				{
					xls.setLabel( xx+1, y, ""+xx );
				}
				for ( int yy = 0 ; yy < 4 ; yy++ )
				{
					xls.setLabel( 0, y+yy+1, ""+ yy );
				}

				for ( int xx = 0 ; xx < 4 ; xx++ )
					for ( int yy = 0 ; yy < 4 ; yy++ )
					{
						if ( xx <= yy )
						{
							int r = result[xx][yy];
							if ( r!=0 )
							{
								xls.setNumber( xx+1, yy+ y+1 , r );
							}
						}
					}			
			}
			y+=6;
			
			// save number of link between each single cell
			{
				xls.createNewPage("Cross link");
				y=0;

				// create labels
				
				for ( int xx = 0 ; xx < cellList.size() ; xx++ )
				{
					xls.setLabel( xx+1, y, ""+xx + "/" + cellList.get( xx ).cellClass  );
				}

				for ( int yy = 0 ; yy < cellList.size() ; yy++ )
				{
					xls.setLabel( 0, y+yy+1, ""+yy + "/" + cellList.get( yy ).cellClass  );
				}
				
				// create results
				
				int result[][] = getNumberOfConnexionBetweenCell();
				xls.setLabel( 0, 0, "cell#/class#" );

				for ( int xx = 0 ; xx < result.length ; xx++ )
					for ( int yy = 0 ; yy < result[xx].length ; yy++ )
					{
						if ( xx <= yy )
						{				
							int r = result[xx][yy];
							if ( r != 0 )
							{
								xls.setNumber( xx+1, yy+ y+1 , r );
							}
						}
					}
				
			}
			
			// tnt length
			
			{
				y=0;
				xls.createNewPage("TNT length");

				xls.setLabel( 0, y, "TNT length (px)" );
				xls.setLabel( 1, y, "TNT length (m)" );
				xls.setLabel( 2, y, "cell A#" );
				xls.setLabel( 3, y, "cell A class#" );
				xls.setLabel( 4, y, "cell B#" );
				xls.setLabel( 5, y, "cell B class#" );
				
				y++;
				
				for ( TNT tnt : getTNTList() )
				{
					Cell cellA = getClosestCell( tnt.a );
					Cell cellB = getClosestCell( tnt.b );
					double distancePix = tnt.a.distance( tnt.b );
					xls.setNumber( 0, y, distancePix );
					
					double distanceScaled = distancePix * scale;
					xls.setNumber( 1, y, distanceScaled );
					
					xls.setNumber( 2, y, getCellIndex( cellA ) );
					xls.setNumber( 3, y, cellA.cellClass );
					
					xls.setNumber( 4, y, getCellIndex( cellB ) );
					xls.setNumber( 5, y, cellB.cellClass );
					y++;
				}
			}
			
			
			
			
			xls.SaveAndClose();
			
		} catch (IOException e) {
			e.printStackTrace();
		}
		
		
		
	}

	public int getCellIndex( Cell cell )
	{
		return cellList.indexOf( cell );
	}
	
	private int[][] getNumberOfConnexionBetweenCell() {

		int[][] connexion = new int[cellList.size()][cellList.size()];
		
		for ( TNT tnt : tntList )
		{
			Cell cellA = getClosestCell( tnt.a );
			Cell cellB = getClosestCell( tnt.b );
			
			int indexX = getCellIndex( cellA );
			int indexY = getCellIndex( cellB );
			
			if ( indexX < indexY )
			{
				connexion[indexX][indexY]++;
			}else
			{
				connexion[indexY][indexX]++;
			}
			
		}
		
		return connexion;

	}

	private int[][] getNumberOfConnexionBetweenClass() {

		int[][] connexion = new int[4][4];
		
		for ( TNT tnt : tntList )
		{
			Cell cellA = getClosestCell( tnt.a );
			Cell cellB = getClosestCell( tnt.b );
			
			if ( cellA.cellClass > cellB.cellClass )
			{
				connexion[cellB.cellClass][cellA.cellClass]++;
			}else
			{
				connexion[cellA.cellClass][cellB.cellClass]++;
			}
			
		}
		
		return connexion;
		
	}

	private double getAverageTNTConnexionForClass(int i) {

		float nbCell = 0;
		float nbTNT = 0;
		for ( Cell cell : cellList )
		{
			if ( cell.cellClass == i )
			{
				nbCell++;
				nbTNT+= getNumberOfTNTConnectedToCell( cell );
			}
		}
	
		if ( nbCell != 0 )
		{
			return nbTNT / nbCell;
		}
		return 0;
	}
	
}
