"""
Jython proxy classes.

Copyright 2013 Institut Pasteur.
 
This file is part of ICY.
 
ICY is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.
 
ICY is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
 
You should have received a copy of the GNU General Public License
along with ICY. If not, see <http://www.gnu.org/licenses/>.

Created on Tue Nov 14

@author: tlecomte
Inspired from https://github.com/jimbaker/jython-storm-starter/blob/master/clamp/__init__.py (for serialization and package name),
and from the MiniClampMaker.java test in Jython (for Java annotations).
"""

import java
import os
import os.path

from java.lang.reflect import Modifier
from java.lang import Class
from org.python.util import CodegenUtils
from org.python.compiler import CustomMaker, ProxyCodeHelpers
from org.python.util import Generic
from org.python.compiler.ProxyCodeHelpers import AnnotationDescr
from org.python.core import Py

__all__ = ["PackageProxy", "SerializableProxies"]


def extractAnnotation(pyAnnotations):
    """
    Return the list of AnnotationDesc 
    """
    annotationDescrs = []
    for annotationClass, annotationFields in pyAnnotations.iteritems():
        if annotationFields == None:
            annotationDescr = AnnotationDescr(annotationClass)
        else:
            annotationDescr = AnnotationDescr(annotationClass, annotationFields)
        annotationDescrs.append(annotationDescr)

    return annotationDescrs


class SerializableProxies(CustomMaker):

    """
    SerializableProxies is a custom java proxy.
    It exports the class file to a specific directory, and knows how to handle class annotations
    that are marked with __clamp_class_annotations__ in the class definition.
    """

    serialized_path = None
    
    def __init__(self, superclass, interfaces, pythonClass, pythonModule, myClass, methods):
        super(CustomMaker, self).__init__(superclass, interfaces, pythonClass, pythonModule, myClass, methods)

        try:
            pyAnnotations = methods["__clamp_class_annotations__"]
            if pyAnnotations != None:
                self.classAnnotations = extractAnnotation(pyAnnotations)
        except KeyError:
            self.classAnnotations = []

    def doConstants(self):
        self.classfile.addField("serialVersionUID",
                                CodegenUtils.ci(java.lang.Long.TYPE), Modifier.PUBLIC | Modifier.STATIC | Modifier.FINAL)
        code = self.classfile.addMethod("<clinit>", ProxyCodeHelpers.makeSig("V"), Modifier.STATIC)
        code.visitLdcInsn(java.lang.Long(1))
        code.putstatic(self.classfile.name, "serialVersionUID", CodegenUtils.ci(java.lang.Long.TYPE))
        code.return_()

    def saveBytes(self, bytes):
        if self.serialized_path:
            path = os.path.join(self.serialized_path, os.path.join(*self.myClass.split(".")) + ".class")
            parent = os.path.dirname(path)
            try:
                os.makedirs(parent)
            except OSError:
                pass  # Directory exists
            with open(path, "wb") as f:
                f.write(bytes.toByteArray())

    def makeClass(self):
        try:
            # If already defined on CLASSPATH, simply return this class
            cls = java.lang.Class.forName(self.myClass)
            print "Class defined on CLASSPATH", cls
        except:
            # Otherwise build it
            cls = CustomMaker.makeClass(self)
        return cls

    def visitClassAnnotations(self):
        for annotation in self.classAnnotations:
            self.addClassAnnotation(annotation)


class PackageProxy(object):
    """
    PackageProxy is a wrapper around SerializableProxies, that add the handling of
    Java package name.
    """
    def __init__(self, package):
        self.package = package

    def __call__(self, superclass, interfaces, className, pythonModuleName, fullProxyName, mapping):
        """Constructs a usable proxy name that does not depend on ordering"""
        if "." in pythonModuleName:
            # get around that will be called differently from regrtest, as test.module instead of module
            pythonModuleName = pythonModuleName.split(".")[-1]
        return SerializableProxies(superclass, interfaces, className, pythonModuleName, self.package + "." + pythonModuleName + "." + className, mapping)
