package plugins.jrogers.blocks;

/**
 * Plugin which takes a workbook input of co-ordinates (or alternatively a VarROIArray),
 * and an input ROI array, and produces a workbook. This workbook lists the points
 * in a sheet named ROI1/2/3/etc. in case multiple ROIs are present in a sequence.
 * The sequence name along with the co-ordinates of the large ROI will be printed
 * as a header. 
 * 
 * @author Joel Rogers
 */

import icy.file.FileUtil;
import icy.plugin.abstract_.Plugin;
import icy.roi.BooleanMask2D;
import icy.roi.BooleanMask3D;
import icy.roi.ROI;
import icy.roi.ROI2D;
import icy.roi.ROI3D;
import icy.type.point.Point3D;
import icy.type.point.Point5D;
import icy.type.rectangle.Rectangle5D;
import icy.sequence.Sequence;

import java.awt.geom.Point2D;
import java.util.List;

import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.WorkbookUtil;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;

import plugins.adufour.blocks.util.VarList;
import plugins.adufour.vars.lang.VarROIArray;
import plugins.adufour.vars.lang.VarInteger;
import plugins.adufour.vars.lang.VarWorkbook;
import plugins.adufour.vars.lang.VarSequence;
import plugins.adufour.vars.lang.VarEnum;
import plugins.adufour.vars.util.VarException;

public class PointInROI extends Plugin implements ROIBlock
{
	protected enum InputType
    {
        WORKBOOK("Workbook"), ROIS("Regions of Interest");
        
        private final String displayText;
        
        private InputType(String displayText)
        {
            this.displayText = displayText;
        }
        
        @Override
        public String toString()
        {
            return displayText;
        }
    }
    VarROIArray roiIN   = new VarROIArray("Large ROIs");
    VarROIArray queryROI = new VarROIArray("Query ROIs");
    VarWorkbook pointsIn = new VarWorkbook("List of points", (Workbook) null);
    final VarSequence sequence = new VarSequence("Sequence", null);
    VarEnum<InputType> IType = new VarEnum<InputType>("Input Type", InputType.ROIS);
    VarInteger h = new VarInteger("Header Rows", 1);
    
    VarWorkbook pointsOut = new VarWorkbook("Workbook", "Sheet 1");
       
    @Override
    public void declareInput(VarList inputMap)
    {
    	pointsIn.setDefaultEditorModel(null);
        inputMap.add("List of points", pointsIn);
        inputMap.add("ROIs to class", queryROI);
        inputMap.add("Large ROIs", roiIN);
        inputMap.add("Sequence", sequence);
        inputMap.add("Input Type", IType);
        inputMap.add("Number of Header Rows (Workbook only)", h);
    }
    
    @Override
    public void declareOutput(VarList outputMap)
    {
        pointsOut.setDefaultEditorModel(null);
        outputMap.add("Points by ROI", pointsOut);
    }
        
    @Override
    public void run()
    { 	 
    	switch (IType.getValue())
        {
        case WORKBOOK:
        	pointsOut.setValue(pointsInROI(roiIN, sequence.getValue(), pointsIn, h));
            break;
        case ROIS:
        	pointsOut.setValue(pointsInROI(roiIN, sequence.getValue(), queryROI));
            break;
        default:
            throw new VarException(IType, "Unsupported Input");
        }
    }
    	
    
    
    /**
     * Assigns each point in the input workbook to one (or more) of the input ROIs
     * 
     * @param inROI
     *  		the array of ROIs to probe
     * @param sequence
     *  		the input sequence associated with the ROIs
     * @param points
     *  		input workbook
     * @param headR
     *  		the number of header rows in the input workbook
     * @returns wb
     *  		a workbook containing the points correctly assigned to ROIs
     */
    private static Workbook pointsInROI(VarROIArray inROI, Sequence s, VarWorkbook pointsIN, VarInteger headR)
    {
    	ROI[] inputROI = inROI.getValue();
    	//Initialise the (output) workbook
        HSSFWorkbook wb = new HSSFWorkbook();
        wb.setMissingCellPolicy(Row.CREATE_NULL_AS_BLANK);
        for (int sheetNum=0;sheetNum<inputROI.length;++sheetNum) {
        		String sN = String.valueOf(sheetNum + 1);
        		wb.createSheet("ROI" + sN);
        }
        //Get the number of input points and dimensions of input points
        Workbook points = pointsIN.getValue();
        Sheet sheet = points.getSheetAt(0);
        int num = sheet.getPhysicalNumberOfRows();
        Row row = sheet.getRow(headR.getValue());
        String hR = String.valueOf(headR.getValue() + 1);
        int dim = row.getPhysicalNumberOfCells();
        
        if (dim != 3 && dim != 2) {
        	System.out.println("Invalid workbook format. X, Y, (Z) expected at row" + hR + ". Check number of input header rows?");
        }
        
        HSSFSheet sh;
    	for (int ROINum=0;ROINum<inputROI.length;ROINum++) {
    		ROI roi = inputROI[ROINum];
    		Point5D p = getMassCenter(roi);
    		sh = (HSSFSheet)wb.getSheetAt(ROINum);
    		Row head = sh.createRow(0);
    		for (int i=0;i<3;i++) {
    			Cell cell = head.createCell(i);
    			if (i == 0) {
    				cell.setCellValue(getDataSetName(inROI, s));
    			}
    			if (i == 1) {
    				cell.setCellValue((double)p.getX());
    			}
    			if (i == 2) {
    				cell.setCellValue((double)p.getY());
    			}	
    		}
    		double X = 0;
    		double Y = 0;
    		double Z = 0;
    		int rowNum = 1;
    		if (dim == 3) {
    			Cell cell = head.createCell(3);
    			cell.setCellValue((double)p.getZ());
    			for(int r = headR.getValue(); r < num; r++) {
    		        row = sheet.getRow(r);
    		        if(row != null) {
    		            for(int c = 0; c < 3; c++) {
    		                cell = row.getCell((short)c);
    		                if(cell != null) {
    		                    if(c == 0) {
    		                    	X = (double)cell.getNumericCellValue();
    		                    }
    		                    if(c == 1) {
    		                    	Y = (double)cell.getNumericCellValue();
    		                    }
    		                    if(c == 2) {
    		                    	Z = (double)cell.getNumericCellValue();
    		                    }
    		                }
    		            }
    		            p.setLocation(X, Y, Z, p.getT(), p.getC());
    		            if (roi.contains(p)) {
            				row = sh.createRow(rowNum);
            				rowNum++;
            				for (int i=0;i<4;i++) {
            	    			cell = row.createCell(i);
            	    			if (i == 0) {
            	    				cell.setCellValue(r - 1);
            	    			}
            	    			if (i == 1) {
            	    				cell.setCellValue(X);
            	    			}
            	    			if (i == 2) {
            	    				cell.setCellValue(Y);
            	    			}
            	    			if (i == 3) {
            	    				cell.setCellValue(Z);
            	    			}
            				}
    		            }
    		        }
    		    }
    		}
    		if (dim == 2) {
    			Cell cell;
    			for(int r = headR.getValue(); r < num; r++) {
    		        row = sheet.getRow(r);
    		        if(row != null) {
    		            for(int c = 0; c < 2; c++) {
    		                cell = row.getCell((short)c);
    		                if(cell != null) {
    		                    if(c == 0) {
    		                    	X = (double)cell.getNumericCellValue();
    		                    }
    		                    if(c == 1) {
    		                    	Y = (double)cell.getNumericCellValue();
    		                    }
    		                }
    		            }
    		            p.setLocation(X, Y, p.getZ(), p.getT(), p.getC());
    		            if (roi.contains(p)) {
            				row = sh.createRow(rowNum);
            				rowNum++;
            				for (int i=0;i<3;i++) {
            	    			cell = row.createCell(i);
            	    			if (i == 0) {
            	    				cell.setCellValue(r - 1);
            	    			}
            	    			if (i == 1) {
            	    				cell.setCellValue(X);
            	    			}
            	    			if (i == 2) {
            	    				cell.setCellValue(Y);
            	    			}
            				}
    		            }
    		        }
    		    }
    		}
    	} return wb;
    }
    
    /**
     * Assigns each MassCenter in the query ROI array to one (or more) of the input ROIs
     * 
     * @param inROI
     *  		the array of ROIs to probe('headers')
     * @param sequence
     *  		the input sequence associated with the ROIs
     * @param queryROI
     *  		input ROIs to be classed according to 'inROI'
     * @returns wb
     *  		a workbook containing the points correctly assigned to ROIs
     */
    private static Workbook pointsInROI(VarROIArray inROI, Sequence s, VarROIArray queryROI)
    {
    	ROI[] inputROI = inROI.getValue();
    	ROI[] qROI = queryROI.getValue();
    	//Initialise the workbook (output)
        HSSFWorkbook wb = new HSSFWorkbook();
        wb.setMissingCellPolicy(Row.CREATE_NULL_AS_BLANK);
        for (int sheetNum=0;sheetNum<inputROI.length;++sheetNum) {
        		String sN = String.valueOf(sheetNum + 1);
        		wb.createSheet("ROI" + sN);
        }
        
        HSSFSheet sh;
    	for (int ROINum=0;ROINum<inputROI.length;ROINum++) {
    		ROI roi = inputROI[ROINum];
    		Point5D p = getMassCenter(roi);
    		sh = (HSSFSheet)wb.getSheetAt(ROINum);
    		Row head = sh.createRow(0);
    		for (int i=0;i<3;i++) {
    			Cell cell = head.createCell(i);
    			if (i == 0) {
    				cell.setCellValue(getDataSetName(inROI, s));
    			}
    			if (i == 1) {
    				cell.setCellValue((double)p.getX());
    			}
    			if (i == 2) {
    				cell.setCellValue((double)p.getY());
    			}	
    		}
    		int rowNum = 1;
    		Row row;
    		if (qROI[0] instanceof ROI3D) {
    			Cell cell = head.createCell(3);
    			cell.setCellValue((double)p.getZ());
    			for(int q=0;q<qROI.length;q++) {
    				ROI qR = qROI[q];
    		        p = getMassCenter(qR);
    		        if (roi.contains(p)) {
            			row = sh.createRow(rowNum);
            			rowNum++;
            			for (int i=0;i<4;i++) {
            	    		cell = row.createCell(i);
            	    		if (i == 0) {
            	    			cell.setCellValue(q + 1);
            	    		}
            	    		if (i == 1) {
            	    			cell.setCellValue(p.getX());
            	    		}
            	    		if (i == 2) {
            	    			cell.setCellValue(p.getY());
            	    		}
            	    		if (i == 3) {
            	    			cell.setCellValue(p.getZ());
            	    		}
            			}
    		        }
    			}
    		}
    		if (qROI[0] instanceof ROI2D) {
    			Cell cell;
    			for(int q=0;q<qROI.length;q++) {
    				ROI qR = qROI[q];
    		        p = getMassCenter(qR);
    		        if (roi.contains(p)) {
            			row = sh.createRow(rowNum);
            			rowNum++;
            			for (int i=0;i<3;i++) {
            	    		cell = row.createCell(i);
            	    		if (i == 0) {
            	    			cell.setCellValue(q + 1);
            	    		}
            	    		if (i == 1) {
            	    			cell.setCellValue(p.getX());
            	    		}
            	    		if (i == 2) {
            	    			cell.setCellValue(p.getY());
            	    		}
            			}
    		        }
    		    }
		    }
    	} return wb;
    }
    
    /**
     * Utility Methods
     * @param rois
     * @param sequence
     * 
     * @author Alexandre Dufour
     */
    
    private static String getDataSetName(VarROIArray rois, Sequence sequence)
    {
        String dataSetName = "Points in ROI";
        
        if (sequence == null)
        {
            // try retrieving the sequence attached to the first ROI
            List<Sequence> sequences = rois.getValue()[0].getSequences();
            if (sequences.size() > 0) sequence = sequences.get(0);
        }
        
        if (sequence == null)
        {
            // no hope...
            dataSetName = "--";
        }
        else
        {
            // replace the sheet name by the file or sequence name
            dataSetName = FileUtil.getFileName(sequence.getFilename());
            if (dataSetName.isEmpty()) dataSetName = sequence.getName();
        }
        
        // make the name "safe"
        return WorkbookUtil.createSafeSheetName(dataSetName);
    }
    
    public static Point5D getMassCenter(ROI roi)
    {
        if (roi instanceof ROI2D)
        {
            ROI2D r2 = (ROI2D) roi;
            Point2D center = getMassCenter(r2);
            return new Point5D.Double(center.getX(), center.getY(), r2.getZ(), r2.getT(), r2.getC());
        }
        else if (roi instanceof ROI3D)
        {
            ROI3D r3 = (ROI3D) roi;
            Point3D center = getMassCenter(r3);
            return new Point5D.Double(center.getX(), center.getY(), center.getZ(), r3.getT(), r3.getC());
        }
        
        else
        {
            Rectangle5D b5 = roi.getBounds5D();
            return new Point5D.Double(b5.getCenterX(), b5.getCenterY(), b5.getCenterZ(), b5.getCenterT(), b5.getCenterC());
        }
    }
    
    public static Point2D getMassCenter(ROI2D roi)
    {
        double x = 0, y = 0;
        
        int cpt = 0;
        BooleanMask2D mask = ((ROI2D) roi).getBooleanMask(true);
        for (int j = 0, off = 0; j < mask.bounds.height; j++)
            for (int i = 0; i < mask.bounds.width; i++, off++)
                if (mask.mask[off])
                {
                    cpt++;
                    x += i;
                    y += j;
                }
        return new Point2D.Double(mask.bounds.getX() + (x / cpt), mask.bounds.getY() + (y / cpt));
    }
    
    public static Point3D getMassCenter(ROI3D roi)
    {
        double x = 0, y = 0, z = 0;
        
        int cpt = 0;
        BooleanMask3D mask = ((ROI3D) roi).getBooleanMask(true);
        for (int zSlice : mask.mask.keySet())
        {
            BooleanMask2D mask2 = mask.getMask2D(zSlice);
            for (int j = 0, off = 0; j < mask2.bounds.height; j++)
                for (int i = 0; i < mask2.bounds.width; i++, off++)
                    if (mask2.mask[off])
                    {
                        cpt++;
                        x += i;
                        y += j;
                        z += zSlice;
                    }
        }
        return new Point3D.Double(mask.bounds.getX() + (x / cpt), mask.bounds.getY() + (y / cpt), mask.bounds.getZ() + (z / cpt));
    }
}

