package plugins.adufour.hcs;

import java.awt.BasicStroke;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GridLayout;
import java.awt.RenderingHints;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.geom.Ellipse2D;
import java.awt.image.RenderedImage;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.TreeMap;

import javax.media.jai.RenderedImageAdapter;
import javax.swing.JComponent;
import javax.swing.JPanel;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

import com.drew.imaging.tiff.TiffMetadataReader;
import com.drew.metadata.Directory;
import com.drew.metadata.Metadata;
import com.sun.media.jai.codec.ImageDecoder;
import com.sun.media.jai.codec.SeekableStream;
import com.sun.media.jai.codec.TIFFDecodeParam;
import com.sun.media.jai.codecimpl.TIFFCodec;

import icy.canvas.IcyCanvas;
import icy.file.FileUtil;
import icy.file.Loader;
import icy.gui.frame.IcyFrame;
import icy.image.IcyBufferedImage;
import icy.image.colormap.IcyColorMap;
import icy.image.colormap.LinearColorMap;
import icy.main.Icy;
import icy.painter.Overlay;
import icy.sequence.Sequence;
import icy.system.IcyHandledException;
import icy.system.thread.ThreadUtil;
import icy.util.OMEUtil;
import icy.util.StringUtil;
import icy.util.XMLUtil;
import loci.formats.ome.OMEXMLMetadataImpl;
import ome.units.quantity.Length;
import ome.units.unit.Unit;
import plugins.adufour.ezplug.EzLabel;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzStoppable;
import plugins.adufour.ezplug.EzVar;
import plugins.adufour.ezplug.EzVarBoolean;
import plugins.adufour.ezplug.EzVarFolder;
import plugins.adufour.ezplug.EzVarInteger;
import plugins.adufour.ezplug.EzVarListener;

public class OperaImporter extends EzPlug implements EzStoppable
{
    //// PluginFileImporter stuff
    // @Override
    // public boolean acceptFile(String path)
    // {
    // return FileUtil.isDirectory(path) && path.contains("Meas");
    // }
    //
    // @Override
    // public List<FileFilter> getFileFilters()
    // {
    // FileFilter filter = new FileFilter()
    // {
    // @Override
    // public String getDescription()
    // {
    // return "Opera plate";
    // }
    //
    // @Override
    // public boolean accept(File f)
    // {
    // return acceptFile(f.getPath());
    // }
    // };
    //
    // return Arrays.asList(filter);
    // }
    //
    // @Override
    // public boolean load(String path, FileFrame loadingFrame) throws UnsupportedFormatException,
    // IOException
    // {
    //
    // return true;
    // }
    
    Sequence sequence;
    Overlay  labelOverlay;
    
    @SuppressWarnings("serial")
    class Well extends JComponent
    {
        private float                         wellDiameter = 30;
        
        private final Ellipse2D               wellShape    = new Ellipse2D.Float();
        
        private final int                     row, col;
        
        private final TreeMap<String, String> paths        = new TreeMap<String, String>();
        
        public Well(int row, int col, float factor)
        {
            this.row = row;
            this.col = col;
            
            int displayDiameter = Math.round(wellDiameter * factor);
            
            setPreferredSize(new Dimension(displayDiameter, displayDiameter));
            
            addMouseListener(new MouseAdapter()
            {
                @Override
                public void mouseClicked(MouseEvent event)
                {
                    if (!wellShape.contains(event.getPoint()))
                    {
                        event.consume();
                    }
                    else
                    {
                        super.mouseClicked(event);
                        ThreadUtil.bgRun(wellLoader);
                    }
                }
            });
        }
        
        private final Runnable wellLoader = new Runnable()
        {
            Unit<Length> wavelength = Unit.CreateBaseUnit("nanometers", "lambda");
            
            @Override
            public void run()
            {
                getStatus().setCompletion(Double.NaN);
                
                if (paths.isEmpty())
                {
                    getUI().setProgressBarMessage("No image in well [" + row + "," + col + "].");
                    return;
                }
                
                try
                {
                    getUI().setProgressBarMessage("Loading well [" + row + "," + col + "]...");
                    
                    char alphaRow = 'A';
                    alphaRow += (row - 1);
                    sequence.setName(alphaRow + StringUtil.toString(col, 2));
                    
                    // Re-build the overlay
                    if (labelOverlay != null) sequence.removeOverlay(labelOverlay);
                    final List<String> labels = new ArrayList<String>();
                    sequence.addOverlay(labelOverlay = new Overlay("Plate layout")
                    {
                        @Override
                        public void paint(Graphics2D g, Sequence sequence, IcyCanvas canvas)
                        {
                            Graphics2D g2 = (Graphics2D) g.create();
                            g2.setColor(Color.yellow);
                            float textF = (float) canvas.canvasToImageLogDeltaX(20);
                            g2.setFont(g2.getFont().deriveFont(textF));
                            try
                            {
                                String label = labels.get(canvas.getPositionT());
                                g2.drawString(label, 10, sequence.getSizeY() - 10);
                            }
                            catch (Exception e)
                            {
                                
                            }
                        }
                    });
                    
                    int fieldCounter = 0;
                    for (String path : paths.keySet())
                    {
                        if (Thread.currentThread().isInterrupted()) break;
                        
                        try
                        {
                            labels.add(paths.get(path));
                            
                            if (path.endsWith(".tif"))
                            {
                                // This data might have been exported (by IM)
                                // => much of the meta-data is now gone...
                                
                                Sequence _seq = Loader.loadSequence(path, 0, false);
                                ArrayList<IcyBufferedImage> channels = new ArrayList<IcyBufferedImage>(_seq.getSizeT());
                                for (int c = 0; c < _seq.getSizeT(); c++)
                                    channels.add(_seq.getImage(c, 0));
                                
                                IcyBufferedImage image = IcyBufferedImage.createFrom(channels);
                                sequence.setImage(fieldCounter, 0, image);
                            }
                            else
                            {
                                // Bypass BioFormats for faster access (!!)
                                SeekableStream ss = SeekableStream.wrapInputStream(new FileInputStream(path), true);
                                ImageDecoder dec = TIFFCodec.createImageDecoder("tiff", ss, new TIFFDecodeParam());
                                
                                Metadata mmd = TiffMetadataReader.readMetadata(new File(path));
                                Directory dir = mmd.getDirectories().iterator().next();
                                // The first tag is the XML description of the entire plate
                                // (sweet!!)
                                Document flex = XMLUtil.createDocument(dir.getTags().iterator().next().getDescription());
                                Element _root = XMLUtil.getElement(flex, "Root");
                                Element _flex = XMLUtil.getElement(_root, "FLEX");
                                
                                // Store light sources
                                // TreeMap<String, Element> lasers = new TreeMap<String, Element>();
                                // Element _lasers = XMLUtil.getElement(_flex, "LightSources");
                                // for (Element _laser : XMLUtil.getElements(_lasers,
                                // "LightSource"))
                                // lasers.put(XMLUtil.getAttributeValue(_laser, "ID", ""), _laser);
                                
                                // Store light source combinations
                                // TreeMap<String, Element> lscs = new TreeMap<String, Element>();
                                // Element _lscs = XMLUtil.getElement(_flex,
                                // "LightSourceCombinations");
                                // for (Element _lsc : XMLUtil.getElements(_lscs,
                                // "LightSourceCombination"))
                                // lscs.put(XMLUtil.getAttributeValue(_lsc, "ID", ""), _lsc);
                                
                                // Store filter combinations
                                TreeMap<String, Element> fcs = new TreeMap<String, Element>();
                                Element _fcs = XMLUtil.getElement(_flex, "FilterCombinations");
                                for (Element _fc : XMLUtil.getElements(_fcs, "FilterCombination"))
                                {
                                    // Store the combination as the images reference them (ExpXCamY)
                                    String expID = XMLUtil.getAttributeValue(_fc, "ID", "");
                                    expID = "Exp" + expID.substring(expID.length() - 1);
                                    
                                    for (Element _sliderRef : XMLUtil.getElements(_fc, "SliderRef"))
                                    {
                                        String camID = XMLUtil.getAttributeValue(_sliderRef, "ID", "");
                                        // FIXME Anything other than "Camera..." is not supported
                                        if (!camID.startsWith("Camera")) continue;
                                        
                                        camID = "Cam" + camID.substring(camID.length() - 1);
                                        fcs.put(expID + camID, _sliderRef);
                                    }
                                }
                                
                                // Store Z-stacks
                                TreeMap<String, Element> stackRefs = new TreeMap<String, Element>();
                                Element _stacks = XMLUtil.getElement(_flex, "Stacks");
                                for (Element _stack : XMLUtil.getElements(_stacks, "Stack"))
                                    stackRefs.put(XMLUtil.getAttributeValue(_stack, "ID", ""), _stack);
                                
                                // Start reading the well contents
                                Element _well = XMLUtil.getElement(_flex, "Well");
                                Element _images = XMLUtil.getElement(_well, "Images");
                                
                                // Prepare the Z-stack
                                Element _stack = stackRefs.get(XMLUtil.getElementValue(_well, "StackRef", null));
                                int sizeZ = _stack.getChildNodes().getLength();
                                
                                // For each slice, read and compile the different channels
                                // NB: slice indexes start at 1
                                for (int z = 1; z <= sizeZ; z++)
                                {
                                    // Create a counter for the light source combination
                                    // This is necessary when a light source is used more than once
                                    // HashMap<Element, Integer> counter = new HashMap<Element,
                                    // Integer>();
                                    // for (Element _lsc : XMLUtil.getElements(_lscs,
                                    // "LightSourceCombination"))
                                    // counter.put(_lsc, 0);
                                    
                                    // Read the individual channels for the current slice
                                    ArrayList<IcyBufferedImage> channels = new ArrayList<IcyBufferedImage>();
                                    int currentChannel = 0;
                                    
                                    for (Element _image : XMLUtil.getElements(_images, "Image"))
                                    {
                                        int currentZ = XMLUtil.getElementIntValue(_image, "Stack", 0);
                                        if (z != currentZ) continue;
                                        
                                        // Read the image data
                                        int tiffPage = XMLUtil.getAttributeIntValue(_image, "BufferNo", 0);
                                        RenderedImage _plane = dec.decodeAsRenderedImage(tiffPage);
                                        channels.add(IcyBufferedImage.createFrom(new RenderedImageAdapter(_plane)));
                                        
                                        // Read the channel information (mostly wavelength)
                                        
                                        String cameraRef = XMLUtil.getElementValue(_image, "CameraRef", null);
                                        Element _fc = fcs.get(cameraRef);
                                        String filter = XMLUtil.getAttributeValue(_fc, "Filter", null);
                                        String[] filterValues = filter.split("/");
                                        int lambda = Integer.parseInt(filterValues[0]) - Integer.parseInt(filterValues[1]);
                                        
                                        // Element _lsc = lscs.get(XMLUtil.getElementValue(_image,
                                        // "LightSourceCombinationRef", null));
                                        // int index = counter.get(_lsc);
                                        // counter.put(_lsc, index + 1);
                                        // Element _lsr = XMLUtil.getElements(_lsc,
                                        // "LightSourceRef").get(index);
                                        // Element _laser =
                                        // lasers.get(XMLUtil.getAttributeValue(_lsr, "ID", null));
                                        // String lambda = XMLUtil.getElementValue(_laser,
                                        // "Wavelength", "ch " + currentChannel);
                                        
                                        metadata.setChannelName(filter, 0, currentChannel);
                                        metadata.setChannelExcitationWavelength(new Length(lambda, wavelength), 0, currentChannel);
                                        
                                        currentChannel++;
                                    }
                                    
                                    // Channels are ready to be merged
                                    IcyBufferedImage plane = IcyBufferedImage.createFrom(channels);
                                    // The plane can be added to the stack
                                    sequence.setImage(fieldCounter, z - 1, plane);
                                }
                                
                                sequence.endUpdate();
                                if (!Icy.getMainInterface().isOpened(sequence))
                                {
                                    // Set the color map (if available)
                                    try
                                    {
                                        for (int c = 0; c < sequence.getSizeC(); c++)
                                        {
                                            int excitation = metadata.getChannelExcitationWavelength(0, c).value().intValue();
                                            // Apply a (very) arbitrary Stokes shift of 50nm
                                            int emission = excitation + 50;
                                            Color rgb = getColorFromWavelength(emission);
                                            IcyColorMap colorMap = new LinearColorMap(excitation + "nm", rgb);
                                            sequence.setColormap(c, colorMap, true);
                                        }
                                    }
                                    catch (Exception npe)
                                    {
                                        // forget it...
                                    }
                                    
                                    // Show the viewer
                                    addSequence(sequence);
                                    // Wait for the viewer to really be visible
                                    // (otherwise multiple viewers may open)
                                    while (sequence.getFirstViewer() == null);
                                    
                                }
                                sequence.setMetaData(metadata);
                                
                                fieldCounter++;
                            }
                        }
                        catch (Exception e1)
                        {
                            e1.printStackTrace();
                        }
                    }
                    
                }
                finally
                {
                    getStatus().done();
                }
            }
        };
        
        @Override
        protected void paintComponent(Graphics g)
        {
            super.paintComponent(g);
            
            Graphics2D g2 = (Graphics2D) g.create();
            
            g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
            
            if (!paths.isEmpty())
            {
                g2.setColor(Color.gray);
                g2.fill(wellShape);
            }
            
            g2.setColor(Color.black);
            g2.setStroke(new BasicStroke(2f));
            g2.draw(wellShape);
        }
        
        @Override
        public void setBounds(int x, int y, int w, int h)
        {
            wellShape.setFrame(1, 1, w - 3, h - 3);
            super.setBounds(x, y, w, h);
        }
        
        public void addImage(int pos, String path)
        {
            String imageID = "Row " + row + ", Col " + col + " #" + pos;
            if (showAlphanum.getValue())
            {
                char rowChar = 'A';
                rowChar += (row - 1);
                imageID = "Well " + rowChar + StringUtil.toString(col, 2) + " #" + pos;
            }
            
            paths.put(path, imageID);
        }
    }
    
    @SuppressWarnings("serial")
    class WellViewer extends JComponent
    {
        public WellViewer()
        {
            setPreferredSize(new Dimension(100, 100));
        }
        
        @Override
        protected void paintComponent(Graphics g)
        {
            super.paintComponent(g);
            
            Graphics2D g2 = (Graphics2D) g.create();
            
            g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
            g2.setColor(Color.black);
            g2.setStroke(new BasicStroke(2f));
            g2.drawOval(1, 1, getWidth() - 3, getHeight() - 3);
        }
    }
    
    private final static java.io.FileFilter operaFilter  = new java.io.FileFilter()
                                                         {
                                                             @Override
                                                             public boolean accept(File pathname)
                                                             {
                                                                 String fileName = pathname.getName();
                                                                 String extension = FileUtil.getFileExtension(fileName, false);
                                                                 
                                                                 if (extension.equalsIgnoreCase("flex")) return true;
                                                                 
                                                                 if (extension.equalsIgnoreCase("tif"))
                                                                 {
                                                                     // The files could be named
                                                                     // "Y{row}X{col}P{pos}.tif"
                                                                     return fileName.startsWith("Y") && fileName.contains("X") && fileName.contains("P");
                                                                 }
                                                                 
                                                                 return false;
                                                             }
                                                         };
    
    private final Comparator<String>        operaSorter  = new Comparator<String>()
                                                         {
                                                             
                                                             @Override
                                                             public int compare(String file1, String file2)
                                                             {
                                                                 // Flex data: use default sorting
                                                                 if (file1.endsWith("flex")) return file1.compareTo(file2);
                                                                 
                                                                 // TIF data: isolate row, col, pos
                                                                 int sorted = 0;
                                                                 file1 = FileUtil.getFileName(file1, false);
                                                                 file2 = FileUtil.getFileName(file2, false);
                                                                 
                                                                 // Start with the row
                                                                 int i_x1 = file1.indexOf('X');
                                                                 int i_x2 = file2.indexOf('X');
                                                                 Integer row1 = Integer.parseInt(file1.substring(1, i_x1));
                                                                 Integer row2 = Integer.parseInt(file2.substring(1, i_x2));
                                                                 
                                                                 sorted = row1.compareTo(row2);
                                                                 if (sorted != 0) return sorted;
                                                                 
                                                                 // Row isn't enough => use col
                                                                 int i_p1 = file1.indexOf('P');
                                                                 int i_p2 = file2.indexOf('P');
                                                                 Integer col1 = Integer.parseInt(file1.substring(i_x1 + 1, i_p1));
                                                                 Integer col2 = Integer.parseInt(file2.substring(i_x2 + 1, i_p2));
                                                                 
                                                                 sorted = col1.compareTo(col2);
                                                                 if (sorted != 0) return sorted;
                                                                 
                                                                 // Col isn't enough => use pos
                                                                 Integer pos1 = Integer.parseInt(file1.substring(i_p1 + 1));
                                                                 Integer pos2 = Integer.parseInt(file2.substring(i_p2 + 1));
                                                                 
                                                                 return pos1.compareTo(pos2);
                                                             }
                                                         };
    
    EzVarFolder                             folder       = new EzVarFolder("Plate folder", null);
    
    EzVarBoolean                            showAlphanum = new EzVarBoolean("Show alphanumeric well coord.", true);
    
    /**
     * Standard plate formats (rows * columns):
     * 
     * <pre>
     * 6   :  2 *  3
     * 24  :  4 *  6 
     * 96  :  8 * 12 
     * 384 : 16 * 24 
     * 1536: 32 * 48
     * </pre>
     */
    EzVarInteger                            rowNumber    = new EzVarInteger("Row", 1, 1, 24, 1);
    
    /**
     * Standard plate formats (rows * columns):
     * 
     * <pre>
     * 6   :  2 *  3
     * 24  :  4 *  6 
     * 96  :  8 * 12 
     * 384 : 16 * 24 
     * 1536: 32 * 48
     * </pre>
     */
    EzVarInteger                            colNumber    = new EzVarInteger("Column", 1, 1, 48, 1);
    
    EzLabel                                 plateLabel   = new EzLabel(" ");
    
    String[]                                files        = null;
    
    OMEXMLMetadataImpl                      metadata     = null;
    
    @Override
    protected void initialize()
    {
        addEzComponent(folder);
        
        folder.addVarChangeListener(new EzVarListener<File>()
        {
            @Override
            public void variableChanged(EzVar<File> source, File newValue)
            {
                getUI().setProgressBarMessage("Reading plate info...");
                
                if (newValue == null)
                {
                    plateLabel.setText(" ");
                    getUI().setProgressBarMessage("");
                    return;
                }
                
                String plateName = "unknown plate";
                int numberOfWells = 0;
                
                // determine the plate properties from the first file in the folder
                files = FileUtil.getFiles(newValue.getPath(), operaFilter, true, false, false);
                if (files.length == 0) throw new IcyHandledException("Invalid folder: " + newValue.getPath());
                
                metadata = OMEUtil.createOMEMetadata();
                
                // Is there a XML file describing the plate?
                if (FileUtil.exists(newValue.getPath() + ".xml"))
                {
                    Element grid = XMLUtil.loadDocument(newValue.getPath() + ".xml").getDocumentElement();
                    // Safety check
                    if (grid.getNodeName() != "ExperimentGrid") throw new IcyHandledException("Invalid plate info");
                    
                    // Retrieve the plate name
                    Element barcode = XMLUtil.getElement(grid, "Barcode");
                    plateName = XMLUtil.getAttributeValue(barcode, "Value", plateName);
                    
                    // Retrieve the plate format
                    Element plateSetting = XMLUtil.getElement(grid, "PlateSetting");
                    int nbRows = XMLUtil.getAttributeIntValue(plateSetting, "YWells", 0);
                    int nbCols = XMLUtil.getAttributeIntValue(plateSetting, "XWells", 0);
                    numberOfWells = nbRows * nbCols;
                    
                    int channel = 0;
                    for (Element wavelength : XMLUtil.getElements(grid, "Wavelength"))
                    {
                        String chName = XMLUtil.getAttributeValue(wavelength, "Value", "ch. " + channel);
                        metadata.setChannelName(chName, 0, channel);
                        channel++;
                    }
                }
                else try
                {
                    Metadata mmd = TiffMetadataReader.readMetadata(new File(files[0]));
                    Directory dir = mmd.getDirectories().iterator().next();
                    Document flex = XMLUtil.createDocument(dir.getTags().iterator().next().getDescription());
                    Element _root = XMLUtil.getElement(flex, "Root");
                    Element _flex = XMLUtil.getElement(_root, "FLEX");
                    
                    Element _plate = XMLUtil.getElement(_flex, "Plate");
                    plateName = XMLUtil.getElementValue(_plate, "PlateName", plateName);
                    String barcode = XMLUtil.getElementValue(_plate, "Barcode", "");
                    if (!barcode.isEmpty()) plateName = barcode + " (" + plateName + ")";
                    
                    int nbRows = XMLUtil.getElementIntValue(_plate, "XSize", 3);
                    int nbCols = XMLUtil.getElementIntValue(_plate, "YSize", 4);
                    numberOfWells = nbRows * nbCols;
                }
                catch (Exception e)
                {
                    throw new RuntimeException(e);
                }
                
                metadata.setPlateExternalIdentifier(plateName, 0);
                
                switch (numberOfWells)
                {
                case 6:
                    rowNumber.setMaxValue(2);
                    colNumber.setMaxValue(3);
                    break;
                case 24:
                    rowNumber.setMaxValue(4);
                    colNumber.setMaxValue(6);
                    break;
                case 96:
                    rowNumber.setMaxValue(8);
                    colNumber.setMaxValue(12);
                    break;
                case 384:
                    rowNumber.setMaxValue(16);
                    colNumber.setMaxValue(24);
                    break;
                case 1536:
                    rowNumber.setMaxValue(32);
                    colNumber.setMaxValue(48);
                    break;
                }
                
                plateLabel.setText("<html><p align=\"center\">" + plateName + "</p></html>");
                getUI().setProgressBarMessage("");
                
                sequence = new Sequence(metadata, plateName);
            }
        });
        
        addEzComponent(plateLabel);
        
        addEzComponent(showAlphanum);
        
        getUI().setParametersIOVisible(false);
    }
    
    @Override
    protected void execute()
    {
        getUI().setProgressBarMessage("Loading well plate...");
        
        Arrays.sort(files, operaSorter);
        
        final Well[][] wells = new Well[rowNumber.getMaxValue()][colNumber.getMaxValue()];
        
        ThreadUtil.invokeNow(new Runnable()
        {
            @Override
            public void run()
            {
                IcyFrame plateViewer = new IcyFrame(metadata.getPlateExternalIdentifier(0));
                
                plateViewer.setLayout(new BorderLayout());
                
                JPanel wellViews = new JPanel(new GridLayout(rowNumber.getMaxValue(), colNumber.getMaxValue()));
                
                int nbWells = rowNumber.getMaxValue() * colNumber.getMaxValue();
                float factor = 96 / (float) nbWells;
                factor *= (factor < 1f ? 2 : factor > 1f ? 0.5f : 1f);
                
                for (int row = 0; row < rowNumber.getMaxValue(); row++)
                    for (int col = 0; col < colNumber.getMaxValue(); col++)
                    {
                        Well well = new Well(row + 1, col + 1, factor);
                        wells[row][col] = well;
                        wellViews.add(well);
                    }
                
                plateViewer.add(wellViews, BorderLayout.CENTER);
                
                JPanel wellDetails = new JPanel(new BorderLayout());
                wellDetails.add(new WellViewer(), BorderLayout.NORTH);
                
                plateViewer.add(wellDetails, BorderLayout.EAST);
                
                plateViewer.pack();
                plateViewer.addToDesktopPane();
                plateViewer.setVisible(true);
            }
        });
        
        int cpt = 1;
        for (String path : files)
        {
            getUI().setProgressBarValue(cpt / (double) files.length);
            
            if (path.endsWith("xml")) continue;
            
            if (Thread.currentThread().isInterrupted()) break;
            
            String fileName = FileUtil.getFileName(path, false);
            int row = -1, col = -1, pos = -1;
            if (FileUtil.getFileExtension(path, false).endsWith("flex"))
            {
                // {row}{col}{pos}.flex
                row = Integer.parseInt(fileName.substring(0, 3));
                col = Integer.parseInt(fileName.substring(3, 6));
                pos = Integer.parseInt(fileName.substring(6, 9));
            }
            else
            {
                // Y{row}X{col}P{pos}.tif
                int i_x = fileName.indexOf('X');
                int i_p = fileName.indexOf('P');
                row = Integer.parseInt(fileName.substring(1, i_x));
                col = Integer.parseInt(fileName.substring(i_x + 1, i_p));
                pos = Integer.parseInt(fileName.substring(i_p + 1));
            }
            
            wells[row - 1][col - 1].addImage(pos, path);
            
            cpt++;
        }
        
    }
    
    @Override
    public void clean()
    {
        // TODO Auto-generated method stub
        
    }
    
    /**
     * Converts a wavelength into a {@link Color} object.<br/>
     * Taken from Earl F. Glynn's web page:
     * <a href="http://www.efg2.com/Lab/ScienceAndEngineering/Spectra.htm">Spectra Lab Report</a>
     * 
     * @param wavelength
     *            the wavelength to convert (in nanometers)
     * @return a {@link Color} object representing the specified wavelength
     */
    public static Color getColorFromWavelength(double wavelength)
    {
        double factor;
        double r, g, b;
        
        if ((wavelength >= 380) && (wavelength < 440))
        {
            r = -(wavelength - 440) / (440 - 380);
            g = 0.0;
            b = 1.0;
        }
        else if ((wavelength >= 440) && (wavelength < 490))
        {
            r = 0.0;
            g = (wavelength - 440) / (490 - 440);
            b = 1.0;
        }
        else if ((wavelength >= 490) && (wavelength < 510))
        {
            r = 0.0;
            g = 1.0;
            b = -(wavelength - 510) / (510 - 490);
        }
        else if ((wavelength >= 510) && (wavelength < 580))
        {
            r = (wavelength - 510) / (580 - 510);
            g = 1.0;
            b = 0.0;
        }
        else if ((wavelength >= 580) && (wavelength < 645))
        {
            r = 1.0;
            g = -(wavelength - 645) / (645 - 580);
            b = 0.0;
        }
        else if ((wavelength >= 645) && (wavelength < 781))
        {
            r = 1.0;
            g = 0.0;
            b = 0.0;
        }
        else
        {
            r = 0.0;
            g = 0.0;
            b = 0.0;
        }
        
        // Let the intensity fall off near the vision limits
        
        if ((wavelength >= 380) && (wavelength < 420))
        {
            factor = 0.3 + 0.7 * (wavelength - 380) / (420 - 380);
        }
        else if ((wavelength >= 420) && (wavelength < 701))
        {
            factor = 1.0;
        }
        else if ((wavelength >= 701) && (wavelength < 781))
        {
            factor = 0.3 + 0.7 * (780 - wavelength) / (780 - 700);
        }
        else
        {
            factor = 0.0;
        }
        
        int[] rgb = new int[3];
        
        rgb[0] = r == 0.0 ? 0 : (int) Math.round(255 * r * factor);
        rgb[1] = g == 0.0 ? 0 : (int) Math.round(255 * g * factor);
        rgb[2] = b == 0.0 ? 0 : (int) Math.round(255 * b * factor);
        
        return new Color(rgb[0], rgb[1], rgb[2]);
    }
    
    public static class WellPlateField
    {
        private final Unit<Length>       wavelength = Unit.CreateBaseUnit("nanometers", "lambda");
        
        private final OMEXMLMetadataImpl metadata;
        
        private final String             path;
        
        private final String             wellID;
        
        private final int                field;
        
        public WellPlateField(OMEXMLMetadataImpl omeMetadata, String filePath)
        {
            this.metadata = omeMetadata;
            this.path = filePath;
            
            if (FileUtil.getFileExtension(filePath, false).endsWith("flex"))
            {
                String fieldName = FileUtil.getFileName(filePath, false);
                // {row}{col}{pos}.flex
                int row = Integer.parseInt(fieldName.substring(0, 3));
                int col = Integer.parseInt(fieldName.substring(3, 6));
                
                char rowChar = 'A';
                rowChar += (row - 1);
                wellID = rowChar + String.valueOf(col);
                field = Integer.parseInt(fieldName.substring(6, 9));
            }
            else throw new RuntimeException("Invalid file: " + filePath);
        }
        
        public int getField()
        {
            return field;
        }
        
        public String getWellID()
        {
            return wellID;
        }
        
        /**
         * Loads the {@link Sequence} in this well plate field.
         * 
         * @return
         */
        public Sequence loadSequence() throws IOException, SAXException
        {
            // Bypass BioFormats for faster access (!!)
            SeekableStream ss = SeekableStream.wrapInputStream(new FileInputStream(path), true);
            ImageDecoder dec = TIFFCodec.createImageDecoder("tiff", ss, new TIFFDecodeParam());
            
            Metadata mmd = TiffMetadataReader.readMetadata(new File(path));
            Directory dir = mmd.getDirectories().iterator().next();
            // The first tag is the XML description of the entire plate
            // (sweet!!)
            Document flex = XMLUtil.createDocument(dir.getTags().iterator().next().getDescription());
            Element _root = XMLUtil.getElement(flex, "Root");
            Element _flex = XMLUtil.getElement(_root, "FLEX");
            
            // Store filter combinations
            TreeMap<String, Element> fcs = new TreeMap<String, Element>();
            Element _fcs = XMLUtil.getElement(_flex, "FilterCombinations");
            for (Element _fc : XMLUtil.getElements(_fcs, "FilterCombination"))
            {
                // Store the combination as the images reference them (ExpXCamY)
                String expID = XMLUtil.getAttributeValue(_fc, "ID", "");
                expID = "Exp" + expID.substring(expID.length() - 1);
                
                for (Element _sliderRef : XMLUtil.getElements(_fc, "SliderRef"))
                {
                    String camID = XMLUtil.getAttributeValue(_sliderRef, "ID", "");
                    // FIXME Anything other than "Camera..." is not supported
                    if (!camID.startsWith("Camera")) continue;
                    
                    camID = "Cam" + camID.substring(camID.length() - 1);
                    fcs.put(expID + camID, _sliderRef);
                }
            }
            
            // Store Z-stacks
            TreeMap<String, Element> stackRefs = new TreeMap<String, Element>();
            Element _stacks = XMLUtil.getElement(_flex, "Stacks");
            for (Element _stack : XMLUtil.getElements(_stacks, "Stack"))
                stackRefs.put(XMLUtil.getAttributeValue(_stack, "ID", ""), _stack);
            
            // Start reading the well contents
            Element _well = XMLUtil.getElement(_flex, "Well");
            Element _images = XMLUtil.getElement(_well, "Images");
            
            // Prepare the Z-stack
            Element _stack = stackRefs.get(XMLUtil.getElementValue(_well, "StackRef", null));
            int sizeZ = _stack.getChildNodes().getLength();
            
            Sequence sequence = new Sequence(metadata);
            sequence.beginUpdate();
            
            // For each slice, read and compile the different channels
            // NB: slice indexes start at 1
            for (int z = 1; z <= sizeZ; z++)
            {
                // Read the individual channels for the current slice
                ArrayList<IcyBufferedImage> channels = new ArrayList<IcyBufferedImage>();
                int currentChannel = 0;
                
                for (Element _image : XMLUtil.getElements(_images, "Image"))
                {
                    int currentZ = XMLUtil.getElementIntValue(_image, "Stack", 0);
                    if (z != currentZ) continue;
                    
                    // Read the image data
                    int tiffPage = XMLUtil.getAttributeIntValue(_image, "BufferNo", 0);
                    RenderedImage _plane = dec.decodeAsRenderedImage(tiffPage);
                    channels.add(IcyBufferedImage.createFrom(new RenderedImageAdapter(_plane)));
                    
                    // Read the channel information (mostly wavelength)
                    
                    String cameraRef = XMLUtil.getElementValue(_image, "CameraRef", null);
                    Element _fc = fcs.get(cameraRef);
                    String filter = XMLUtil.getAttributeValue(_fc, "Filter", null);
                    String[] filterValues = filter.split("/");
                    int lambda = Integer.parseInt(filterValues[0]) - Integer.parseInt(filterValues[1]);
                    
                    metadata.setChannelName(filter, 0, currentChannel);
                    metadata.setChannelExcitationWavelength(new Length(lambda, wavelength), 0, currentChannel);
                    
                    currentChannel++;
                }
                
                // Channels are ready to be merged
                IcyBufferedImage plane = IcyBufferedImage.createFrom(channels);
                
                // The plane can be added to the stack
                sequence.setImage(0, z - 1, plane); // FIXME handle time-lapse data
            }
            
            // Set the color map (if available)
            try
            {
                for (int c = 0; c < sequence.getSizeC(); c++)
                {
                    int excitation = metadata.getChannelExcitationWavelength(0, c).value().intValue();
                    // Apply a (very) arbitrary Stokes shift of 50nm
                    int emission = excitation + 50;
                    Color rgb = getColorFromWavelength(emission);
                    IcyColorMap colorMap = new LinearColorMap(excitation + "nm", rgb);
                    sequence.setColormap(c, colorMap, true);
                }
            }
            catch (Exception npe)
            {
                // forget it...
            }
            
            sequence.endUpdate();
            sequence.setMetaData(metadata);
            
            return sequence;
        }
    }
    
    public static abstract class OperaWellPlate implements Iterable<WellPlateField>
    {
        private final String name;
        
        private final int    nbImages;
        
        public OperaWellPlate(String plateName, int nbTotalImages)
        {
            name = plateName;
            nbImages = nbTotalImages;
        }
        
        public String getName()
        {
            return name;
        }
        
        public int getNbImages()
        {
            return nbImages;
        }
    }
    
    public static OperaWellPlate loadOperaWellPlate(String plateFolderPath)
    {
        // determine the plate properties from the first file in the folder
        final String[] files = FileUtil.getFiles(plateFolderPath, operaFilter, true, false, false);
        if (files.length == 0) throw new IcyHandledException("Invalid folder: " + plateFolderPath);
        
        Arrays.sort(files);
        
        final OMEXMLMetadataImpl metadata = OMEUtil.createOMEMetadata();
        
        String plateName = "Unnamed plate";
        
        try
        {
            Metadata mmd = TiffMetadataReader.readMetadata(new File(files[0]));
            Directory dir = mmd.getDirectories().iterator().next();
            Document flex = XMLUtil.createDocument(dir.getTags().iterator().next().getDescription());
            Element _root = XMLUtil.getElement(flex, "Root");
            Element _flex = XMLUtil.getElement(_root, "FLEX");
            
            Element _plate = XMLUtil.getElement(_flex, "Plate");
            plateName = XMLUtil.getElementValue(_plate, "PlateName", plateName);
            String barcode = XMLUtil.getElementValue(_plate, "Barcode", "");
            if (!barcode.isEmpty()) plateName = barcode + " (" + plateName + ")";
        }
        catch (Exception e)
        {
            throw new RuntimeException(e);
        }
        
        metadata.setPlateExternalIdentifier(plateName, 0);
        
        return new OperaWellPlate(plateName, files.length)
        {
            @Override
            public Iterator<WellPlateField> iterator()
            {
                return new Iterator<WellPlateField>()
                {
                    private int fileCounter = 0;
                    
                    @Override
                    public void remove()
                    {
                        throw new UnsupportedOperationException("Cannot remove from this iterator");
                    }
                    
                    @Override
                    public WellPlateField next()
                    {
                        return new WellPlateField(metadata, files[fileCounter++]);
                    }
                    
                    @Override
                    public boolean hasNext()
                    {
                        return fileCounter < files.length;
                    }
                };
            }
        };
    }
    
}
