

from icy.image import IcyBufferedImage, IcyBufferedImageUtil
from icy.type import DataType
from icy.type.collection.array import Array1DUtil
from icyexecnetgateway import IcyExecnetGateway, unpack_image, pack_image

# assign a parameter to CPython code
def callCode(parm):
    code = (
"""

from numpyexecnet import unpack_image, pack_image
import numpy as np
import cv2

packed_img = channel.receive()
packed_mask = channel.receive()

img = unpack_image(packed_img)
mask = unpack_image(packed_mask)

"""

+

"""
# --- main processing --- #

# get affinebase image needed for affine convert
def _get_affine_base(image):

    height = image.shape[0]
    width = image.shape[1]
    diagonal = np.ceil((np.sqrt(height**2 + width**2)))
    add_top_bottom = int(np.ceil((diagonal - image.shape[0]) / 2))
    add_left_right = int(np.ceil((diagonal - image.shape[1]) / 2))

    affinebase = cv2.copyMakeBorder(image,add_top_bottom,add_top_bottom,
                                    add_left_right,add_left_right,
                                    cv2.BORDER_ISOLATED)
    cropper = (add_top_bottom,add_top_bottom + height,
               add_left_right,add_left_right + width)

    return affinebase,cropper

# rotate an input image [angle] degrees
def _get_affine_image(affinebase,angle):

    M = cv2.getRotationMatrix2D((affinebase.shape[1]/2, affinebase.shape[0]/2),
                                angle,1)
    rotation_image = cv2.warpAffine(affinebase,M,affinebase.shape)

    return rotation_image

# detect a spot of the specific size
def rmp(image,barsize):

    #    
    # RMP(rotational morphological processing)
    #
    # Kimori et al. BMC Bioinformatics 2010,11:373
    # http://www.biomedcentral.com/1471-2105/11/373
    #

    affinebase,cropper = _get_affine_base(image)
    
    N = 180 / 36
    maxvalue = np.zeros(image.shape,dtype=np.int16)
    bar = np.array([np.ones(barsize,dtype=np.uint8)])

    for m in range(0, 180, N):
        rotation = _get_affine_image(affinebase,m)
        opening = cv2.morphologyEx(rotation,cv2.MORPH_OPEN,bar)
        reverse = _get_affine_image(opening,-m)
        reverse_crop = reverse[cropper[0]:cropper[1],
                               cropper[2]:cropper[3]]
        maxvalue = np.maximum(maxvalue,reverse_crop)

    return np.uint8(maxvalue)

# detect multiple spots by repeating RMP
def segmental_rmp(image,threshold=2,interval=2,lower_limit=3,upper_limit=19):

    stack = np.zeros(image.shape,dtype=np.uint8)

    for i in range(lower_limit,upper_limit,interval):
        
        j = i / 2
        y,x = np.ogrid[-j: j+1, -j: j+1]
        disk = np.uint8(1*((y**2 + x**2 <= j**2) > 0))
        
        opening_ = rmp(image,i)
        opening__ = rmp(opening_,i+interval)

        tophat = np.int16(opening_) - opening__
        tophat[np.where(tophat < 0)] = 0
        tophat = np.uint8(tophat)
        
        thresh,wb = cv2.threshold(tophat,threshold,255,cv2.THRESH_BINARY)

        # remove background noise using by 2D morphology opening
        # shape of structuring element is disk
        wb_open = cv2.morphologyEx(wb,cv2.MORPH_OPEN,disk)

        # overlay an image which detected a spot of specific size 
        stack = cv2.bitwise_or(stack,wb_open)
        
    return stack

def main(image, threshold, lower_limit, upper_limit):

    draw_img = cv2.cvtColor(image,cv2.COLOR_GRAY2BGR)

    image_gau = cv2.GaussianBlur(image,(3,3),-1)
    rmp_img = segmental_rmp(image_gau,threshold=threshold,interval=2,
                            lower_limit=lower_limit,upper_limit=upper_limit)

    rmp_masked = cv2.bitwise_and(rmp_img, mask)
    bin_masked = np.copy(rmp_masked)

    contours, hierarchy = cv2.findContours(rmp_masked,cv2.RETR_EXTERNAL,
                                           cv2.CHAIN_APPROX_NONE)
    cv2.drawContours(draw_img,contours,-1,(0,0,255),1)
    
    return draw_img, bin_masked

# ----------------------- #
"""

+

"draw_img, bin_masked = main(img, threshold={0}, lower_limit={1}, upper_limit={2})\n".format(parm[2],parm[1],parm[0])

+

"""
for c in cv2.split(draw_img):
    out_c = np.empty(c.shape,dtype=np.int32)
    out_c[:] = c[:]
    packed_c = pack_image(out_c)
    channel.send(packed_c)

out_bin = np.empty(img.shape,dtype=np.int32)
out_bin[:] = bin_masked[:]
packed_bin = pack_image(out_bin)
channel.send(packed_bin)

"""
)
    return code

# ([1channel,1channel,1channel], INT) => ([3channel, UBYTE]) 
def _receive3ChannelsConvertUBYTE(gateway, img):

    # receive image (1channel, 1channel, 1channel)
    bgr = []
    for i in range(3):
        packed_channel = gateway.receive()
        channel = unpack_image(packed_channel)
        bgr.append(channel)

    # generate new image and set received data to new image(3channel,UBYTE)
    out = IcyBufferedImage(img.getSizeX(),img.getSizeY(),
                           3,DataType.UBYTE)

    for i in range(3):
        Array1DUtil.intArrayToSafeArray(bgr[i].getDataXY(0),
                                        out.getDataXY(2-i), 
                                        out.isSignedDataType())
    out.dataChanged()
    return out 

# ([1channel], INT) => ([1channel], UBYTE)
def _receive1ChannelConvertUBYTE(gateway, img):

    # receive image (1channel)
    packed_binary = gateway.receive()
    binary = unpack_image(packed_binary)

    # generate new image and set received data to new image(1channel,UBYTE)
    bin = IcyBufferedImage(img.getSizeX(),img.getSizeY(),
                           1,DataType.UBYTE)

    Array1DUtil.intArrayToSafeArray(binary.getDataXY(0),
                                    bin.getDataXY(0), 
                                    bin.isSignedDataType())
    bin.dataChanged()
    return bin

# handle it in CPython by using Execnet
def useExecnet(code, img, mask, pypath):
    
    with IcyExecnetGateway(python_path = pypath) as gateway:
            
        gateway.remote_exec(code)
        
        packed_img = pack_image(img)
        packed_mask = pack_image(mask)
        
        gateway.send(packed_img)
        gateway.send(packed_mask)
        
        out = _receive3ChannelsConvertUBYTE(gateway,img)
        bin = _receive1ChannelConvertUBYTE(gateway,img)
        return out, bin



if __name__ == "__main__" :
    from icy.main import Icy
    from icy.sequence import Sequence, SequenceUtil
    
    seq = Icy.getMainInterface().getFocusedSequence() #Sequence
    img = Icy.getMainInterface().getFocusedImage() #IcyBufferedImage
            
    if seq == None:
        import sys
        sys.exit("No sequence")

    code = callCode([21,11,5])
    print code
            
    out, bin = useExecnet(code, img, img, pypath=None)
            
    out_seq = Sequence(out)
    Icy.addSequence(out_seq)
