package plugins.dmandache.denoise;

public class Fusion {

	/**
	 * Spatially-adaptive fusion function for combining multiple CS reconstructions 
	 * 
	 * It implements the fusion operator described in:
	 * W. Meiniel, E. Angelini and J.-C. Olivo-Marin
	 * "Image denoising by adaptive compressed sensing reconstructions and fusions", in Proc. SPIE 9597,September 2015.
	 * doi: 
	 * URL: 
	 * 
	 * @author Diana Mandache
	 * @version 1.0
	 * @date 2017-02-20
	 * @license gpl v3.0
	 */
	
	/**
	 * Computes the average image of all the reconstructions
	 * 
	 * @param xi a double array containing pixel values xi[k][x + y*width] corresponds to the pixel value at coordinate {x, y} of the k-th reconstruction where k ranges form 1 to R
	 * @param R number of reconstructions
	 * @param w width of the image
	 * @param h height of the image 
	 * */
	public static double[] mean(double[][] xi, int R, int w, int h){
		double[] xm = new double[w*h];
		for (int r=0; r<R; r++)
			for (int k=0; k<w*h; k++)
				xm[k] += xi[r][k];
		for (int k=0; k<w*h; k++)
			xm[k] /= R;
		return xm;		
	}
	
	/**
	 * Computes the variance image between the reconstructions
	 * 
	 * @param xi a double array containing pixel values xi[k][x + y*width] corresponds to the pixel value at coordinate {x, y} of the k-th reconstruction where k ranges form 1 to R
	 * @param R number of reconstructions
	 * @param w width of the image
	 * @param h height of the image 
	 * */
	public static double[] varianceMap(double[][] xi, int R, int w, int h){
		double[] sigma = new double[w*h];
		double[] xm = mean(xi,R, w, h);
		if (R > 1) {
			for (int r=0; r<R; r++){
				double[] diff = Util.subtract1D(xi[r], xm);
				double[] diffSquared = Util.multiplyElementwise1D(diff, diff);
				sigma = Util.add1D(sigma, diffSquared);
			}		
			for (int k=0; k<w*h; k++)
				sigma[k] = Math.sqrt( sigma[k] / (double)(R-1)) ;
		}
		return sigma;		
	}
	
	/**
	 * Computes the normalized variance map
	 * 
	 * @param xi a double array containing pixel values xi[k][x + y*width] corresponds to the pixel value at coordinate {x, y} of the k-th reconstruction where k ranges form 1 to R
	 * @param R number of reconstructions
	 * @param w width of the image
	 * @param h height of the image 
	 * */
	public static double[] normalizedVarianceMap(double[][] xi, int R, int w, int h){
		double[] sigma = varianceMap(xi, R, w, h);
		double norm = Util.normL2(sigma);
		for (int k=0; k<w*h; k++)			
			sigma[k] = sigma[k] / norm;		
		return sigma;		
	}
	
	/**
	 * Fusion operator
	 * 
	 * @param xi a double array containing pixel values xi[k][x + y*width] corresponds to the pixel value at coordinate {x, y} of the k-th reconstruction where k ranges form 1 to R
	 * @param R number of reconstructions
	 * @param w width of the image
	 * @param h height of the image 
	 * */
	public static double[] fusion(double[][] xi, double[] y, int R, int w, int h){
		double[] xhat = new double[w*h];
		if(R==1){
			xhat = xi[0];		
		}
		else{
			double[] xmean = mean(xi, R, w, h);
			double[] sigma = normalizedVarianceMap(xi, R, w, h); 
			for (int k=0; k<w*h; k++)
				xhat[k] = sigma[k]*y[k] + (1.0-sigma[k])*xmean[k];
		}
		return xhat;
	}
}
