package plugins.fab.internetconnectivitymonitor;

import icy.sequence.Sequence;
import icy.sequence.SequenceAdapter;

import java.util.Calendar;
import java.util.Date;

/**
 * 
 * @author Yoann Le Montagner
 * 
 * Check periodically a condition, and report it in the given connectivity map
 */
public abstract class MonitoringDeamon
{
	private ConnectivityMap _map;
	private boolean _stop;
	
	/**
	 * Constructor
	 */
	public MonitoringDeamon(ConnectivityMap map)
	{
		_map = map;
		_stop = false;
		_map.addListener(new SequenceAdapter()
		{
			@Override
			public void sequenceClosed(Sequence sequence)
			{
				stop();
			}
		});
	}
	
	/**
	 * Assert the stop flag
	 */
	public void stop()
	{
		_stop = true;
	}
	
	/**
	 * Run the monitoring loop
	 */
	public void run()
	{
		while(!_stop)
		{	
			// Check the connectivity
			boolean connected = checkConnectivity();
			
			// Compute the current date
			Date now = new Date();
			Calendar calendar = Calendar.getInstance(); 
			calendar.setTime(now);
			int year  = calendar.get(Calendar.YEAR        );
			int month = calendar.get(Calendar.MONTH       )+1; // Java returns month valued between 0 and 11 by default :-( 
			int day   = calendar.get(Calendar.DAY_OF_MONTH);
			int hr    = calendar.get(Calendar.HOUR_OF_DAY );
			int min   = calendar.get(Calendar.MINUTE      );
			int sec   = calendar.get(Calendar.SECOND      );
			
			// Feed the connectivity map
			_map.updateMap(connected, year, month, day, hr, min, sec);
			
			// Waiting loop
			int counter = 10*Math.max(1, _map.getInterval()-2); // Wait 'interval-2' seconds (but at least 1 second)
			while(counter>0 && !_stop) {
				try {
					Thread.sleep(100); // Check the stop flag every 100 milliseconds
				}
				catch (InterruptedException e) {}
				--counter;
			}
		}
		
		// Don't forget to save the sequence at the end
		_map.save();
	}
	
	/**
	 * Check connectivity function to implement
	 */
	protected abstract boolean checkConnectivity();
}
