package plugins.ylemontag.noisegenerator;

import icy.gui.frame.GenericFrame;
import icy.sequence.Sequence;
import icy.system.IcyHandledException;

import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JScrollPane;
import javax.swing.JTextPane;
import javax.swing.Timer;

import plugins.adufour.ezplug.EzButton;
import plugins.adufour.ezplug.EzGUI;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzStoppable;
import plugins.adufour.ezplug.EzVarDouble;
import plugins.adufour.ezplug.EzVarEnum;
import plugins.adufour.ezplug.EzVarSequence;

/**
 * 
 * @author Yoann Le Montagner
 * 
 * EzPlug interface to generate a noise on a sequence
 */
public class NoiseGeneratorPlugin extends EzPlug implements EzStoppable
{
	private Timer                      _timer     ;
	private Controller                 _controller;
	private EzVarSequence              _seq       ;
	private EzVarEnum<NoiseModel.Type> _type      ;
	private EzVarDouble                _sigma     ;
	private EzVarDouble                _intensity ;
	private EzVarDouble                _lowerBound;
	private EzVarDouble                _upperBound;
	
	@Override
	protected void initialize()
	{
		// Main components
		_seq        = new EzVarSequence("Input");
		_type       = new EzVarEnum<NoiseModel.Type>("Noise model", NoiseModel.Type.values());
		_sigma      = new EzVarDouble("Sigma"      );
		_intensity  = new EzVarDouble("Intensity"  );
		_lowerBound = new EzVarDouble("Lower bound");
		_upperBound = new EzVarDouble("Upper bound");
		_type.addVisibilityTriggerTo(_sigma     , NoiseModel.Type.GAUSSIAN   );
		_type.addVisibilityTriggerTo(_intensity , NoiseModel.Type.SALT_PEPPER);
		_type.addVisibilityTriggerTo(_lowerBound, NoiseModel.Type.SALT_PEPPER);
		_type.addVisibilityTriggerTo(_upperBound, NoiseModel.Type.SALT_PEPPER);
		_sigma     .setValue( 30.0);
		_intensity .setValue(  0.2);
		_lowerBound.setValue(  0.0);
		_upperBound.setValue(255.0);
		addEzComponent(_seq       );
		addEzComponent(_type      );
		addEzComponent(_sigma     );
		addEzComponent(_intensity );
		addEzComponent(_lowerBound);
		addEzComponent(_upperBound);
		
		// Details button
		EzButton detailsButton = new EzButton("Help on noise models", new ActionListener()
		{
			@Override
			public void actionPerformed(ActionEvent e)
			{
				onDetailsClicked();
			}
		});
		addEzComponent(detailsButton);
		
		// Periodic refresh of the progress bar
		_timer = new Timer(100, new ActionListener()
		{
			@Override
			public void actionPerformed(ActionEvent e)
			{
				refreshProgressBar();
			}
		});
		_timer.start();
	}

	@Override
	protected void execute()
	{
		// Inputs
		Sequence   in         = _seq.getValue(true);
		NoiseModel noiseModel = retrieveNoiseModel();
		
		// Generate a noisy version of the input sequence
		try {
			_controller  = new Controller();
			Sequence out = noiseModel.addNoise(in, _controller);
			addSequence(out);
		}
		catch(Controller.CanceledByUser err) {}
		finally {
			_controller = null;
			refreshProgressBar();
		}
	}
	
	@Override
	public void clean()
	{
		_timer.stop();
	}
	
	@Override
	public void stopExecution()
	{
		Controller controller = _controller;
		if(controller!=null) {
			controller.cancelComputation();
		}
	}
	
	/**
	 * Refresh the progress bar with the progress value in held by the controller, if available
	 */
	private void refreshProgressBar()
	{
		EzGUI gui = getUI();
		if(gui==null) {
			return;
		}
		Controller controller = _controller;
		gui.setProgressBarValue(controller==null ? 0.0 : controller.getProgress());
	}
	
	/**
	 * Return a newly allocated noise model corresponding to the parameter
	 * selected by the user
	 */
	private NoiseModel retrieveNoiseModel()
	{
		try {
			switch(_type.getValue(true))
			{
				case GAUSSIAN:
					return NoiseModel.createGaussianNoiseModel(_sigma.getValue(true));
				
				case POISSON:
					return NoiseModel.createPoissonNoiseModel();
				
				case SALT_PEPPER:
					return NoiseModel.createSaltPepperNoiseModel(_intensity.getValue(true),
						_lowerBound.getValue(true), _upperBound.getValue(true));
				
				default:
					throw new RuntimeException("Unreachable code point");			
			}
		}
		catch(IllegalNoiseModelArgumentException err) {
			throw new IcyHandledException(err.getMessage());
		}
	}
	
	/**
	 * Action performed when the user clicks on the details button
	 */
	private void onDetailsClicked()
	{
		String    title   = "Noise models";
		JTextPane message = new JTextPane();
		message.setEditable(false);
		message.setContentType("text/html");
		message.setText(
			"<p>" +
				"In what follows, <tt>A</tt> represents the input clean sequence, " +
				"while <tt>B</tt> is the output noisy sequence." +
			"</p>" +
			"<h2>White additive Gaussian noise</h2>" +
			"<p>" +
				"1 parameter: <tt>sigma &gt;= 0</tt> (standard deviation of the Gaussian random variables)." +
			"</p>" +
			"<p>" +
				"This noise model enforces <tt>B = A + n</tt>, where <tt>n</tt> is a " +
				"random sequence such that the samples <tt>n(x,y,z,t,c)</tt> are random " +
				"independant variables following a Gaussian probability distribution " +
				"of mean <tt>0</tt> and variance <tt>sigma^2</tt>." +
			"</p>" +
			"<h2>Poisson noise</h2>" +
			"<p>" +
				"No parameter." +
			"</p>" +
			"<p>" +
				"In this model, each output sample <tt>B(x,y,z,t,c)</tt> is generated from a Poisson random " +
				"distribution of intensity <tt>A(x,y,z,t,c)</tt>, which is supposed to be <tt>&gt;=0</tt>. " +
				"If <tt>A(x,y,z,t,c) &lt; 0</tt>, then <tt>B(x,y,z,t,c)</tt> is set to <tt>NaN</tt>." +
			"</p>" +
			"<h2>Salt &amp; pepper noise</h2>" +
			"<p>" +
				"3 parameters:" +
				"<ul>" +
					"<li><tt>intensity</tt>, that must satisfy <tt>0 &lt;= intensity &lt;= 1</tt>,</li>" +
					"<li><tt>lowerBound</tt> and <tt>upperBound</tt>, with <tt>lowerBound &lt;= upperBound</tt>.</li>" +
				"</ul>" +
			"</p>" +
			"<p>" +
				"In this model, each output sample <tt>B(x,y,z,t,c):</tt>" +
				"<ul>" +
					"<li>" +
						"either is let unchanged (i.e. is set to <tt>A(x,y,z,t,c)</tt>), " +
						"with probability <tt>intensity</tt>," +
					"</li>" +
					"<li>" +
						"or takes a value that is selected in a random uniform manner " +
						"between <tt>lowerBound</tt> and <tt>upperBound</tt>, with " +
						"probability <tt>1-intensity</tt>." +
					"</li>" +
				"</ul>" +
			"</p>"
		);
		Dimension dim = message.getPreferredSize();
		dim.setSize(600, dim.getHeight()+100);
		message.setPreferredSize(dim);
		JScrollPane scroll = new JScrollPane(message);
		dim = scroll.getPreferredSize();
		dim.setSize(600, 500);
		scroll.setPreferredSize(dim);
		GenericFrame infoFrame = new GenericFrame(title, scroll);
		infoFrame.addToMainDesktopPane();
		infoFrame.setVisible(true);
		infoFrame.requestFocus();
	}
}
