/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.blobgenerator;

import icy.image.IcyBufferedImage;
import icy.image.ImageUtil;
import icy.main.Icy;
import icy.sequence.Sequence;
import icy.sequence.SequenceEvent;
import icy.sequence.SequenceListener;
import icy.swimmingPool.SwimmingObject;
import icy.type.DataType;

import java.awt.Image;
import java.util.ArrayList;
import java.util.List;

import javax.swing.ImageIcon;
import javax.swing.SwingUtilities;

import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;

import plugins.adufour.blocks.lang.Block;
import plugins.adufour.blocks.util.VarList;
import plugins.adufour.ezplug.EzComponent;
import plugins.adufour.ezplug.EzGroup;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzStoppable;
import plugins.adufour.ezplug.EzVarBoolean;
import plugins.adufour.vars.lang.Var;
import plugins.adufour.vars.lang.VarSequence;
import plugins.adufour.vars.lang.VarWorkbook;
import plugins.adufour.vars.util.VarListener;
import plugins.big.bigsnakeutils.icy.ellipsoid.AbstractEllipsoid;
import plugins.big.bigsnakeutils.icy.ellipsoid.Ellipsoid2D;
import plugins.big.bigsnakeutils.icy.ellipsoid.Ellipsoid3D;
import plugins.big.bigsnakeutils.icy.gui.curve.Curve;
import plugins.big.bigsnakeutils.icy.gui.curve.PolynomialCurve;
import plugins.big.blobgenerator.filters.BlendingFilter;
import plugins.big.blobgenerator.filters.BlurFilter;
import plugins.big.blobgenerator.parameters.Parameters;
import plugins.big.blobgenerator.parameters.ParametersCategory;
import plugins.fab.trackmanager.TrackGroup;
import plugins.fab.trackmanager.TrackManager;
import plugins.fab.trackmanager.TrackSegment;

/**
 * Main class of the Blob Generator plugin.
 * 
 * This plugin generates random cells (ellipsoids) in a noisy background. The
 * result consists in a 2D or 3D sequence images. The user can tune the noisy
 * background by editing the noise density function and choosing the blending
 * function.He can also choose the noise scale, i.e. the spatial frequency of
 * the noise. Moreover, ROI corresponding to the cells are added to the
 * sequence.
 * 
 * This plugin works with Protocols as well.
 * 
 * @version May 3, 2014
 * 
 * @author Julien Jacquemot
 */
public class BlobGenerator extends EzPlug implements Block, EzStoppable {
	private static final int PREVIEW_SIZE = 256;

	private Parameters _parameters = null;
	private Sequence _sequence = null;
	private VarSequence _varSequence = null;

	private List<List<AbstractEllipsoid>> _cells;
	private VarWorkbook _cellsWorkbook;

	private EzVarBoolean _addROI;
	private EzVarBoolean _useTrackManager;
	private EzVarBoolean _previewNoise;
	private IcyBufferedImage _emptyPreviewImg;
	private Sequence _noisePreviewSequence;
	private boolean _stop;

	@Override
	protected void initialize() {
		_parameters = new Parameters(this, true);

		initializeGroup(ParametersCategory.CellsProperties, "Cells Properties");
		EzGroup noiseGroup = initializeGroup(
				ParametersCategory.NoiseProperties, "Noise Properties");
		EzGroup sequenceGroup = initializeGroup(
				ParametersCategory.SequenceProperties, "Sequence Properties");

		_previewNoise = new EzVarBoolean("Preview", false);
		_previewNoise.getVariable().addListener(new VarListener<Boolean>() {
			@Override
			public void valueChanged(Var<Boolean> source, Boolean oldValue,
					Boolean newValue) {
				if (newValue) {
					addSequence(_noisePreviewSequence);
					BlobGenerator.this.valueChanged(null);
				} else {
					removeSequence(_noisePreviewSequence);
				}
			}

			@Override
			public void referenceChanged(Var<Boolean> source,
					Var<? extends Boolean> oldReference,
					Var<? extends Boolean> newReference) {
			}
		});
		noiseGroup.addEzComponent(_previewNoise);

		_noisePreviewSequence = new Sequence("Noise");
		_noisePreviewSequence.addListener(new SequenceListener() {
			@Override
			public void sequenceClosed(Sequence sequence) {
				_previewNoise.setValue(false);
			}

			@Override
			public void sequenceChanged(SequenceEvent sequenceEvent) {
			}
		});

		_emptyPreviewImg = new IcyBufferedImage(PREVIEW_SIZE, PREVIEW_SIZE, 1,
				DataType.DOUBLE);
		double data[] = _emptyPreviewImg.getDataXYAsDouble(0);
		for (int i = 0; i < data.length; ++i) {
			data[i] = _parameters.backgroundValue();
		}

		_addROI = new EzVarBoolean("Add cells ROI to the sequence", true);
		sequenceGroup.addEzComponent(_addROI);
		_useTrackManager = new EzVarBoolean("Use Track Manager", false);
		sequenceGroup.addEzComponent(_useTrackManager);

		// Set icons
		try {
			List<Image> icons = new ArrayList<Image>();
			icons.add(ImageUtil.load(this.getClass().getResourceAsStream(
					"icons/icon_16.png")));
			icons.add(ImageUtil.load(this.getClass().getResourceAsStream(
					"icons/icon_32.png")));
			icons.add(ImageUtil.load(this.getClass().getResourceAsStream(
					"icons/icon_64.png")));
			icons.add(ImageUtil.load(this.getClass().getResourceAsStream(
					"icons/icon_128.png")));
			icons.add(ImageUtil.load(this.getClass().getResourceAsStream(
					"icons/icon_256.png")));

			super.getUI().getExternalFrame().setIconImages(icons);
			super.getUI().getInternalFrame()
					.setFrameIcon(new ImageIcon(icons.get(0)));
		} catch (Exception e) {
			System.err.println("Failed to set the window icon.");
			e.printStackTrace();
		}
	}

	@Override
	protected void execute() {
		_sequence = new Sequence("Cells");
		_cells = new ArrayList<List<AbstractEllipsoid>>();
		_stop = false;

		// Reset the workbook
		if (isHeadLess()) {
			Sheet cellsSheet = _cellsWorkbook.getValue().getSheet("Cells");
			for (int i = cellsSheet.getFirstRowNum(); i <= cellsSheet
					.getLastRowNum(); ++i) {
				Row row = cellsSheet.getRow(i);
				if (row != null) {
					cellsSheet.removeRow(row);
				}
			}
			cellsSheet.createRow(0);
			Row headers = cellsSheet.getRow(0);

			int idx = 0;
			headers.createCell(idx);
			headers.getCell(idx++).setCellValue("Cell");
			headers.createCell(idx);
			headers.getCell(idx++).setCellValue("Time");
			headers.createCell(idx);
			headers.getCell(idx++).setCellValue("Dimension");

			if (_parameters.imageDepth() == 1) {
				headers.createCell(idx);
				headers.getCell(idx++).setCellValue("x0");
				headers.createCell(idx);
				headers.getCell(idx++).setCellValue("y0");
				headers.createCell(idx);
				headers.getCell(idx++).setCellValue("a");
				headers.createCell(idx);
				headers.getCell(idx++).setCellValue("b");
				headers.createCell(idx);
				headers.getCell(idx++).setCellValue("alpha");
			} else {
				headers.createCell(idx);
				headers.getCell(idx++).setCellValue("x0");
				headers.createCell(idx);
				headers.getCell(idx++).setCellValue("y0");
				headers.createCell(idx);
				headers.getCell(idx++).setCellValue("z0");
				headers.createCell(idx);
				headers.getCell(idx++).setCellValue("a");
				headers.createCell(idx);
				headers.getCell(idx++).setCellValue("b");
				headers.createCell(idx);
				headers.getCell(idx++).setCellValue("c");
				headers.createCell(idx);
				headers.getCell(idx++).setCellValue("alpha");
				headers.createCell(idx);
				headers.getCell(idx++).setCellValue("beta");
				headers.createCell(idx);
				headers.getCell(idx++).setCellValue("gamma");
			}
		} else {
			super.getUI().setProgressBarVisible(true);
		}

		IcyBufferedImage[] cellsImg = new IcyBufferedImage[_parameters
				.imageDepth()];
		for (int z = 0; z < _parameters.imageDepth(); ++z) {
			cellsImg[z] = new IcyBufferedImage(_parameters.imageWidth(),
					_parameters.imageHeight(), 1, DataType.DOUBLE);
		}
		List<AbstractEllipsoid> cells = new ArrayList<AbstractEllipsoid>();

		double progress = 0;
		double progressMax = _parameters.sequenceDuration()
				* _parameters.cellsCount();

		for (int t = 0; t < _parameters.sequenceDuration() && !_stop; ++t) {
			// Reset the images
			for (int z = 0; z < _parameters.imageDepth(); ++z) {
				double data[] = cellsImg[z].getDataXYAsDouble(0);
				for (int i = 0; i < data.length; ++i) {
					data[i] = _parameters.backgroundValue();
				}
			}

			if (_stop) {
				return;
			}

			// Generate cells
			if (t == 0) {
				// Initialize probability map
				if (!isHeadLess()) {
					super.getUI().setProgressBarMessage(
							"Initialize probability map");
					super.getUI().setProgressBarValue(0);
				}

				int layerMax = (int) (Math.log(Math.min(
						_parameters.imageWidth(), _parameters.imageHeight())) / Math
						.log(2));
				Curve probabilityProfile = new PolynomialCurve(1, 1);
				probabilityProfile.insertControlPoint(1, 0.5, 0);

				IcyBufferedImage[] probabilityLayers = NoiseGenerator
						.generateNoise(_parameters, probabilityProfile,
								Math.max(0, layerMax - 4),
								Math.max(0, layerMax - 2));

				double probabilityMap[] = new double[_parameters.imageWidth()
						* _parameters.imageHeight() * _parameters.imageDepth()];
				double f = 1.0 / (_parameters.cellsRadiusRange().getFirst() * _parameters
						.cellsRadiusRange().getSecond());
				int index = 0;
				for (int z = 0; z < probabilityLayers.length; ++z) {
					double probabilitySrc[] = probabilityLayers[z]
							.getDataXYAsDouble(0);
					double zf = (probabilityLayers.length < 3) ? 1.0 : Math
							.min(1, f * z * (probabilityLayers.length - 1 - z));
					int i = 0;

					for (int y = 0; y < _parameters.imageHeight(); ++y) {
						double yf = Math.min(1,
								f * y * (_parameters.imageHeight() - 1 - y));

						for (int x = 0; x < _parameters.imageWidth(); ++x, ++i, ++index) {
							double xf = Math.min(1,
									f * x * (_parameters.imageWidth() - 1 - x));

							probabilityMap[index] = xf * yf * zf
									* probabilitySrc[i];
						}
					}
				}

				if (_stop) {
					return;
				}
				if (!isHeadLess()) {
					super.getUI().setProgressBarMessage(
							"Generate cells (t=" + t + ")");
				}

				// Generate the cells
				for (int i = 0; i < _parameters.cellsCount() && !_stop; ++i, ++progress) {
					_cells.add(new ArrayList<AbstractEllipsoid>());
					AbstractEllipsoid cell = CellGenerator.generateCell(
							_parameters, cellsImg, probabilityMap);
					cells.add(cell);
					_cells.get(i).add(cell.clone());

					if (!isHeadLess()) {
						super.getUI().setProgressBarMessage(
								"Generate cells (t=0) : Cell " + i
										+ " generated");
						super.getUI().setProgressBarValue(
								progress / progressMax);
					}
				}
			} else {
				int i = 0;
				for (AbstractEllipsoid cell : cells) {
					if (_stop) {
						return;
					}

					CellGenerator.generateCell(_parameters, cellsImg, cell);
					cell.setT(t);
					_cells.get(i).add(cell.clone());

					if (!isHeadLess()) {
						super.getUI().setProgressBarMessage(
								"Generate cells (t=" + t + ") : Cell " + i
										+ " generated");
						super.getUI().setProgressBarValue(
								progress / progressMax);
					}

					++i;
					++progress;
				}
			}

			if (_stop) {
				return;
			}

			// Blur the cells
			if (!isHeadLess()) {
				super.getUI().setProgressBarMessage("Blur cells");
			}

			cellsImg = BlurFilter
					.apply(cellsImg, _parameters.cellsBlurRadius());

			if (_stop) {
				return;
			}

			if (_parameters.noiseIntensity() == 0) {
				for (int z = 0; z < _parameters.imageDepth(); ++z) {
					_sequence.setImage(t, z, convertToByteImage(cellsImg[z]));
				}
			} else {
				// Generate noise
				if (!isHeadLess()) {
					super.getUI().setProgressBarMessage(
							"Generate noise (t=" + t + ")");
				}
				IcyBufferedImage[] noiseLayers = NoiseGenerator
						.generateNoise(_parameters);

				if (_stop) {
					return;
				}

				// Generate noise
				if (!isHeadLess()) {
					super.getUI().setProgressBarMessage(
							"Mix cell and noise (t=" + t + ")");
				}

				// Mix the cells and the noise and add the result to the
				// sequence
				for (int z = 0; z < _parameters.imageDepth(); ++z) {
					_sequence.setImage(t, z, convertToByteImage(BlendingFilter
							.apply(cellsImg[z], noiseLayers[z],
									_parameters.noiseBlendingMode(),
									_parameters.noiseIntensity())));
				}
			}

			// Add the cells to the workbook
			if (isHeadLess()) {
				Sheet cellsSheet = _cellsWorkbook.getValue().getSheet("Cells");
				int i = 0;
				int index = cellsSheet.getLastRowNum() + 1;
				for (AbstractEllipsoid cell : cells) {
					cellsSheet.createRow(index + i);
					Row row = cellsSheet.getRow(index + i);

					int idx = 0;
					row.createCell(idx);
					row.getCell(idx++).setCellValue("Cell " + i);
					row.createCell(idx);
					row.getCell(idx++).setCellValue(t);

					if (cell instanceof Ellipsoid2D) {
						Ellipsoid2D ellipse = (Ellipsoid2D) cell;
						row.createCell(idx);
						row.getCell(idx++).setCellValue(2);
						row.createCell(idx);
						row.getCell(idx++).setCellValue(ellipse.x0);
						row.createCell(idx);
						row.getCell(idx++).setCellValue(ellipse.y0);
						row.createCell(idx);
						row.getCell(idx++).setCellValue(ellipse.a);
						row.createCell(idx);
						row.getCell(idx++).setCellValue(ellipse.b);
						row.createCell(idx);
						row.getCell(idx++).setCellValue(ellipse.alpha);
					} else if (cell instanceof Ellipsoid3D) {
						Ellipsoid3D ellipsoid = (Ellipsoid3D) cell;
						row.createCell(idx);
						row.getCell(idx++).setCellValue(3);
						row.createCell(idx);
						row.getCell(idx++).setCellValue(ellipsoid.x0);
						row.createCell(idx);
						row.getCell(idx++).setCellValue(ellipsoid.y0);
						row.createCell(idx);
						row.getCell(idx++).setCellValue(ellipsoid.z0);
						row.createCell(idx);
						row.getCell(idx++).setCellValue(ellipsoid.a);
						row.createCell(idx);
						row.getCell(idx++).setCellValue(ellipsoid.b);
						row.createCell(idx);
						row.getCell(idx++).setCellValue(ellipsoid.c);
						row.createCell(idx);
						row.getCell(idx++).setCellValue(ellipsoid.alpha);
						row.createCell(idx);
						row.getCell(idx++).setCellValue(ellipsoid.beta);
						row.createCell(idx);
						row.getCell(idx++).setCellValue(ellipsoid.gamma);
					}

					++i;
				}
			}
		}

		if (isHeadLess()) { // Protocol output
			_varSequence.setValue(_sequence);
		} else {
			// Create tracks
			if (_useTrackManager.getValue()
					&& _parameters.sequenceDuration() > 1) {
				TrackGroup trackGroup = new TrackGroup(_sequence);
				trackGroup.setDescription("Cells Tracks");

				for (List<AbstractEllipsoid> cellList : _cells) {
					TrackSegment trackSegment = new TrackSegment();
					trackSegment.generateId();

					for (AbstractEllipsoid cell : cellList) {
						if (cell instanceof Ellipsoid2D) {
							trackSegment.addDetection(new CellDetection2D(
									(Ellipsoid2D) cell));
						} else if (cell instanceof Ellipsoid3D) {
							trackSegment.addDetection(new CellDetection3D(
									(Ellipsoid3D) cell));
						}
					}

					trackGroup.addTrackSegment(trackSegment);
				}
				SwingUtilities.invokeLater(new SendTracksToPoolThr(trackGroup,
						_sequence));
			} else if (_addROI.getValue()) {
				for (List<AbstractEllipsoid> cellList : _cells) {
					for (AbstractEllipsoid cell : cellList) {
						_sequence.addROI(cell.toROI());
					}
				}
			}

			super.getUI().setProgressBarVisible(false);

			// Display the sequence
			addSequence(_sequence);
		}
	}

	@Override
	public void stopExecution() {
		_stop = true;
	}

	@Override
	public void clean() {
	}

	@Override
	public void declareInput(VarList inputMap) {
		_parameters = new Parameters(this, false);

		initializeGroup(ParametersCategory.CellsProperties, inputMap);
		initializeGroup(ParametersCategory.NoiseProperties, inputMap);
		initializeGroup(ParametersCategory.SequenceProperties, inputMap);
	}

	@Override
	public void declareOutput(VarList outputMap) {
		_varSequence = new VarSequence("Generated Sequence", _sequence);
		outputMap.add(_varSequence);

		_cellsWorkbook = new VarWorkbook("Cells Workbook", "Cells");
		outputMap.add(_cellsWorkbook);
	}

	/**
	 * Method called when a noise property changed. Updates the noise preview
	 * sequence if visible.
	 */
	public void valueChanged(Var<?> source) {
		if (source != null && source.getName() == "Background Value") {
			double data[] = _emptyPreviewImg.getDataXYAsDouble(0);
			for (int i = 0; i < data.length; ++i) {
				data[i] = _parameters.backgroundValue();
			}
		}

		if (_previewNoise.getValue()) {
			IcyBufferedImage noise[] = NoiseGenerator.generateNoise(
					_parameters, PREVIEW_SIZE, PREVIEW_SIZE, 1);
			IcyBufferedImage result = BlendingFilter.apply(_emptyPreviewImg,
					noise[0], _parameters.noiseBlendingMode(),
					_parameters.noiseIntensity());
			_noisePreviewSequence.setImage(0, 0, result);
		}
	}

	/**
	 * Initialize the given properties group.
	 * 
	 * @param name
	 *            Name to display for this group.
	 */
	private EzGroup initializeGroup(final ParametersCategory category,
			final String name) {
		if (name.isEmpty()) {
			for (EzComponent component : _parameters.ezVariables.get(category)) {
				addEzComponent(component);
			}
		} else {
			EzGroup group = new EzGroup(name);
			for (EzComponent component : _parameters.ezVariables.get(category)) {
				group.addEzComponent(component);
			}

			addEzComponent(group);
			return group;
		}
		return null;
	}

	/** Initialize the given properties group (Protocols). */
	private void initializeGroup(final ParametersCategory category,
			VarList inputMap) {
		for (Var<?> variable : _parameters.variables.get(category)) {
			inputMap.add(variable);
		}
	}

	/** Convert the given Double image into a UBYTE image. */
	private IcyBufferedImage convertToByteImage(final IcyBufferedImage srcImg) {
		IcyBufferedImage res = new IcyBufferedImage(srcImg.getWidth(),
				srcImg.getHeight(), 1, DataType.UBYTE);
		final double src[] = srcImg.getDataXYAsDouble(0);
		byte dst[] = res.getDataXYAsByte(0);

		for (int i = 0; i < src.length; ++i) {
			dst[i] = (byte) (255 * src[i]);
		}

		return res;
	}

	public class SendTracksToPoolThr implements Runnable {
		TrackGroup tgList;
		Sequence seq;

		public SendTracksToPoolThr(final TrackGroup tgList, final Sequence seq) {
			this.tgList = tgList;
			this.seq = seq;
		}

		@Override
		public void run() {
			TrackManager trackManager = new TrackManager();
			SwimmingObject result = new SwimmingObject(tgList);
			Icy.getMainInterface().getSwimmingPool().add(result);
			// export the detections
			trackManager.setDisplaySequence(seq);
		}
	}
}
