/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.blobgenerator.filters;

import icy.image.IcyBufferedImage;
import icy.type.DataType;

/**
 * Abstract class to blend two images.
 * 
 * @version May 3, 2014
 * 
 * @author Julien Jacquemot
 */
public abstract class BlendingFilter {

	/**
	 * Returns the result of the blending of two images accordingly to the given
	 * blending mode.
	 */
	public static IcyBufferedImage apply(final IcyBufferedImage A,
			final IcyBufferedImage B, BlendingMode mode, double intensity) {
		IcyBufferedImage C = new IcyBufferedImage(A.getWidth(), B.getHeight(),
				1, DataType.DOUBLE);

		double dst[] = C.getDataXYAsDouble(0);
		double src1[] = A.getDataXYAsDouble(0);
		double src2[] = B.getDataXYAsDouble(0);

		switch (mode) {
		case Add:
			add(src1, src2, dst);
			break;
		case Average:
			average(src1, src2, dst);
			break;
		case ColorBurn:
			colorBurn(src1, src2, dst);
			break;
		case ColorDodge:
			colorDodge(src1, src2, dst);
			break;
		case Darken:
			darken(src1, src2, dst);
			break;
		case Difference:
			difference(src1, src2, dst);
			break;
		case Exclusion:
			exclusion(src1, src2, dst);
			break;
		case Glow:
			glow(src1, src2, dst);
			break;
		case HardLight:
			hardLight(src1, src2, dst);
			break;
		case HardMix:
			hardMix(src1, src2, dst);
			break;
		case Lighten:
			lighten(src1, src2, dst);
			break;
		case LinearBurn:
			linearBurn(src1, src2, dst);
			break;
		case LinearDodge:
			linearDodge(src1, src2, dst);
			break;
		case LinearLight:
			linearLight(src1, src2, dst);
			break;
		case Multiply:
			multiply(src1, src2, dst);
			break;
		case Negation:
			negation(src1, src2, dst);
			break;
		case Overlay:
			overlay(src1, src2, dst);
			break;
		case Phoenix:
			phoenix(src1, src2, dst);
			break;
		case PinLight:
			pinLight(src1, src2, dst);
			break;
		case Reflect:
			reflect(src1, src2, dst);
			break;
		case Screen:
			screen(src1, src2, dst);
			break;
		case SoftLight:
			softLight(src1, src2, dst);
			break;
		case Subtract:
			subtract(src1, src2, dst);
			break;
		case VividLight:
			vividLight(src1, src2, dst);
			break;
		}

		if (intensity < 1) {
			for (int i = 0; i < dst.length; ++i) {
				dst[i] = (1 - intensity) * src1[i] + intensity * dst[i];
			}
		}

		return C;
	}

	private static void lighten(final double src1[], final double src2[],
			double dst[]) {
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = lighten(src1[i], src2[i]);
		}
	}

	private static void darken(final double src1[], final double src2[],
			double dst[]) {
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = darken(src1[i], src2[i]);
		}
	}

	private static void multiply(final double src1[], final double src2[],
			double dst[]) {
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = src1[i] * src2[i];
		}
	}

	private static void average(final double src1[], final double src2[],
			double dst[]) {
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = 0.5 * (src1[i] + src2[i]);
		}
	}

	private static void add(final double src1[], final double src2[],
			double dst[]) {
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = add(src1[i], src2[i]);
		}
	}

	private static void subtract(final double src1[], final double src2[],
			double dst[]) {
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = subtract(src1[i], src2[i]);
		}
	}

	private static void difference(final double src1[], final double src2[],
			double dst[]) {
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = Math.abs(src1[i] - src2[i]);
		}
	}

	private static void negation(final double src1[], final double src2[],
			double dst[]) {
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = 1 - Math.abs(1 - src1[i] - src2[i]);
		}
	}

	private static void screen(final double src1[], final double src2[],
			double dst[]) {
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = 1 - (1 - src1[i]) * (1 - src2[i]);
		}
	}

	private static void exclusion(final double src1[], final double src2[],
			double dst[]) {
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = src1[i] + src2[i] - 2 * src1[i] * src2[i];
		}
	}

	private static void overlay(final double src1[], final double src2[],
			double dst[]) {
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = (src2[i] < 0.5) ? (2 * src1[i] * src2[i]) : (1 - 2
					* (1 - src1[i]) * (1 - src2[i]));
		}
	}

	private static void softLight(final double src1[], final double src2[],
			double dst[]) {
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = (src2[i] < 0.5) ? ((src1[i] + 0.5) * src2[i])
					: (1 - (1.5 - src1[i]) * (1 - src2[i]));
		}
	}

	private static void hardLight(final double src1[], final double src2[],
			double dst[]) {
		overlay(src2, src1, dst);
	}

	private static void colorDodge(final double src1[], final double src2[],
			double dst[]) {
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = colorDodge(src1[i], src2[i]);
		}
	}

	private static void colorBurn(final double src1[], final double src2[],
			double dst[]) {
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = colorBurn(src1[i], src2[i]);
		}
	}

	private static void linearDodge(final double src1[], final double src2[],
			double dst[]) {
		add(src1, src2, dst);
	}

	private static void linearBurn(final double src1[], final double src2[],
			double dst[]) {
		subtract(src1, src2, dst);
	}

	private static void linearLight(final double src1[], final double src2[],
			double dst[]) {
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = (src2[i] < 0.5) ? subtract(src1[i], 2 * src2[i]) : add(
					src1[i], 2 * (src2[i] - 0.5));
		}
	}

	private static void vividLight(final double src1[], final double src2[],
			double dst[]) {
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = vividLight(src1[i], src2[i]);
		}
	}

	private static void pinLight(final double src1[], final double src2[],
			double dst[]) {
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = (src2[i] < 0.5) ? darken(src1[i], 2 * src2[i]) : lighten(
					src1[i], 2 * (src2[i] - 0.5));
		}
	}

	private static void hardMix(final double src1[], final double src2[],
			double dst[]) {
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = (vividLight(src1[i], src2[i]) < 0.5) ? 0 : 1;
		}
	}

	private static void reflect(final double src1[], final double src2[],
			double dst[]) {
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = (src1[i] == 1) ? 1 : Math.min(1, src2[i] * src2[i]
					/ (1 - src1[i]));
		}
	}

	private static void glow(final double src1[], final double src2[],
			double dst[]) {
		reflect(src2, src1, dst);
	}

	private static void phoenix(final double src1[], final double src2[],
			double dst[]) {
		for (int i = 0; i < dst.length; ++i) {
			dst[i] = Math.min(src1[i], src2[i]) - Math.max(src1[i], src2[i])
					+ 1;
		}
	}

	private static double lighten(double a, double b) {
		return (a > b) ? a : b;
	}

	private static double darken(double a, double b) {
		return (a < b) ? a : b;
	}

	private static double add(double a, double b) {
		return Math.min(1, a + b);
	}

	private static double subtract(double a, double b) {
		return Math.max(0, a + b - 1);
	}

	private static double colorDodge(double a, double b) {
		return (b == 1) ? 1 : Math.min(1, a / (1 - b));
	}

	private static double colorBurn(double a, double b) {
		return (b == 0) ? 0 : Math.max(0, (1 - (1 - a) / b));
	}

	private static double vividLight(double a, double b) {
		return (b < 0.5) ? colorBurn(a, 2 * b) : colorDodge(a, 2 * (b - 0.5));
	}
}
