/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.blobgenerator;

import icy.image.IcyBufferedImage;
import icy.type.point.Point3D;

import java.util.Arrays;
import java.util.Comparator;

import plugins.big.bigsnakeutils.icy.ellipsoid.AbstractEllipsoid;
import plugins.big.bigsnakeutils.icy.ellipsoid.Ellipsoid2D;
import plugins.big.bigsnakeutils.icy.ellipsoid.Ellipsoid3D;
import plugins.big.blobgenerator.parameters.Parameters;

/**
 * Abstract class generating cells.
 * 
 * @version May 3, 2014
 * 
 * @author Julien Jacquemot
 */
public abstract class CellGenerator {
	/** Maximal angle deviation for a cell between two consecutive images. */
	private static final double ANGLE_DEVIATION = 0.1 * Math.PI;

	/**
	 * Maximal percentage of speed deviation. This percentage is applied to the
	 * possible range of speeds.
	 */
	private static final double VELOCITY_DEVIATION = 0.5;

	/**
	 * Generates a cell.
	 * 
	 * @param parameters
	 *            Global parameters
	 * @param images
	 *            Image in which display the cell
	 * @param probabilityMap
	 *            Probability map describing where the cell can be. The
	 *            probability that the center of the generated cell is into
	 *            another cell is 0. This map is updated during the generation.
	 */
	static public AbstractEllipsoid generateCell(final Parameters parameters,
			IcyBufferedImage[] images, double[] probabilityMap) {
		// Create an index array
		Point3D.Integer indexes[] = new Point3D.Integer[probabilityMap.length];
		int i = 0;
		double sum = 0;
		for (int z = 0; z < parameters.imageDepth(); ++z) {
			for (int y = 0; y < parameters.imageHeight(); ++y) {
				for (int x = 0; x < parameters.imageWidth(); ++x) {
					sum += probabilityMap[i];
					indexes[i++] = new Point3D.Integer(x, y, z);
				}
			}
		}

		// Sort the index array according to the probability map
		IndexComparator comparator = new IndexComparator(probabilityMap,
				parameters.imageWidth(), parameters.imageHeight());
		Arrays.sort(indexes, comparator);

		// Pick a random index according to the probability map
		double r = Math.random() * sum;
		double acc = probabilityMap[index(indexes[0], parameters.imageWidth(),
				parameters.imageHeight())];
		i = 0;
		while (acc < r) {
			acc += probabilityMap[index(indexes[++i], parameters.imageWidth(),
					parameters.imageHeight())];
		}

		// Create the cell descriptor
		Point3D.Integer center = indexes[i];
		AbstractEllipsoid cell;
		if (images.length == 1) {
			cell = new Ellipsoid2D();
			((Ellipsoid2D) cell).x0 = center.getX();
			((Ellipsoid2D) cell).y0 = center.getY();
		} else {
			cell = new Ellipsoid3D();
			((Ellipsoid3D) cell).x0 = center.getX();
			((Ellipsoid3D) cell).y0 = center.getY();
			((Ellipsoid3D) cell).z0 = center.getZ();
		}

		double eccentricityRange = parameters.cellsEccentricityRange1()
				.getSecond() - parameters.cellsEccentricityRange1().getFirst();
		double e1 = parameters.cellsEccentricityRange1().getFirst()
				+ Math.random() * eccentricityRange;

		double radiusRange = parameters.cellsRadiusRange().getSecond()
				- parameters.cellsRadiusRange().getFirst();
		double b = parameters.cellsRadiusRange().getFirst() + Math.random()
				* radiusRange;
		if (images.length == 1) {
			((Ellipsoid2D) cell).b = b;
			((Ellipsoid2D) cell).a = Math.sqrt(b * b / (1 - e1 * e1));
		} else {
			double e2 = parameters.cellsEccentricityRange2().getFirst()
					+ Math.random() * eccentricityRange;

			((Ellipsoid3D) cell).b = b;
			((Ellipsoid3D) cell).a = Math.sqrt(b * b / (1 - e1 * e1));
			((Ellipsoid3D) cell).c = Math.sqrt(b * b * (1 - e2 * e2));
		}

		if (images.length == 1) {
			((Ellipsoid2D) cell).alpha = Math.random() * 2 * Math.PI;
		} else {
			((Ellipsoid3D) cell).alpha = Math.random() * 2 * Math.PI;
			((Ellipsoid3D) cell).beta = Math.random() * 2 * Math.PI;
			((Ellipsoid3D) cell).gamma = Math.random() * 2 * Math.PI;
		}

		double velocityRange = parameters.cellsVelocityRange().getSecond()
				- parameters.cellsVelocityRange().getFirst();
		cell.setProperty("vx", parameters.cellsVelocityRange().getFirst()
				+ Math.random() * velocityRange);
		cell.setProperty("vy", parameters.cellsVelocityRange().getFirst()
				+ Math.random() * velocityRange);
		if (images.length > 1) {
			cell.setProperty("vz", parameters.cellsVelocityRange().getFirst()
					+ Math.random() * velocityRange);
		}

		double valueRange = parameters.foregroundValueRange().getSecond()
				- parameters.foregroundValueRange().getFirst();
		double value = parameters.foregroundValueRange().getFirst()
				+ Math.random() * valueRange;
		cell.setProperty("value", value);

		// Display the cell and remove its location from the probability map
		if (images.length == 1) {
			int xMin = (int) Math.max(0, ((Ellipsoid2D) cell).x0
					- ((Ellipsoid2D) cell).a), xMax = (int) Math.min(
					parameters.imageWidth() - 1, ((Ellipsoid2D) cell).x0
							+ ((Ellipsoid2D) cell).a);
			int yMin = (int) Math.max(0, ((Ellipsoid2D) cell).y0
					- ((Ellipsoid2D) cell).a), yMax = (int) Math.min(
					parameters.imageHeight() - 1, ((Ellipsoid2D) cell).y0
							+ ((Ellipsoid2D) cell).a);

			double R[] = cell.getRotationMatrix();
			double data[] = images[0].getDataXYAsDouble(0);

			for (int y = yMin; y <= yMax; ++y) {
				for (int x = xMin; x <= xMax; ++x) {
					if (((Ellipsoid2D) cell).contains(x, y, R)) {
						probabilityMap[index(x, y, 0, parameters.imageWidth(),
								parameters.imageHeight())] = 0;
						data[index(x, y, 0, parameters.imageWidth(),
								parameters.imageHeight())] = value;
					}
				}
			}
		} else {
			int xMin = (int) Math.max(0, ((Ellipsoid3D) cell).x0
					- ((Ellipsoid3D) cell).a), xMax = (int) Math.min(
					parameters.imageWidth() - 1, ((Ellipsoid3D) cell).x0
							+ ((Ellipsoid3D) cell).a);
			int yMin = (int) Math.max(0, ((Ellipsoid3D) cell).y0
					- ((Ellipsoid3D) cell).a), yMax = (int) Math.min(
					parameters.imageHeight() - 1, ((Ellipsoid3D) cell).y0
							+ ((Ellipsoid3D) cell).a);
			int zMin = (int) Math.max(0, ((Ellipsoid3D) cell).z0
					- ((Ellipsoid3D) cell).a), zMax = (int) Math.min(
					parameters.imageDepth() - 1, ((Ellipsoid3D) cell).z0
							+ ((Ellipsoid3D) cell).a);
			double R[] = cell.getRotationMatrix();

			for (int z = zMin; z <= zMax; ++z) {
				double data[] = images[z].getDataXYAsDouble(0);

				for (int y = yMin; y <= yMax; ++y) {
					for (int x = xMin; x <= xMax; ++x) {
						if (((Ellipsoid3D) cell).contains(x, y, z, R)) {
							probabilityMap[index(x, y, z,
									parameters.imageWidth(),
									parameters.imageHeight())] = 0;
							data[index(x, y, 0, parameters.imageWidth(),
									parameters.imageHeight())] = value;
						}
					}
				}
			}
		}

		return cell;
	}

	/**
	 * Generates a cell at time t using a cell at time t-1.
	 * 
	 * @param parameters
	 *            Global parameters.
	 * @param images
	 *            Image in which display the cell.
	 * @param cell
	 *            Cell at time t-1. This cell will be modified.
	 * @return Returns cell.
	 */
	static public AbstractEllipsoid generateCell(final Parameters parameters,
			IcyBufferedImage[] images, AbstractEllipsoid cell) {
		boolean closeLeft = false, closeRight = false, closeTop = false, closeBottom = false;

		if (images.length == 1) {
			((Ellipsoid2D) cell).x0 += cell.getProperty("vx");
			((Ellipsoid2D) cell).y0 += cell.getProperty("vy");

			((Ellipsoid2D) cell).alpha += ANGLE_DEVIATION
					* (Math.random() - 0.5);

			if (((Ellipsoid2D) cell).x0 <= parameters.cellsRadiusRange()
					.getFirst()) {
				closeLeft = true;
			} else if (((Ellipsoid2D) cell).x0 >= parameters.imageWidth()
					- parameters.cellsRadiusRange().getFirst()) {
				closeRight = true;
			}

			if (((Ellipsoid2D) cell).y0 <= parameters.cellsRadiusRange()
					.getFirst()) {
				closeTop = true;
			} else if (((Ellipsoid2D) cell).y0 >= parameters.imageHeight()
					- parameters.cellsRadiusRange().getFirst()) {
				closeBottom = true;
			}

		} else {
			((Ellipsoid3D) cell).x0 += cell.getProperty("vx");
			((Ellipsoid3D) cell).y0 += cell.getProperty("vy");
			((Ellipsoid3D) cell).z0 += cell.getProperty("vz");

			((Ellipsoid3D) cell).alpha += ANGLE_DEVIATION
					* (Math.random() - 0.5);
			((Ellipsoid3D) cell).beta += ANGLE_DEVIATION
					* (Math.random() - 0.5);
			((Ellipsoid3D) cell).gamma += ANGLE_DEVIATION
					* (Math.random() - 0.5);

			if (((Ellipsoid3D) cell).x0 <= parameters.cellsRadiusRange()
					.getFirst()) {
				closeLeft = true;
			} else if (((Ellipsoid3D) cell).x0 >= parameters.imageWidth()
					- parameters.cellsRadiusRange().getFirst()) {
				closeRight = true;
			}

			if (((Ellipsoid3D) cell).y0 <= parameters.cellsRadiusRange()
					.getFirst()) {
				closeTop = true;
			} else if (((Ellipsoid3D) cell).y0 >= parameters.imageHeight()
					- parameters.cellsRadiusRange().getFirst()) {
				closeBottom = true;
			}
		}

		// Update velocity (if the cell is too close of the borders, it's send
		// to the opposite direction)
		double velocityRange = (parameters.cellsVelocityRange().getSecond() - parameters
				.cellsVelocityRange().getFirst());

		double vx = cell.getProperty("vx");
		if (closeLeft) {
			vx += parameters.cellsVelocityRange().getSecond();
		} else if (closeRight) {
			vx -= parameters.cellsVelocityRange().getSecond();
		} else {
			vx += VELOCITY_DEVIATION * (Math.random() - 0.5) * velocityRange;
		}
		cell.setProperty("vx", Math.min(
				Math.max(vx, parameters.cellsVelocityRange().getFirst()),
				parameters.cellsVelocityRange().getSecond()));

		double vy = cell.getProperty("vy");
		if (closeTop) {
			vy += parameters.cellsVelocityRange().getSecond();
		} else if (closeBottom) {
			vy -= parameters.cellsVelocityRange().getSecond();
		} else {
			vy += VELOCITY_DEVIATION * (Math.random() - 0.5) * velocityRange;
		}
		cell.setProperty("vy", Math.min(
				Math.max(vy, parameters.cellsVelocityRange().getFirst()),
				parameters.cellsVelocityRange().getSecond()));

		if (images.length > 1) {
			double vz = cell.getProperty("vz");
			if (((Ellipsoid3D) cell).z0 <= parameters.cellsRadiusRange()
					.getFirst()) {
				vz += parameters.cellsVelocityRange().getSecond();
			} else if (((Ellipsoid3D) cell).z0 >= parameters.imageDepth()
					- parameters.cellsRadiusRange().getFirst()) {
				vz -= parameters.cellsVelocityRange().getSecond();
			} else {
				vz += VELOCITY_DEVIATION * (Math.random() - 0.5)
						* velocityRange;
			}
			cell.setProperty("vz", Math.min(
					Math.max(vz, parameters.cellsVelocityRange().getFirst()),
					parameters.cellsVelocityRange().getSecond()));
		}

		// Display the cell
		double value = cell.getProperty("value");
		if (images.length == 1) {
			int xMin = (int) Math.max(0, ((Ellipsoid2D) cell).x0
					- ((Ellipsoid2D) cell).a), xMax = (int) Math.min(
					parameters.imageWidth() - 1, ((Ellipsoid2D) cell).x0
							+ ((Ellipsoid2D) cell).a);
			int yMin = (int) Math.max(0, ((Ellipsoid2D) cell).y0
					- ((Ellipsoid2D) cell).a), yMax = (int) Math.min(
					parameters.imageHeight() - 1, ((Ellipsoid2D) cell).y0
							+ ((Ellipsoid2D) cell).a);

			double R[] = cell.getRotationMatrix();
			double data[] = images[0].getDataXYAsDouble(0);

			for (int y = yMin; y <= yMax; ++y) {
				for (int x = xMin; x <= xMax; ++x) {
					if (((Ellipsoid2D) cell).contains(x, y, R)) {
						data[index(x, y, 0, parameters.imageWidth(),
								parameters.imageHeight())] = value;
					}
				}
			}
		} else {
			int xMin = (int) Math.max(0, ((Ellipsoid3D) cell).x0
					- ((Ellipsoid3D) cell).a), xMax = (int) Math.min(
					parameters.imageWidth() - 1, ((Ellipsoid3D) cell).x0
							+ ((Ellipsoid3D) cell).a);
			int yMin = (int) Math.max(0, ((Ellipsoid3D) cell).y0
					- ((Ellipsoid3D) cell).a), yMax = (int) Math.min(
					parameters.imageHeight() - 1, ((Ellipsoid3D) cell).y0
							+ ((Ellipsoid3D) cell).a);
			int zMin = (int) Math.max(0, ((Ellipsoid3D) cell).z0
					- ((Ellipsoid3D) cell).a), zMax = (int) Math.min(
					parameters.imageDepth() - 1, ((Ellipsoid3D) cell).z0
							+ ((Ellipsoid3D) cell).a);
			double R[] = cell.getRotationMatrix();

			for (int z = zMin; z <= zMax; ++z) {
				double data[] = images[z].getDataXYAsDouble(0);

				for (int y = yMin; y <= yMax; ++y) {
					for (int x = xMin; x <= xMax; ++x) {
						if (((Ellipsoid3D) cell).contains(x, y, z, R)) {
							data[index(x, y, 0, parameters.imageWidth(),
									parameters.imageHeight())] = value;
						}
					}
				}
			}
		}

		return cell;
	}

	/** Returns the image index corresponding to the given coordinates. */
	private static int index(int x, int y, int z, int imgWidth, int imgHeight) {
		return x + (y + z * imgHeight) * imgWidth;
	}

	/** Returns the image index corresponding to the given coordinates. */
	private static int index(final Point3D.Integer pt, int imgWidth,
			int imgHeight) {
		return index((int) pt.getX(), (int) pt.getY(), (int) pt.getZ(),
				imgWidth, imgHeight);
	}

	/**
	 * Class allowing the comparison of two indexes. It is used to sort the
	 * image indexes, to pick at random one coordinate accordingly to the
	 * probability map.
	 * 
	 * @version October 18, 2013
	 * @author Julien Jacquemot
	 * 
	 */
	static private class IndexComparator implements Comparator<Point3D.Integer> {
		private final double probabilityMap[];
		private final int imgWidth;
		private final int imgHeight;

		/** Default constructor. */
		public IndexComparator(final double probabilityMap[], int imgWidth,
				int imgHeight) {
			this.probabilityMap = probabilityMap;
			this.imgWidth = imgWidth;
			this.imgHeight = imgHeight;
		}

		@Override
		public int compare(Point3D.Integer arg0, Point3D.Integer arg1) {
			return (int) Math.signum(probabilityMap[index(arg0, imgWidth,
					imgHeight)]
					- probabilityMap[index(arg1, imgWidth, imgHeight)]);
		}
	}
}
