/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.blobgenerator;

import icy.image.IcyBufferedImage;
import icy.type.DataType;
import plugins.big.bigsnakeutils.icy.gui.curve.Curve;
import plugins.big.blobgenerator.parameters.Parameters;

/**
 * Abstract class generating noise.
 * 
 * @version May 3, 2014
 * 
 * @author Julien Jacquemot
 */
public abstract class NoiseGenerator {
	private final static int INTEGRAL_RES = 256;

	/**
	 * Generate a noisy 3D image. Values are in the range [0; 1].
	 * 
	 * @param parameters
	 *            Global parameters
	 * @param noiseCurve
	 *            Noise density function
	 * @param levelMin
	 *            Minimum noise scale
	 * @param levelMax
	 *            Maximum noise scale
	 * @param imgWidth
	 *            Width of the generated image
	 * @param imgHeight
	 *            Width of the generated image
	 * @param imgDepth
	 *            Width of the generated image
	 */
	static public IcyBufferedImage[] generateNoise(final Parameters parameters,
			final Curve noiseCurve, int levelMin, int levelMax, int imgWidth,
			int imgHeight, int imgDepth) {
		double[] noiseProfile = computeNoiseProfile(noiseCurve);

		// Initialize buffers
		double scaleFactor = Math.pow(2.0, -levelMax);
		int width = (int) Math.ceil(imgWidth * scaleFactor);
		int height = (int) Math.ceil(imgHeight * scaleFactor);
		int depth = (int) Math.ceil(imgDepth * scaleFactor);

		IcyBufferedImage[] buffers = new IcyBufferedImage[depth];
		for (int z = 0; z < depth; ++z) {
			buffers[z] = new IcyBufferedImage(width, height, 1, DataType.DOUBLE);

			double[] data = buffers[z].getDataXYAsDouble(0);
			for (int i = 0; i < data.length; ++i) {
				data[i] = 0;
			}
		}

		// Compute noise layers weight
		double weights[] = new double[levelMax - levelMin + 1];
		double sum = weights.length * (weights.length + 1) / 2;
		for (int i = 0; i < weights.length; ++i) {
			weights[i] = (i + 1) / sum;
		}

		// Generate noise layers
		for (int l = levelMax; l >= levelMin; --l) {
			// Generate one layer of noise
			IcyBufferedImage[] layers = generateLayers(parameters, l,
					noiseProfile, imgWidth, imgHeight, imgDepth);

			// Resize the maps
			if (l > levelMin) {
				layers = resizeLayers(parameters, l - 1, layers, imgWidth,
						imgHeight, imgDepth);
				buffers = resizeLayers(parameters, l - 1, buffers, imgWidth,
						imgHeight, imgDepth);
			}

			// Update the buffers
			for (int z = 0; z < buffers.length; ++z) {
				double[] src = layers[z].getDataXYAsDouble(0);
				double[] dst = buffers[z].getDataXYAsDouble(0);

				for (int i = 0; i < src.length; ++i) {
					dst[i] += weights[l - levelMin] * src[i];
				}
			}
		}

		if (levelMin != 0) {
			return resizeLayers(parameters, 0, buffers, imgWidth, imgHeight,
					imgDepth);
		}
		return buffers;
	}

	/**
	 * Generate a noisy 3D image. Values are in the range [0; 1].
	 * 
	 * @param parameters
	 *            Global parameters
	 * @param imgWidth
	 *            Width of the generated image
	 * @param imgHeight
	 *            Width of the generated image
	 * @param imgDepth
	 *            Width of the generated image
	 */
	static public IcyBufferedImage[] generateNoise(final Parameters parameters,
			int imgWidth, int imgHeight, int imgDepth) {
		return generateNoise(parameters, parameters.noiseProfile(), 0,
				parameters.noiseLevels() - 1, imgWidth, imgHeight, imgDepth);
	}

	/**
	 * Generate a noisy 3D image. Values are in the range [0; 1].
	 * 
	 * @param parameters
	 *            Global parameters
	 * @param noiseCurve
	 *            Noise density function
	 * @param levelMin
	 *            Minimum noise scale
	 * @param levelMax
	 *            Maximum noise scale
	 */
	static public IcyBufferedImage[] generateNoise(final Parameters parameters,
			final Curve noiseCurve, int levelMin, int levelMax) {
		return generateNoise(parameters, noiseCurve, levelMin, levelMax,
				parameters.imageWidth(), parameters.imageHeight(),
				parameters.imageDepth());
	}

	/**
	 * Generate a noisy 3D image. Values are in the range [0; 1].
	 * 
	 * @param parameters
	 *            Global parameters
	 */
	static public IcyBufferedImage[] generateNoise(final Parameters parameters) {
		return generateNoise(parameters, parameters.noiseProfile(), 0,
				parameters.noiseLevels() - 1, parameters.imageWidth(),
				parameters.imageHeight(), parameters.imageDepth());
	}

	/**
	 * Generate a 2D noisy image.
	 * 
	 * @param parameters
	 *            Global parameters
	 * @param layer
	 *            Z coordinate of the generated layer in the final image
	 * @param noiseProfile
	 *            Array to pick a random value accordingly to the noise density
	 *            function
	 * @param imgWidth
	 *            Width of the layer
	 * @param imgHeight
	 *            Height of the layer
	 */
	static private IcyBufferedImage[] generateLayers(
			final Parameters parameters, int layer,
			final double noiseProfile[], int imgWidth, int imgHeight,
			int imgDepth) {
		double scaleFactor = Math.pow(2.0, -layer);
		int width = (int) Math.ceil(imgWidth * scaleFactor);
		int height = (int) Math.ceil(imgHeight * scaleFactor);
		int depth = (int) Math.ceil(imgDepth * scaleFactor);

		IcyBufferedImage[] layers = new IcyBufferedImage[depth];
		for (int z = 0; z < depth; ++z) {
			layers[z] = new IcyBufferedImage(width, height, 1, DataType.DOUBLE);
			double[] data = layers[z].getDataXYAsDouble(0);

			// Use the noiseProfile to generate a random value
			for (int i = 0; i < data.length; ++i) {
				data[i] = generateRandomValue(noiseProfile);
			}
		}

		return layers;
	}

	/** Generate a random value accordingly to the noise density function */
	static private double generateRandomValue(final double noiseProfile[]) {
		double r = Math.random();
		return noiseProfile[(int) Math.floor(r * noiseProfile.length)];
	}

	/**
	 * Resize a 3D image accordingly to noise scale of this image.
	 * 
	 * @param parameters
	 *            Global parameters
	 * @param layer
	 *            Noise scale
	 * @param layers
	 *            Image to resize
	 * @param imgWidth
	 *            Width of the desired image
	 * @param imgHeight
	 *            Height of the desired image
	 * @param imgDepth
	 *            Depth of the desired image
	 */
	static private IcyBufferedImage[] resizeLayers(final Parameters parameters,
			int layer, final IcyBufferedImage[] layers, int imgWidth,
			int imgHeight, int imgDepth) {
		double scaleFactor = Math.pow(2.0, -layer);
		int depth = (int) Math.ceil(imgDepth * scaleFactor);

		IcyBufferedImage[] res = new IcyBufferedImage[depth];

		// Resize image in depth if needed
		if (depth == layers.length) {
			for (int z = 0; z < depth; ++z) {
				res[z] = layers[z];
			}
		} else {
			for (int z = 0; z < depth; ++z) {
				res[z] = new IcyBufferedImage(layers[0].getWidth(),
						layers[0].getHeight(), 1, DataType.DOUBLE);
				double z0 = ((double) z * (layers.length - 1))
						/ ((double) depth - 1);
				int previous = (int) Math.floor(z0);
				int next = (int) Math.ceil(z0);
				if (previous == next) {
					++next;
				}
				double dp = z0 - previous;
				double dn = next - z0;
				if (next == layers.length) {
					next = 0;
				}

				double[] dst = res[z].getDataXYAsDouble(0);
				double[] src1 = layers[previous].getDataXYAsDouble(0);
				double[] src2 = layers[next].getDataXYAsDouble(0);

				for (int i = 0; i < dst.length; ++i) {
					dst[i] = dn * src1[i] + dp * src2[i];
				}
			}
		}

		// Resize each layer
		for (int z = 0; z < depth; ++z) {
			res[z] = resizeLayer(parameters, layer, res[z], imgWidth, imgHeight);
		}

		return res;
	}

	/**
	 * Resize a 2D image accordingly to noise scale of this image.
	 * 
	 * @param parameters
	 *            Global parameters
	 * @param layer
	 *            Noise scale
	 * @param srcImg
	 *            Image to resize
	 * @param imgWidth
	 *            Width of the desired image
	 * @param imgHeight
	 *            Height of the desired image
	 */
	static private IcyBufferedImage resizeLayer(final Parameters parameters,
			int layer, final IcyBufferedImage srcImg, int imgWidth,
			int imgHeight) {
		double scaleFactor = Math.pow(2.0, -layer);
		int width = (int) Math.ceil(imgWidth * scaleFactor);
		int height = (int) Math.ceil(imgHeight * scaleFactor);

		IcyBufferedImage buffer = new IcyBufferedImage(width,
				srcImg.getHeight(), 1, DataType.DOUBLE);
		IcyBufferedImage res = new IcyBufferedImage(width, height, 1,
				DataType.DOUBLE);

		// Resize rows
		double src[] = srcImg.getDataXYAsDouble(0);
		double dst[] = buffer.getDataXYAsDouble(0);
		if (width == srcImg.getWidth()) {
			for (int i = 0; i < src.length; ++i) {
				dst[i] = src[i];
			}
		} else {
			for (int c = 0; c < width; ++c) {
				double c0 = ((double) c * (srcImg.getWidth() - 1))
						/ ((double) width - 1);
				int previous = (int) Math.floor(c0);
				int next = (int) Math.ceil(c0);
				if (previous == next) {
					++next;
				}
				double dp = c0 - previous;
				double dn = next - c0;
				if (next == srcImg.getWidth()) {
					next = 0;
				}

				int i = c, ip = previous, in = next;
				for (int r = 0; r < srcImg.getHeight(); ++r, i += width, ip += srcImg
						.getWidth(), in += srcImg.getWidth()) {
					dst[i] = dn * src[ip] + dp * src[in];
				}
			}
		}

		// Resize cols
		src = buffer.getDataXYAsDouble(0);
		dst = res.getDataXYAsDouble(0);
		if (height == srcImg.getHeight()) {
			for (int i = 0; i < src.length; ++i) {
				dst[i] = src[i];
			}
		} else {
			for (int r = 0; r < height; ++r) {
				double r0 = ((double) r * (srcImg.getHeight() - 1))
						/ ((double) height - 1);
				int previous = (int) Math.floor(r0);
				int next = (int) Math.ceil(r0);
				if (previous == next) {
					++next;
				}
				double dp = r0 - previous;
				double dn = next - r0;
				if (next == srcImg.getHeight()) {
					next = 0;
				}

				int i = r * width, ip = previous * width, in = next * width;
				for (int c = 0; c < width; ++c, ++i, ++ip, ++in) {
					dst[i] = dn * src[ip] + dp * src[in];
				}
			}
		}

		return res;
	}

	/**
	 * Compute an array to pick random values. The array represents the inverse
	 * of the noise density function integral.
	 * 
	 * @param noiseCurve
	 *            Noise density function
	 */
	static private double[] computeNoiseProfile(final Curve noiseCurve) {
		// Compute the integral of the noise profile entered by the user
		// Fout times the final res to minimize some aliasing problems, but it's
		// clearly not optimal.
		double integral[] = new double[4 * INTEGRAL_RES];

		integral[0] = 0;
		for (int i = 1; i < integral.length; ++i) {
			integral[i] = integral[i - 1]
					+ Math.max(Math.min(
							noiseCurve.valueAt((i)
									/ ((double) integral.length - 1)), 1), 0);
		}

		// Approximate the inverse of the integral
		double res[] = new double[INTEGRAL_RES];
		int j = 0;
		for (int i = 0; i < res.length; ++i) {
			double y = (i * integral[integral.length - 1])
					/ ((double) res.length - 1);
			for (; j < integral.length; ++j) {
				if (integral[j] >= y) {
					break;
				}
			}
			res[i] = (j) / ((double) integral.length - 1);
		}

		return res;
	}
}
